// This class is based on the FishDisplay class, version 1 August 2002
// by Julie Zenekski, Alyce Brady, and Chris Nevison

// Original copyright notice:

// AP(r) Computer Science Marine Biology Simulation:
// The FishDisplay class is copyright(c) 2002 College Entrance
// Examination Board (www.collegeboard.com).
//
// This class is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation.
//
// This class is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.


package model.fish.display;

import sysModel.fish.IFishDisplay;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.awt.geom.Ellipse2D;
import java.awt.geom.GeneralPath;

/**
 * Parameterized fish display.
 *
 * @author Mathias Ricken
 */
public class ParamFishDisplay implements IFishDisplay {
    private static final double BODY_WIDTH = .55;
    private static final double BODY_LENGTH = .75;
    private static final double TAIL_WIDTH = .5;
    private static final double TAIL_LENGTH = .4;
    private static final double EYE_SIZE = .08;

    private static final int GRADIENT_SIZE = 50;
    private static final AffineTransform ATX = AffineTransform.getScaleInstance(GRADIENT_SIZE, GRADIENT_SIZE);

    private Shape bodyAndTail;
    private Shape eye1;
    private Shape eye2;

    /**
     * Make an object that knows how to drawFish simple fish.
     *
     * @param bodyWidth  width of the body
     * @param bodyLength length of the body
     * @param tailWidth  width of the tail
     * @param tailLength length of the tail
     * @param eyeSize    size of the eyes
     */
    public ParamFishDisplay(double bodyWidth, double bodyLength, double tailWidth, double tailLength, double eyeSize) {
        buildPaths(bodyWidth, bodyLength, tailWidth, tailLength, eyeSize);
    }

    /**
     * Default constructor.
     */
    public ParamFishDisplay() {
        this(BODY_WIDTH, BODY_LENGTH, TAIL_WIDTH, TAIL_LENGTH, EYE_SIZE);
    }

    /**
     * Set up the paths used for the fish body, tail, and eyes. Different parameters will change the proportions, and
     * thereby control the "look" of the fish.  The various parameters should be specified assuming the fish will occupy
     * a cell of size (1, 1).
     *
     * @param bodyWidth  width of the elliptical body
     * @param bodyLength length of the elliptical body
     * @param tailWidth  width of the triangular tail
     * @param tailLength length of the triangular tail
     * @param eyeSize    diameter of the eye
     */
    private void buildPaths(double bodyWidth, double bodyLength, double tailWidth, double tailLength, double eyeSize) {
        // Build a set of paths for a fish facing North in a unit-length cell.
        // We will rotate/scale as needed later.

        float halfFishLength = (float)(bodyLength + tailLength / 3) / 2;

        // The fish body is an ellipse of the given body width and length.
        // The ellipse is horizontally centered and slightly above vertical
        // center (to leave room for tail).
        Shape body = new Ellipse2D.Double(-bodyWidth / 2, -halfFishLength, bodyWidth, bodyLength);

        // The fish tail is a triangle overlapping the end of body.
        GeneralPath tail = new GeneralPath();
        tail.moveTo(-(float)tailWidth / 2, halfFishLength);	// lower left
        tail.lineTo(0, halfFishLength - (float)tailLength);	// top of tail
        tail.lineTo((float)tailWidth / 2, halfFishLength);	// lower right
        tail.closePath();

        // Join body and tail together in one path.
        tail.append(body, false);
        bodyAndTail = tail;

        // The fish eyes are circles.
        eye1 = new Ellipse2D.Double(-bodyWidth / 4, -halfFishLength + bodyLength / 4, eyeSize, eyeSize);
        eye2 = new Ellipse2D.Double(+bodyWidth / 4 - eyeSize, -halfFishLength + bodyLength / 4, eyeSize, eyeSize);
    }

    /**
     * Draw the fish facing north on the Graphics2D object. The Graphics2D object has been prepared such that the center
     * of the fish is at the origin.
     *
     * @param g2        drawing surface
     * @param comp      the component to drawFish on
     * @param fishColor color of the fish
     */
    public void draw(Graphics2D g2, Component comp, Color fishColor) {
        Color oldColor = g2.getColor();

        // Stroke outline of fish body and tail in slightly darker color.
        g2.setPaint(fishColor.darker());
        g2.draw(bodyAndTail);

        // Fill fish body and tail with gradient (scale up temporarily to get smooth dither).
        g2.scale(1.0 / GRADIENT_SIZE, 1.0 / GRADIENT_SIZE);
        g2.setPaint(new GradientPaint(-GRADIENT_SIZE / 4,
            -GRADIENT_SIZE / 2,
            Color.white,
            GRADIENT_SIZE / 4,
            GRADIENT_SIZE / 4,
            fishColor));
        g2.fill(ATX.createTransformedShape(bodyAndTail));
        g2.scale(GRADIENT_SIZE, GRADIENT_SIZE);

        // Fill black circles for the eyes.
        g2.setPaint(Color.black);
        g2.fill(eye1);
        g2.fill(eye2);

        g2.setColor(oldColor);
    }
}
