// This class is based on the FishDisplay class, version 1 August 2002
// by Julie Zenekski

// Original copyright notice:

// AP(r) Computer Science Marine Biology Simulation:
// The FishImageDisplay class is copyright(c) 2002 College Entrance
// Examination Board (www.collegeboard.com).
//
// This class is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation.
//
// This class is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

package model.fish.display;

import sysModel.fish.IFishDisplay;

import javax.swing.*;
import java.awt.*;
import java.awt.geom.Ellipse2D;
import java.awt.image.FilteredImageSource;
import java.awt.image.RGBImageFilter;
import java.util.HashMap;
import java.net.URL;

/**
 * Use an image file to display a fish.
 *
 * @author Mathias Ricken
 */
public class ImageFishDisplay implements IFishDisplay {
    private ImageIcon _icon;
    private Image _originalImage;
    private double _radians;
    private HashMap<Color,Image> _tintedVersions = new HashMap<Color,Image>();

    /**
     * Make an object that knows how to display a fish as an image. Looks for the named file first in the jar file, then
     * in the current directory. If the named file is not found or the file is malformed, a colored circle will be
     * substituted instead.
     *
     * @param imageFilename name of file containing image
     * @param radians       radians the fish has to be rotated by to make it face north
     */
    public ImageFishDisplay(String imageFilename, double radians) {
        URL urlInJarFile = getClass().getResource(imageFilename);
        if (null != urlInJarFile) {
            _icon = new ImageIcon(urlInJarFile);
        }
        else {
            String path = System.getProperty("user.dir") + java.io.File.separator + imageFilename;
            _icon = new ImageIcon(path);
        }
        _radians = radians;
        _originalImage = _icon.getImage();
    }


    /**
     * Draw the fish. The Graphics2D object has been set up so that the origin is in the center of the fish. A fish that
     * is 32x32 wide should thus drawFish from (-16,-16) to (16,16).
     *
     * @param g         graphics object to drawFish on.
     * @param comp      the component to drawFish on
     * @param fishColor color of the fish
     */
    public void draw(Graphics2D g, Component comp, Color fishColor) {
        if (MediaTracker.COMPLETE != _icon.getImageLoadStatus()) {
            // Image failed to load, so fall back to default display.
            g.setPaint(fishColor);
            g.fill(new Ellipse2D.Double(-.4, -.4, .8, .8));
            return;
        }

        // Rotate drawing surface to compensate for the direction of the fish
        // in the image.
        g.rotate(_radians);

        // Scale to shrink or enlarge the image to fit the size 1x1 cell.
        g.scale(1.0 / _icon.getIconWidth(), 1.0 / _icon.getIconHeight());

        // Compose image with fish color using an image filter.
        Image tinted = (Image)_tintedVersions.get(fishColor);
        if (null == tinted) { 	// not cached, need new filter for color
            FilteredImageSource src = new FilteredImageSource(_originalImage.getSource(), new TintFilter(fishColor));
            tinted = comp.createImage(src);
            // Cache tinted image in map by color, we're likely to need it again.
            _tintedVersions.put(fishColor, tinted);
        }

        _icon.setImage(tinted);
        _icon.paintIcon(comp, g, -_icon.getIconWidth() / 2, -_icon.getIconHeight() / 2);
    }

    /**
     * An image filter class that tints colors based on the tint provided to the constructor (the color of a fish).
     */
    private static class TintFilter extends RGBImageFilter {
        private int tintR;
        private int tintG;
        private int tintB;

        /**
         * Construct an image filter for tinting colors in an image.
         *
         * @param color tint
         */
        public TintFilter(Color color) {
            canFilterIndexColorModel = true;
            int rgb = color.getRGB();
            tintR = (rgb >> 16) & 0xff;
            tintG = (rgb >> 8) & 0xff;
            tintB = rgb & 0xff;
        }

        public int filterRGB(int x, int y, int argb) {
            // Separate pixel into its RGB coomponents.
            int alpha = (argb >> 24) & 0xff;
            int red = (argb >> 16) & 0xff;
            int green = (argb >> 8) & 0xff;
            int blue = argb & 0xff;

            // Use NTSC/PAL algorithm to convert RGB to grayscale.
            int lum = (int)(0.2989 * red + 0.5866 * green + 0.1144 * blue);

            // Interpolate along spectrum black->white with tint at midpoint
            double scale = Math.abs((lum - 128) / 128.0); // absolute distance from midpt
            int edge = 128 > lum ? 0 : 255;	// going towards white or black?
            red = tintR + (int)((edge - tintR) * scale); // scale from midpt to edge
            green = tintG + (int)((edge - tintG) * scale);
            blue = tintB + (int)((edge - tintB) * scale);
            return (alpha << 24) | (red << 16) | (green << 8) | blue;
        }
    }
}

