package sysModel;

import sysModel.classFile.ClassFile;
import sysModel.classFile.MethodInfo;
import sysModel.classFile.attributes.CodeAttributeInfo;
import sysModel.classFile.constantPool.APoolInfo;
import sysModel.classFile.constantPool.ClassPoolInfo;
import sysModel.classFile.constantPool.visitors.ADefaultPoolInfoVisitor;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.jar.JarEntry;
import java.util.jar.JarFile;

/**
 * Class loader used for loading fish classes.
 * @author Mathias Ricken
 */
public class MBSClassLoader extends ClassLoader {
    /**
     * Classes already loaded.
     */
    HashMap<String,Class<?>> _classes = new HashMap<String,Class<?>>();

    /**
     * String with class path.
     */
    private String _classPathString;

    /**
     * String with the Java home directory.
     */
    private static final String JAVA_HOME_STRING = System.getProperty("java.home");

    /**
     * Character with separator character.
     */
    private static final char SEPARATOR_CHAR = File.separatorChar;

    /**
     * String with path separator character.
     */
    private static final String PATH_SEPARATOR_STRING = File.pathSeparator;

    /**
     * Security manager in use.
     */
    private MBSSecurityManager _securityManager;

    /**
     * List of built-in classes that should not be checked and be loaded with the regular class loader.
     */
    private ArrayList<String> _buildinClasses = new ArrayList<String>();

    /**
     * Create a new class loader.
     * @param securityManager security manager that lets us enable and disable security measures
     */
    public MBSClassLoader(MBSSecurityManager securityManager) {
        // prefix with rt.jar
        _classPathString = JAVA_HOME_STRING + SEPARATOR_CHAR + "lib" + SEPARATOR_CHAR + "rt.jar" + PATH_SEPARATOR_STRING +
                           System.getProperty("java.class.path");
        if (!_classPathString.endsWith(PATH_SEPARATOR_STRING)) {
            _classPathString += PATH_SEPARATOR_STRING;
        }
        _classPathString += '.';
        _securityManager = securityManager;
        for(String className: BUILTIN_CLASSES) {
            _buildinClasses.add(className);
        }
    }

    /**
     * Load a class.
     * @param name class name
     * @param resolveIt resolve class?
     * @return class object
     * @throws ClassNotFoundException
     */
    public synchronized Class<?> loadClass(String name, boolean resolveIt) throws ClassNotFoundException {
        try {
            _securityManager.pushProtected(false);
            Class<?> result = null;

            if (!_buildinClasses.contains(name)) {
                // Check our local cache of classes
                result = _classes.get(name);
                if (null != result) {
                    return result;
                }

                result = loadFromClassPath(name, resolveIt);
            }

            if (null == result) {
                // Check with the primordial class loader
                try {
                    result = getParent().loadClass(name);
                    return result;
                }
                catch (ClassNotFoundException e) {
                    throw new ClassNotFoundException(e.toString());
                }
            }

            return result;
        }
        finally {
            _securityManager.popProtected();
        }
    }

    /**
     * Load class from class path.
     * @param name class name
     * @param resolveIt
     * @return loaded class or null if not found
     */
    private Class<?> loadFromClassPath(String name, boolean resolveIt) {
        // Try to load it from our repository
        String[] classPath = _classPathString.split(PATH_SEPARATOR_STRING);

        byte[] classData = null;
        String sourceClassPathEntry = "";
        for(String classPathEntry: classPath) {
            classData = loadBytes(name.replace('.','/'), classPathEntry);
            if (null != classData) {
                sourceClassPathEntry = classPathEntry;
                break;
            }
        }
        if ((null == classData) || (sourceClassPathEntry.startsWith(JAVA_HOME_STRING)) ) {
            // not found or found from rt.jar; try system loader
            return null;
        }

        // Perform the check
        checkClass(name, classData);

        // Define it (parse the class file)
        Class<?> result = defineClass(name, classData, 0, classData.length);
        if (null == result) {
            throw new ClassFormatError();
        }

        if (resolveIt) {
            resolveClass(result);
        }

        _classes.put(name, result);

        return result;
    }

    /**
     * Load byte array from name and path.
     * @param name class file name (without .class)
     * @param path path to look in (or jar file)
     * @return byte array
     */
    private byte[] loadBytes(String name, String path) {
        byte[] classData;
        try {
            InputStream fi = null;
            if (path.toLowerCase().endsWith(".jar")) {
                JarFile jar = new JarFile(path);
                JarEntry entry = jar.getJarEntry(name+".class");
                if (null == entry) {
                    return null;
                }
                fi = jar.getInputStream(entry);
            }
            else {
                fi = new FileInputStream(path+SEPARATOR_CHAR+name+".class");
            }
            classData = loadFromStream(fi);
        }
        catch (IOException e) {
            return null;
        }

        return classData;
    }

    /**
     * Load byte array from stream.
     * @param fi input stream
     * @return byte array
     * @throws IOException
     */
    private byte[] loadFromStream(InputStream fi) throws IOException {
        byte[] classData;
        classData = new byte[fi.available()];
        fi.read(classData);
        return classData;
    }

    /**
     * Check if the class is safe to load.
     * @param name class name
     * @param classData bytes of the class file
     */
    public void checkClass(final String name, byte[] classData) {
        try {
            ClassFile cf = new ClassFile(classData);

            // scan constant pool
            Boolean found = false;
            ADefaultPoolInfoVisitor<Boolean, Object> checkClassUsageVisitor = new ADefaultPoolInfoVisitor<Boolean, Object>() {
                public Boolean classCase(ClassPoolInfo host, Object param) {
                    return (("java/lang/Throwable".equals(host.getName().toString())) ||
                            ("java/lang/Error".equals(host.getName().toString())) ||
                            ("java/lang/ThreadDeath".equals(host.getName().toString())));
                }

                public Boolean defaultCase(APoolInfo host, Object param) {
                    return false;
                }
            };
            for (APoolInfo pi: cf.getConstantPool()) {
                found |= pi.execute(checkClassUsageVisitor, null);
            }
            if (!found) {
                // if none of these classes are referenced, then nothing in this class
                // can catch ThreadDeath, so the class is safe
                return;
            }


            ADefaultPoolInfoVisitor<Object, MethodInfo> checkThrowVisitor = new ADefaultPoolInfoVisitor<Object, MethodInfo>() {
                public Object classCase(ClassPoolInfo host, MethodInfo param) {
                    if (("java/lang/Throwable".equals(host.getName().toString())) ||
                        ("java/lang/Error".equals(host.getName().toString())) ||
                        ("java/lang/ThreadDeath".equals(host.getName().toString()))) {
                        throw new ClassFormatError("Class " + name + " is potentially unsafe: catch clause in " + param
                                                   + " too general");
                    }
                    return null;
                }

                public Object defaultCase(APoolInfo host, MethodInfo param) {
                    // not a class, should never happen
                    throw new ClassFormatError("Exception table entry's catch type in " + name + '.' + param
                                               + " references non-class constant pool item.");
                }
            };

            // scan methods
            for (final MethodInfo mi: cf.getMethods()) {
                if (0 != (mi.getAccessFlags() & (ClassFile.ACC_ABSTRACT | ClassFile.ACC_NATIVE))) {
                    continue;
                }

                CodeAttributeInfo ca;

                try {
                    ca = mi.getCodeAttributeInfo();
                }
                catch (ClassFormatError cfe) {
                    continue;
                }

                // check all exception table entries
                for (CodeAttributeInfo.ExceptionTableEntry eta: ca.getExceptionTableEntries()) {
                    if (0 != eta.catchType) {
                        // if it's not a catch-all from the implementation of finally, we need
                        // to check the type of exception caught
                        cf.getConstantPoolItem(eta.catchType).execute(checkThrowVisitor, null);
                    }
                }
            }

        }
        catch (IOException e) {
            throw new ClassFormatError("Could not check class "+name+"for safety: "+e);
        }
    }

    /**
     * Array of built-in classes that should not be checked and be loaded with the regular class loader.
     */
    private static final String[] BUILTIN_CLASSES = new String[] {
        "controller.IDisplayAdapter",
        "controller.IEditAdapter",
        "controller.IEnvAdapter",
        "controller.IScrollAdapter",
        "controller.ISimAdapter",
        "controller.MBSController$1",
        "controller.MBSController$2",
        "controller.MBSController$3",
        "controller.MBSController$4",
        "controller.MBSController",
        "lrs.ANode$1$1",
        "lrs.ANode$1",
        "lrs.ANode",
        "lrs.EmptyNode",
        "lrs.IAlgo",
        "lrs.LRStruct",
        "lrs.NENode",
        "lrs.visitor.Apply",
        "lrs.visitor.Clear",
        "lrs.visitor.GetLength$1",
        "lrs.visitor.GetLength",
        "lrs.visitor.GetNth",
        "lrs.visitor.Remove",
        "model.ILambda",
        "model.RandNumGenerator",
        "sysModel.env.AEnvFactory",
        "sysModel.env.AGlobalEnv$1",
        "sysModel.env.AGlobalEnv$2",
        "sysModel.env.AGlobalEnv$3",
        "sysModel.env.AGlobalEnv$4",
        "sysModel.env.AGlobalEnv$5$1",
        "sysModel.env.AGlobalEnv$5$2",
        "sysModel.env.AGlobalEnv$5$3",
        "sysModel.env.AGlobalEnv$5$4",
        "sysModel.env.AGlobalEnv$5",
        "sysModel.env.AGlobalEnv$6",
        "sysModel.env.AGlobalEnv$ALocalEnv$1",
        "sysModel.env.AGlobalEnv$ALocalEnv$2",
        "sysModel.env.AGlobalEnv$ALocalEnv",
        "sysModel.env.AGlobalEnv$BreedLambda",
        "sysModel.env.AGlobalEnv$ILocalEnvVisitor",
        "sysModel.env.AGlobalEnv",
        "sysModel.env.ASquareEnv$1",
        "sysModel.env.ASquareEnv$2",
        "sysModel.env.ASquareEnv$3$1",
        "sysModel.env.ASquareEnv$3$2$1",
        "sysModel.env.ASquareEnv$3$2",
        "sysModel.env.ASquareEnv$3",
        "sysModel.env.ASquareEnv$ASquareLocalEnvironment",
        "sysModel.env.ASquareEnv$Direction$1$1$1$1$1",
        "sysModel.env.ASquareEnv$Direction$1$1$1$1",
        "sysModel.env.ASquareEnv$Direction$1$1$1",
        "sysModel.env.ASquareEnv$Direction$1$1",
        "sysModel.env.ASquareEnv$Direction$1",
        "sysModel.env.ASquareEnv$Direction",
        "sysModel.env.ASquareEnv$Location$1$1$1$1$1",
        "sysModel.env.ASquareEnv$Location$1$1$1$1",
        "sysModel.env.ASquareEnv$Location$1$1$1",
        "sysModel.env.ASquareEnv$Location$1$1",
        "sysModel.env.ASquareEnv$Location$1",
        "sysModel.env.ASquareEnv$Location",
        "sysModel.env.ASquareEnv",
        "sysModel.env.BoundedEnv$1",
        "sysModel.env.BoundedEnv$2$1",
        "sysModel.env.BoundedEnv$2",
        "sysModel.env.BoundedEnv$3",
        "sysModel.env.BoundedEnv$EmptyField",
        "sysModel.env.BoundedEnv$IField",
        "sysModel.env.BoundedEnv$IFieldVisitor",
        "sysModel.env.BoundedEnv$LocalEnvironment$MoveLambda",
        "sysModel.env.BoundedEnv$LocalEnvironment",
        "sysModel.env.BoundedEnv$NonEmptyField",
        "sysModel.env.BoundedEnv$Test_BoundedEnv$1",
        "sysModel.env.BoundedEnv$Test_BoundedEnv$2",
        "sysModel.env.BoundedEnv$Test_BoundedEnv$3",
        "sysModel.env.BoundedEnv$Test_BoundedEnv$4",
        "sysModel.env.BoundedEnv$Test_BoundedEnv$5",
        "sysModel.env.BoundedEnv$Test_BoundedEnv$6",
        "sysModel.env.BoundedEnv$Test_BoundedEnv$7",
        "sysModel.env.BoundedEnv$Test_BoundedEnv",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$1",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$10",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$11",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$12",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$13",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$14",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$15",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$16",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$17",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$18",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$19",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$2",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$20",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$21",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$22",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$23",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$24",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$25",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$26",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$27",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$28",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$29",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$3",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$30",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$31",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$32",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$33",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$34",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$35",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$36",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$37",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$38",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$39",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$4",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$40",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$41",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$42",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$43",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$44",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$45",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$46",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$47",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$48",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$49",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$5",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$50",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$51",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$52",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$6",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$7",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$8",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$9",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv$SuccessException",
        "sysModel.env.BoundedEnv$Test_BoundedEnv_LocalEnv",
        "sysModel.env.BoundedEnv",
        "sysModel.env.DeactivatableLambda",
        "sysModel.env.EmptyLocalEnvState",
        "sysModel.env.FishApplyParams",
        "sysModel.env.IBlockedCommand",
        "sysModel.env.ILocalEnvState",
        "sysModel.env.IOpenCommand",
        "sysModel.env.NoGridEnv$1",
        "sysModel.env.NoGridEnv$2",
        "sysModel.env.NoGridEnv$3$1$1$1",
        "sysModel.env.NoGridEnv$3$1$1",
        "sysModel.env.NoGridEnv$3$1",
        "sysModel.env.NoGridEnv$3",
        "sysModel.env.NoGridEnv$4",
        "sysModel.env.NoGridEnv$5",
        "sysModel.env.NoGridEnv$6$1",
        "sysModel.env.NoGridEnv$6$2$1",
        "sysModel.env.NoGridEnv$6$2",
        "sysModel.env.NoGridEnv$6",
        "sysModel.env.NoGridEnv$Direction$1$1$1$1$1",
        "sysModel.env.NoGridEnv$Direction$1$1$1$1",
        "sysModel.env.NoGridEnv$Direction$1$1$1",
        "sysModel.env.NoGridEnv$Direction$1$1",
        "sysModel.env.NoGridEnv$Direction$1",
        "sysModel.env.NoGridEnv$Direction",
        "sysModel.env.NoGridEnv$LocalEnvironment$MoveLambda",
        "sysModel.env.NoGridEnv$LocalEnvironment",
        "sysModel.env.NoGridEnv$Location$1$1$1$1$1",
        "sysModel.env.NoGridEnv$Location$1$1$1$1",
        "sysModel.env.NoGridEnv$Location$1$1$1",
        "sysModel.env.NoGridEnv$Location$1$1",
        "sysModel.env.NoGridEnv$Location$1",
        "sysModel.env.NoGridEnv$Location",
        "sysModel.env.NoGridEnv$Test_NoGridEnv$1",
        "sysModel.env.NoGridEnv$Test_NoGridEnv$2",
        "sysModel.env.NoGridEnv$Test_NoGridEnv$3",
        "sysModel.env.NoGridEnv$Test_NoGridEnv$4",
        "sysModel.env.NoGridEnv$Test_NoGridEnv$5",
        "sysModel.env.NoGridEnv$Test_NoGridEnv$6",
        "sysModel.env.NoGridEnv$Test_NoGridEnv",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$1",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$10",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$11",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$12",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$13",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$14",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$15",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$16",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$17",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$18",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$19",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$2",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$20",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$21",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$22",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$23",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$24",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$25",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$26",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$27",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$28",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$29",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$3",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$30",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$31",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$32",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$33",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$34",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$35",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$36",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$37",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$38",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$39",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$4",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$40",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$41",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$42",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$43",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$44",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$45",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$46",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$47",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$48",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$49",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$5",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$50",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$51",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$52",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$6",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$7",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$8",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$9",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv$SuccessException",
        "sysModel.env.NoGridEnv$Test_NoGridEnv_LocalEnv",
        "sysModel.env.NoGridEnv",
        "sysModel.env.NonEmptyLocalEnvState",
        "sysModel.env.NullEnv$1",
        "sysModel.env.NullEnv$2",
        "sysModel.env.NullEnv",
        "sysModel.env.tests.Test_ASquareEnv_Direction$1",
        "sysModel.env.tests.Test_ASquareEnv_Direction$2",
        "sysModel.env.tests.Test_ASquareEnv_Direction$3",
        "sysModel.env.tests.Test_ASquareEnv_Direction$4",
        "sysModel.env.tests.Test_ASquareEnv_Direction$5",
        "sysModel.env.tests.Test_ASquareEnv_Direction$6",
        "sysModel.env.tests.Test_ASquareEnv_Direction",
        "sysModel.env.tests.Test_ASquareEnv_Location$1",
        "sysModel.env.tests.Test_ASquareEnv_Location$2",
        "sysModel.env.tests.Test_ASquareEnv_Location$3",
        "sysModel.env.tests.Test_ASquareEnv_Location$4",
        "sysModel.env.tests.Test_ASquareEnv_Location$5",
        "sysModel.env.tests.Test_ASquareEnv_Location$6",
        "sysModel.env.tests.Test_ASquareEnv_Location",
        "sysModel.env.tests.Test_DeactivatableLambda$1",
        "sysModel.env.tests.Test_DeactivatableLambda$2",
        "sysModel.env.tests.Test_DeactivatableLambda$3",
        "sysModel.env.tests.Test_DeactivatableLambda",
        "sysModel.env.UnboundedEnv$1",
        "sysModel.env.UnboundedEnv$2",
        "sysModel.env.UnboundedEnv$LocalEnvironment$MoveLambda",
        "sysModel.env.UnboundedEnv$LocalEnvironment",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv$1",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv$2",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv$3",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv$4",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv$5",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv$6",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$1",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$10",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$11",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$12",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$13",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$14",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$15",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$16",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$17",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$18",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$19",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$2",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$20",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$21",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$22",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$23",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$24",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$25",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$26",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$27",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$28",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$29",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$3",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$30",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$31",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$32",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$33",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$34",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$35",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$36",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$37",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$38",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$39",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$4",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$40",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$41",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$42",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$43",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$44",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$45",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$46",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$47",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$48",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$49",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$5",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$50",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$6",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$7",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$8",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$9",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv$SuccessException",
        "sysModel.env.UnboundedEnv$Test_UnboundedEnv_LocalEnv",
        "sysModel.env.UnboundedEnv",
        "sysModel.env.WrappingEnv$1",
        "sysModel.env.WrappingEnv$2$1",
        "sysModel.env.WrappingEnv$2",
        "sysModel.env.WrappingEnv$Location",
        "sysModel.env.WrappingEnv$Test_WrappingEnv$1",
        "sysModel.env.WrappingEnv$Test_WrappingEnv$2",
        "sysModel.env.WrappingEnv$Test_WrappingEnv$3",
        "sysModel.env.WrappingEnv$Test_WrappingEnv$4",
        "sysModel.env.WrappingEnv$Test_WrappingEnv$5",
        "sysModel.env.WrappingEnv$Test_WrappingEnv$6",
        "sysModel.env.WrappingEnv$Test_WrappingEnv$7",
        "sysModel.env.WrappingEnv$Test_WrappingEnv",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$1",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$10",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$11",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$12",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$13",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$14",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$15",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$16",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$17",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$18",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$19",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$2",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$20",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$21",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$22",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$23",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$24",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$25",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$26",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$27",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$28",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$29",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$3",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$30",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$31",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$32",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$33",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$34",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$35",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$36",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$37",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$38",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$39",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$4",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$40",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$41",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$42",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$43",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$44",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$45",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$46",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$47",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$48",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$49",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$5",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$50",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$51",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$52",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$6",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$7",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$8",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$9",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv$SuccessException",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_LocalEnv",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_Location$1",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_Location$2",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_Location$3",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_Location$4",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_Location$5",
        "sysModel.env.WrappingEnv$Test_WrappingEnv_Location",
        "sysModel.env.WrappingEnv",
        "sysModel.fish.DynamicFishFactory",
        "sysModel.fish.AFish",
        "sysModel.fish.FishException",
        "sysModel.fish.AFish$1",
        "sysModel.fish.AFish$2$1",
        "sysModel.fish.AFish$2",
        "sysModel.fish.AFish$3$1",
        "sysModel.fish.AFish$3",
        "sysModel.fish.AFish$4$1",
        "sysModel.fish.AFish$4",
        "sysModel.fish.AFish$5",
        "sysModel.fish.IFishBehavior",
        "sysModel.fish.IFishDisplay",
        "sysModel.fish.IFishFactory",
        "sysModel.fish.tests.Test_Fish$1",
        "sysModel.fish.tests.Test_Fish$2",
        "sysModel.fish.tests.Test_Fish$3",
        "sysModel.fish.tests.Test_Fish$4",
        "sysModel.fish.tests.Test_Fish$5",
        "sysModel.fish.tests.Test_Fish$6",
        "sysModel.fish.tests.Test_Fish$7",
        "sysModel.fish.tests.Test_Fish$TestFish$1",
        "sysModel.fish.tests.Test_Fish$TestFish$2",
        "sysModel.fish.tests.Test_Fish$TestFish",
        "sysModel.fish.tests.Test_Fish$TestGlobalEnv$1",
        "sysModel.fish.tests.Test_Fish$TestGlobalEnv",
        "sysModel.fish.tests.Test_Fish",
        "sysModel.ICmdFactory",
        "sysModel.ISecurityAdapter",
        "sysModel.MBSClassLoader",
        "sysModel.MBSSecurityManager",
        "sysModel.NoOpLambda",
        "sysModel.parser.CloseToken",
        "sysModel.parser.CommaToken",
        "sysModel.parser.DefaultTokenVisitor",
        "sysModel.parser.EndOfStreamToken",
        "sysModel.parser.IToken",
        "sysModel.parser.ITokenVisitor",
        "sysModel.parser.Lexer",
        "sysModel.parser.NumberToken",
        "sysModel.parser.OpenToken",
        "sysModel.parser.Parser$1",
        "sysModel.parser.Parser",
        "sysModel.parser.ParserException",
        "sysModel.parser.WordToken",
        "sysModel.PolicyFile$1",
        "sysModel.PolicyFile",
        "sysModel.SimDriver$1$1",
        "sysModel.SimDriver$1$2",
        "sysModel.SimDriver$1$3",
        "sysModel.SimDriver$1",
        "sysModel.SimDriver$2",
        "sysModel.SimDriver$3",
        "sysModel.SimDriver$4",
        "sysModel.SimDriver",
        "sysModel.SimEngine$1",
        "sysModel.SimEngine",
        "tests.DrJava_Test_BoundedEnv",
        "tests.DrJava_Test_BoundedEnv_LocalEnv",
        "tests.DrJava_Test_NoGridEnv",
        "tests.DrJava_Test_NoGridEnv_LocalEnv",
        "tests.DrJava_Test_UnboundedEnv",
        "tests.DrJava_Test_UnboundedEnv_LocalEnv",
        "tests.DrJava_Test_WrappingEnv",
        "tests.DrJava_Test_WrappingEnv_LocalEnv",
        "tests.DrJava_Test_WrappingEnv_Location",
        "view.CreateEnvDialog$1",
        "view.CreateEnvDialog$2",
        "view.CreateEnvDialog$3",
        "view.CreateEnvDialog$AddEnvChoice",
        "view.CreateEnvDialog$ConcreteEnvChoice",
        "view.CreateEnvDialog$IEnvChoice",
        "view.CreateEnvDialog",
        "view.DisplayPanel$1",
        "view.DisplayPanel",
        "view.DisplayViewport$1",
        "view.DisplayViewport$Pannable",
        "view.DisplayViewport",
        "view.EditToolbar$1",
        "view.EditToolbar$2",
        "view.EditToolbar$3",
        "view.EditToolbar$4",
        "view.EditToolbar$5",
        "view.EditToolbar$ChoiceWithIcon",
        "view.EditToolbar$ChoiceWithIconRenderer",
        "view.EditToolbar$ColorChoice",
        "view.EditToolbar$ColorIcon",
        "view.EditToolbar$FishChoice",
        "view.EditToolbar$FishIcon",
        "view.EditToolbar",
        "view.EnvFileChooser$1",
        "view.EnvFileChooser",
        "view.InputStringDialog$1",
        "view.InputStringDialog$2",
        "view.InputStringDialog",
        "view.IRunIdleAdapter",
        "view.MBSView$1",
        "view.MBSView$10$1",
        "view.MBSView$10",
        "view.MBSView$11$1",
        "view.MBSView$11",
        "view.MBSView$12$1",
        "view.MBSView$12",
        "view.MBSView$13$1",
        "view.MBSView$13",
        "view.MBSView$14$1",
        "view.MBSView$14",
        "view.MBSView$15",
        "view.MBSView$16",
        "view.MBSView$17",
        "view.MBSView$18",
        "view.MBSView$19",
        "view.MBSView$2",
        "view.MBSView$20",
        "view.MBSView$21",
        "view.MBSView$3",
        "view.MBSView$4",
        "view.MBSView$5",
        "view.MBSView$6",
        "view.MBSView$7",
        "view.MBSView$8",
        "view.MBSView$9",
        "view.MBSView$GUIExceptionHandler",
        "view.MBSView$StepItLambda",
        "view.MBSView",
        "view.SimToolbar$1",
        "view.SimToolbar$2",
        "view.SimToolbar$3",
        "view.SimToolbar$4",
        "view.SimToolbar"
    };
}
