package sysModel;

import sysModel.fish.AFish;

import javax.sound.sampled.AudioPermission;
import java.awt.*;
import java.io.FilePermission;
import java.io.SerializablePermission;
import java.lang.reflect.ReflectPermission;
import java.net.NetPermission;
import java.net.SocketPermission;
import java.security.Permission;
import java.security.SecurityPermission;
import java.sql.SQLPermission;
import java.util.PropertyPermission;
import java.util.Stack;

/**
 * Special security manager that does not permit fish to call System.exit() or System.setSecurityManager().
 *
 * @author Mathias Ricken
 */
public class MBSSecurityManager extends SecurityManager {
    /**
     * Constructor.
     * @param drJava true if DrJava is running
     */
    public MBSSecurityManager(boolean drJava) {
        // TODO: Should create new security manager and load policy, but using the command line and DrJava is problematic right now
        //_sm = new SecurityManager();
        //PolicyFile pf = new PolicyFile();
        //Policy.setPolicy(pf);
        _isProtected = true;
        _classPath = System.getProperty("java.class.path");
        _pathSep = System.getProperty("path.separator");
        _classPathDirs = _classPath.split(_pathSep);
        _drJava = drJava;
    }

    /**
     * True if DrJava is present.
     */
    private boolean _drJava = false;

    /**
     * Flag if actions are protected.
     */
    private boolean _isProtected = true;

    /**
     * SecurityManager to delegate to.
     */
    private SecurityManager _sm = new SecurityManager();

    /**
     * Class path.
     */
    private String _classPath;

    /**
     * Path separator.
     */
    private String _pathSep;

    /**
     * Class path directories.
     */
    private String[] _classPathDirs;

    /**
     * Fish thread group.
     */
    private ThreadGroup _fishThreadGroup = new ThreadGroup("FishThreadGroup");

    /**
     * Class loader.
     */
    private MBSClassLoader _classLoader = new MBSClassLoader(this);

    /**
     * Stack of protection flags.
     */
    private Stack<Boolean> _protectionFlagStack = new Stack<Boolean>();

    /**
     * Set the protection flag.
     *
     * @param isProtected true if actions are to be protected
     */
    public void setProtected(boolean isProtected) {
        _isProtected = isProtected;
    }

    /**
     * Get the protection flag.
     * @return true if actions are to be protected
     */
    public boolean isProtected() {
        return _isProtected;
    }

    /**
     * Push the current protection flag onto the stack and set a new value.
     * @param isProtected true if actions are to be protected
     */
    public void pushProtected(boolean isProtected) {
        _protectionFlagStack.push(_isProtected);
        _isProtected = isProtected;
    }

    /**
     * Pops the top protection flag from the stack and sets it as new value.
     */
    public void popProtected() {
        _isProtected = _protectionFlagStack.pop();
    }

    /**
     * Returns true if the filename is on the class path.
     *
     * @param filename filename to test
     * @return true if on classpath
     */
    protected boolean isOnClassPath(String filename) {
        //System.out.println("Checking "+filename);
        for (int i = 0; i < _classPathDirs.length; ++i) {
            //System.out.println("\t"+_classPathDirs[i]);
            if (filename.startsWith(_classPathDirs[i])) {
                return true;
            }
        }

        return false;
    }

    /**
     * Returns true if a fish is somewhere on the call stack.
     *
     * @return true if fish is a caller
     */
    protected boolean isFishCaller() {
        Class[] classes = getClassContext();
        for (int i = 0; i < classes.length; ++i) {
            if (AFish.class.isAssignableFrom(classes[i])) {
                // System.out.println("Found a fish class: "+classes[0].getName());
                return true;
            }
        }
        return false;
    }

    /**
     * Throws a <code>SecurityException</code> if the requested access, specified by the given permission, is not
     * permitted based on the security policy currently in effect.
     * <p/>
     * This method calls <code>AccessController.checkPermission</code> with the given permission.
     *
     * @param perm the requested permission.
     * @throws SecurityException    if access is not permitted based on the current security policy.
     * @throws NullPointerException if the permission argument is <code>null</code>.
     * @since 1.2
     */
    public void checkPermission(Permission perm) {
        if (_drJava) {
            return;
        }

        boolean wasProtected = _isProtected;
        try {
            pushProtected(false);

            if (wasProtected) {
                if ((null != Thread.currentThread().getThreadGroup()) && (Thread.currentThread().getThreadGroup()==_fishThreadGroup) && (isFishCaller())) {
                    // System.err.println("checkPermission: "+perm);
                    if ((perm instanceof RuntimePermission)) {
                        // disallow
                        throw new SecurityException("Not allowed to deal with runtime!");
                    }
                    else if (perm instanceof AudioPermission) {
                        // disallow
                        throw new SecurityException("Not allowed to deal with audio!");
                    }
                    else if (perm instanceof AWTPermission) {
                        // disallow
                        throw new SecurityException("Not allowed to deal with AWT!");
                    }
                    else if ((perm instanceof FilePermission) &&
                            ((!"read".equals(perm.getActions())) || ((!perm.getName().endsWith(".class") && (!perm.getName().endsWith(".jar"))))/*|| (!isOnClassPath(perm.getName()))*/)) {
                        // TODO: isOnClassPath should be checked, but jar files and DrJava are problematic right now
                        // disallow
                        throw new SecurityException("Not allowed to deal with files!");
                    }
                    else if (perm instanceof NetPermission) {
                        // disallow
                        throw new SecurityException("Not allowed to deal with network!");
                    }
                    else if ((perm instanceof PropertyPermission) &&
                            ((!"read".equals(perm.getActions())) || !"line.separator".equals(perm.getName()))) {
                        // disallow
                        StackTraceElement[] stes = Thread.currentThread().getStackTrace();
                        for (StackTraceElement ste: stes) {
                            System.out.println("### "+ste);
                        }
                        throw new SecurityException("Not allowed to deal with properties!");
                    }
                    else if (perm instanceof ReflectPermission) {
                        // disallow
                        throw new SecurityException("Not allowed to do reflection!");
                    }
                    else if (perm instanceof SecurityPermission) {
                        // disallow
                        throw new SecurityException("Not allowed to deal with security!");
                    }
                    else if (perm instanceof SerializablePermission) {
                        // disallow
                        throw new SecurityException("Not allowed to deal with serialization!");
                    }
                    else if (perm instanceof SocketPermission) {
                        // disallow
                        throw new SecurityException("Not allowed to deal with sockets!");
                    }
                    else if (perm instanceof SQLPermission) {
                        // disallow
                        throw new SecurityException("Not allowed to deal with SQL!");
                    }
                }
            }

            // delegate
            // TODO: Should delegate, but using the command line and DrJava is problematic right now
            //_sm.checkPermission(perm);
        }
        finally {
           popProtected();
        }
    }

    /**
     * Throws a <code>SecurityException</code> if the calling thread is not allowed to modify the thread argument.
     * <p/>
     * This method is invoked for the current security manager by the <code>stop</code>, <code>suspend</code>,
     * <code>resume</code>, <code>setPriority</code>, <code>setName</code>, and <code>setDaemon</code> methods of class
     * <code>Thread</code>.
     * <p/>
     * If the thread argument is a system thread (belongs to the thread group with a <code>null</code> parent) then this
     * method calls <code>checkPermission</code> with the <code>RuntimePermission("modifyThread")</code> permission. If
     * the thread argument is <i>not</i> a system thread, this method just returns silently.
     * <p/>
     * Applications that want a stricter policy should override this method. If this method is overridden, the method
     * that overrides it should additionally check to see if the calling thread has the
     * <code>RuntimePermission("modifyThread")</code> permission, and if so, return silently. This is to ensure that
     * code granted that permission (such as the SDK itself) is allowed to manipulate any thread.
     * <p/>
     * If this method is overridden, then <code>super.checkAccess</code> should be called by the first statement in the
     * overridden method, or the equivalent security check should be placed in the overridden method.
     *
     * @param t the thread to be checked.
     * @throws SecurityException    if the calling thread does not have permission to modify the thread.
     * @throws NullPointerException if the thread argument is <code>null</code>.
     * @see Thread#resume() resume
     * @see Thread#setDaemon(boolean) setDaemon
     * @see Thread#setName(String) setName
     * @see Thread#setPriority(int) setPriority
     * @see Thread#stop() stop
     * @see Thread#suspend() suspend
     * @see #checkPermission(Permission) checkPermission
     */
    public void checkAccess(Thread t) {
        if (_drJava) {
            return;
        }

        //System.err.println("checkAccess: Thread "+t.getId());

        super.checkAccess(t);

        if (_isProtected) {
            checkPermission(new RuntimePermission("modifyThread"));
        }
    }

    /**
     * Throws a <code>SecurityException</code> if the calling thread is not allowed to modify the thread group
     * argument.
     * <p/>
     * This method is invoked for the current security manager when a new child thread or child thread group is created,
     * and by the <code>setDaemon</code>, <code>setMaxPriority</code>, <code>stop</code>, <code>suspend</code>,
     * <code>resume</code>, and <code>destroy</code> methods of class <code>ThreadGroup</code>.
     * <p/>
     * If the thread group argument is the system thread group ( has a <code>null</code> parent) then this method calls
     * <code>checkPermission</code> with the <code>RuntimePermission("modifyThreadGroup")</code> permission. If the
     * thread group argument is <i>not</i> the system thread group, this method just returns silently.
     * <p/>
     * Applications that want a stricter policy should override this method. If this method is overridden, the method
     * that overrides it should additionally check to see if the calling thread has the
     * <code>RuntimePermission("modifyThreadGroup")</code> permission, and if so, return silently. This is to ensure
     * that code granted that permission (such as the SDK itself) is allowed to manipulate any thread.
     * <p/>
     * If this method is overridden, then <code>super.checkAccess</code> should be called by the first statement in the
     * overridden method, or the equivalent security check should be placed in the overridden method.
     *
     * @param g the thread group to be checked.
     * @throws SecurityException    if the calling thread does not have permission to modify the thread group.
     * @throws NullPointerException if the thread group argument is <code>null</code>.
     * @see ThreadGroup#destroy() destroy
     * @see ThreadGroup#resume() resume
     * @see ThreadGroup#setDaemon(boolean) setDaemon
     * @see ThreadGroup#setMaxPriority(int) setMaxPriority
     * @see ThreadGroup#stop() stop
     * @see ThreadGroup#suspend() suspend
     * @see #checkPermission(Permission) checkPermission
     */
    public void checkAccess(ThreadGroup g) {
        if (_drJava) {
            return;
        }

        //System.err.println("checkAccess: ThreadGroup "+g.getName());

        super.checkAccess(g);

        if ((_isProtected) && (null != Thread.currentThread().getThreadGroup()) && (Thread.currentThread().getThreadGroup()==_fishThreadGroup)) {
            if (!"system".equals(g.getName())) {
                throw new SecurityException("Not allowed to manipulate threads!");
            }
            checkPermission(new RuntimePermission("modifyThreadGroup"));
        }
    }

    /**
     * Throws a <code>SecurityException</code> if the calling thread is not allowed to cause the Java Virtual Machine to
     * halt with the specified status code.
     * <p/>
     * This method is invoked for the current security manager by the <code>exit</code> method of class
     * <code>Runtime</code>. A status of <code>0</code> indicates success; other values indicate various errors.
     * <p/>
     * This method calls <code>checkPermission</code> with the <code>RuntimePermission("exitVM")</code> permission.
     * <p/>
     * If you override this method, then you should make a call to <code>super.checkExit</code> at the point the
     * overridden method would normally throw an exception.
     *
     * @param status the exit status.
     * @throws SecurityException if the calling thread does not have permission to halt the Java Virtual Machine with
     *                           the specified status.
     * @see Runtime#exit(int) exit
     * @see #checkPermission(Permission) checkPermission
     */
    public void checkExit(int status) {
        if (_drJava) {
            return;
        }

        if (_isProtected) {
            throw new SecurityException("Not allowed to call System.exit()!");
        }
        else {
            // delegate
            // TODO: Should delegate, but using the command line and DrJava is problematic right now
            //_sm.checkExit(status);
        }
    }

    /**
     * Get the thread group the fish are run in.
     * @return fish thread group
     */
    public ThreadGroup getFishThreadGroup() {
        return _fishThreadGroup;
    }

    /**
     * Return the class loader.
     * @return class loader
     */
    public ClassLoader getClassLoader() {
        if (_drJava) {
            return getClass().getClassLoader();
        }
        else {
            return _classLoader;
        }
    }
}
