package sysModel.classFile;

import sysModel.classFile.attributes.AAttributeInfo;
import sysModel.classFile.attributes.SourceFileAttributeInfo;
import sysModel.classFile.constantPool.*;
import sysModel.classFile.constantPool.visitors.*;

import java.io.*;
import java.util.ArrayList;

/**
 * Represents a Java class file.
 *
 * @author Mathias Ricken
 */
public class ClassFile {
    /**
     * Major version.
     */
    private short _majorVersion;

    /**
     * Minor version.
     */
    private short _minorVersion;

    /**
     * Constant pool.
     */
    private ConstantPool _constantPool = new ConstantPool();

    /**
     * Class access flags.
     */
    private short _classAccessFlags;

    /**
     * Class information about this class.
     */
    private ClassPoolInfo _thisClass;

    /**
     * Class information about the superclass.
     */
    private ClassPoolInfo _superClass;

    /**
     * Class information about implemented interfaces.
     */
    private ArrayList<ClassPoolInfo> _interfaces = new ArrayList<ClassPoolInfo>();

    /**
     * Fields in the class.
     */
    private ArrayList<FieldInfo> _fields = new ArrayList<FieldInfo>();

    /**
     * Methods in the class.
     */
    private ArrayList<MethodInfo> _methods = new ArrayList<MethodInfo>();

    /**
     * Attributes of the class.
     */
    private ArrayList<AAttributeInfo> _attributes = new ArrayList<AAttributeInfo>();

    // Access flags
    // TODO: Separate class and method access flags?
    public static final int ACC_PUBLIC = 0x1;
    public static final int ACC_PRIVATE = 0x2;
    public static final int ACC_PROTECTED = 0x4;
    public static final int ACC_STATIC = 0x8;
    public static final int ACC_FINAL = 0x10;
    public static final int ACC_SYNCHRONIZED = 0x20;
    public static final int ACC_VOLATILE = 0x40;
    public static final int ACC_BRIDGE = 0x40; // for methods
    public static final int ACC_TRANSIENT = 0x80;
    public static final int ACC_VARARGS = 0x80; // for methods
    public static final int ACC_NATIVE = 0x100;
    public static final int ACC_INTERFACE = 0x200;
    public static final int ACC_ABSTRACT = 0x400;
    public static final int ACC_STRICT = 0x800;
    public static final int ACC_SYNTHETIC = 0x1000;
    public static final int ACC_ANNOTATION = 0x2000;
    public static final int ACC_ENUM = 0x4000;

    /**
     * Java file magic, 0xCAFEBABE.
     */
    private static final int JAVA_FILE_MAGIC = 0xCAFEBABE;

    /**
     * Create a new ClassFile instance.
     * @param majorVersion major class file version
     * @param minorVersion minor class file version
     * @param classAccessFlags class access flags
     * @param thisClassName class name
     * @param superClassName superclass name
     */
    public ClassFile(short majorVersion,
                     short minorVersion,
                     short classAccessFlags,
                     String thisClassName,
                     String superClassName) {
        _majorVersion = majorVersion;
        _minorVersion = minorVersion;

        _constantPool = new ConstantPool(1);
        _constantPool.add(EmptyPoolInfo.singleton());

        // Resolve and check constant pool
        for(APoolInfo cpi : _constantPool) {
            cpi.resolve();
        }

        _classAccessFlags = classAccessFlags;

        // add a new name
        AUTFPoolInfo thisClassNameInfo = new ASCIIPoolInfo(thisClassName, _constantPool);
        short[] l = addConstantPoolItems(new APoolInfo[]{thisClassNameInfo});
        thisClassNameInfo = getConstantPoolItem(l[0]).execute(CheckUTFVisitor.singleton(), null);

        // add a new class
        _thisClass = new ClassPoolInfo(thisClassNameInfo, _constantPool);
        l = addConstantPoolItems(new APoolInfo[]{_thisClass});
        _thisClass = getConstantPoolItem(l[0]).execute(CheckClassVisitor.singleton(), null);

        if (!"java/lang/Object".equals(thisClassName)) {
            // add a new name
            AUTFPoolInfo superClassNameInfo = new ASCIIPoolInfo(superClassName, _constantPool);
            l = addConstantPoolItems(new APoolInfo[]{superClassNameInfo});
            superClassNameInfo = getConstantPoolItem(l[0]).execute(CheckUTFVisitor.singleton(), null);

            // add a new class
            _superClass = new ClassPoolInfo(superClassNameInfo, _constantPool);
            l = addConstantPoolItems(new APoolInfo[]{_superClass});
            _superClass = getConstantPoolItem(l[0]).execute(CheckClassVisitor.singleton(), null);
        }

        _interfaces = new ArrayList<ClassPoolInfo>(0);
        _fields = new ArrayList<FieldInfo>(0);
        _methods = new ArrayList<MethodInfo>(0);
        _attributes = new ArrayList<AAttributeInfo>(0);
    }

    /**
     * Constructor.
     *
     * @param in input stream with class file
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    public ClassFile(InputStream in) throws IOException, ClassFormatError {
        DataInputStream di = new DataInputStream(in);

        readHeader(di);
        readConstantPool(di);

        readClassInfo(di);

        readInterfaces(di);
        readFields(di);
        readMethods(di);
        readAttributes(di);
    }

    /**
     * Constructor.
     *
     * @param b byte array
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    public ClassFile(byte[] b) throws IOException, ClassFormatError {
        this(new ByteArrayInputStream(b));
    }

    /**
     * Read a class file header.
     *
     * @param di stream
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    protected void readHeader(DataInputStream di) throws IOException, ClassFormatError {
        int magic = di.readInt();
        if (JAVA_FILE_MAGIC != magic) {
            throw new ClassFormatError("Wrong _magic");
        }

        _majorVersion = di.readShort();
        _minorVersion = di.readShort();
    }

    /**
     * Read constant pool.
     *
     * @param di stream
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    protected void readConstantPool(DataInputStream di) throws IOException, ClassFormatError {
        int count = di.readShort();

        _constantPool = new ConstantPool(count);
        for(int i = 0; i < count; ++i) {
            _constantPool.add(EmptyPoolInfo.singleton());
        }

        int index = 1;
        while(index < _constantPool.size()) {
            APoolInfo cpi = APoolInfo.read(di, _constantPool);
            _constantPool.set(index, cpi);
            index += cpi.execute(GetPoolInfoSizeVisitor.singleton(), null);
        }

        // Resolve and check constant pool
        for(APoolInfo cpi : _constantPool) {
            cpi.resolve();
        }
    }

    /**
     * Read class information.
     *
     * @param di stream
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    protected void readClassInfo(DataInputStream di) throws IOException, ClassFormatError {
        _classAccessFlags = di.readShort();

        _thisClass = _constantPool.get(di.readShort()).execute(CheckClassVisitor.singleton(), null);
        short superClassIndex = di.readShort();
        if ("java/lang/Object".equals(_thisClass.getName().toString())) {
            if (0 != superClassIndex) {
                throw new ClassFormatError("java.lang.Object must have 0 as superclass index");
            }
        }
        else {
            _superClass = _constantPool.get(superClassIndex).execute(CheckClassVisitor.singleton(), null);
        }
    }

    /**
     * Read class file interfaces.
     *
     * @param di stream
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    protected void readInterfaces(DataInputStream di) throws IOException, ClassFormatError {
        int count = di.readShort();
        if (0 != count) {
            _interfaces = new ArrayList<ClassPoolInfo>(count);
            for(int i = 0; i < count; ++i) {
                _interfaces.add(null);
            }

            for(int i = 0; i < count; ++i) {
                int iindex = di.readShort();
                if ((1 > iindex) || (iindex > _constantPool.size() - 1)) {
                    throw new ClassFormatError("Interface number out of range, index=" + i);
                }
                _interfaces.set(i, _constantPool.get(iindex).execute(CheckClassVisitor.singleton(), null));
            }
        }
    }

    /**
     * Read class file fields.
     *
     * @param di stream
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    protected void readFields(DataInputStream di) throws IOException, ClassFormatError {
        int count = di.readShort();
        if (0 != count) {
            _fields = new ArrayList<FieldInfo>(count);
            for(int i = 0; i < count; ++i) {
                _fields.add(null);
            }
            for(int i = 0; i < count; ++i) {
                _fields.set(i, new FieldInfo(di, _constantPool));
            }
        }
    }

    /**
     * Read class file methods.
     *
     * @param di stream
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    protected void readMethods(DataInputStream di) throws IOException, ClassFormatError {
        int count = di.readShort();
        if (0 != count) {
            _methods = new ArrayList<MethodInfo>(count);
            for(int i = 0; i < count; ++i) {
                _methods.add(null);
            }
            for(int i = 0; i < count; ++i) {
                _methods.set(i, new MethodInfo(di, _constantPool));
            }
        }
    }

    /**
     * Read class file attributes.
     *
     * @param di stream
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    protected void readAttributes(DataInputStream di) throws IOException, ClassFormatError {
        int count = di.readShort();
        if (0 != count) {
            _attributes = new ArrayList<AAttributeInfo>(count);
            for(int i = 0; i < count; ++i) {
                _attributes.add(null);
            }
            for(int i = 0; i < count; ++i) {
                _attributes.set(i, AAttributeInfo.read(di, _constantPool));
            }
        }
    }

    /**
     * Write class file into stream.
     *
     * @param out stream
     *
     * @throws IOException
     */
    public void write(OutputStream out) throws IOException {
        DataOutputStream dos = new DataOutputStream(out);

        dos.writeInt(JAVA_FILE_MAGIC);
        dos.writeShort(_majorVersion);
        dos.writeShort(_minorVersion);

        dos.writeShort(_constantPool.size());
        for(APoolInfo cpi : _constantPool) {
            if (null != cpi) {
                cpi.write(dos);
            }
        }

        dos.writeShort(_classAccessFlags);
        dos.writeShort(_constantPool.indexOf(_thisClass));
        if ("java/lang/Object".equals(_thisClass.getName().toString())) {
            dos.writeShort(0);
        }
        else {
            dos.writeShort(_constantPool.indexOf(_superClass));
        }

        dos.writeShort(_interfaces.size());
        for(ClassPoolInfo intrf : _interfaces) {
            dos.writeShort(_constantPool.indexOf(intrf));
        }

        dos.writeShort(_fields.size());
        for(FieldInfo field : _fields) {
            field.write(dos, _constantPool);
        }

        dos.writeShort(_methods.size());
        for(MethodInfo method : _methods) {
            method.write(dos, _constantPool);
        }

        dos.writeShort(_attributes.size());
        for(AAttributeInfo attr : _attributes) {
            attr.write(dos);
        }
    }

    /**
     * Return the name of this class.
     *
     * @return name of this class
     */
    public String getThisClassName() {
        return ClassFileTools.getClassName(_thisClass.getName().toString());
    }

    /**
     * Return the name of the super class or the empty string, if this class is java.lang.Object.
     *
     * @return name of super class
     */
    public String getSuperClassName() {
        if (null != _superClass) {
            return ClassFileTools.getClassName(_superClass.getName().toString());
        }
        else {
            return "";
        }
    }

    /**
     * Return a human-readable version of this class.
     *
     * @return string
     */
    public String toString() {
        return "Class File (Version " + _majorVersion + '.' + _minorVersion + ") for class " + _thisClass.getName();
    }


    /**
     * Return a verbose human-readable version of this class.
     *
     * @return verbose string
     */
    public String toStringVerbose() {
        return toStringVerbose(true, true);
    }

    /**
     * Return a verbose human-readable version of this class.
     *
     * @param lineNumbers print line numbers
     * @param PCs print PC values
     * @return verbose string
     */
    public String toStringVerbose(boolean lineNumbers, boolean PCs) {
        StringWriter sw = new StringWriter();
        PrintWriter pw = new PrintWriter(sw);

        // class declaration
        String thisClassName = getThisClassName();
        String superClassName = getSuperClassName();
        String packageName = null;

        if (0 < thisClassName.indexOf('.')) {
            packageName = thisClassName.substring(0, thisClassName.lastIndexOf('.'));
            thisClassName = thisClassName.substring(thisClassName.lastIndexOf('.') + 1);
            pw.println("\npackage " + packageName);
        }

        pw.print(
            ClassFileTools.getAccessString(_classAccessFlags) + "class " + thisClassName + (!"".equals(superClassName) ? (" extends " + superClassName) : ""));
        if (0 != _interfaces.size()) {
            pw.print(" implements ");
            for(ClassPoolInfo intrf : _interfaces) {
                pw.print(ClassFileTools.getClassName(intrf.getName().toString()) + ", ");
            }
            pw.print(ClassFileTools.getClassName(_interfaces.get(_interfaces.size() - 1).getName().toString()));
        }

        // print constant pool
        pw.println("\n\nConstant Pool: " + _constantPool.size() + " items");
        int i = 0;
        for(APoolInfo cpi : _constantPool) {
            pw.println('#' + String.format("%5d", new Object[]{i++}) + ": " + cpi);
        }

//        final Set<String> classesUsed = ClassFileTools.getClassNamesUsed(this);
//        pw.print("\nUsed Classes: " + classesUsed.size() + '\n');
//        for(String s: classesUsed) {
//            pw.println(s);
//        }

        pw.println("\nAttributes: " + _attributes.size());
        i = 0;
        for(final AAttributeInfo attr : _attributes) {
            pw.println("    Attribute " + (i++ + 1) + ": " + attr);
        }

        pw.println("\nFields: " + _fields.size());
        for(FieldInfo field : _fields) {
            pw.println("    " + field.toString(_constantPool));
        }

        pw.println("\nMethods: " + _methods.size());
        for(MethodInfo method : _methods) {
            pw.println("    " + method.toString(_constantPool, lineNumbers, PCs));
        }

        pw.println();

        return sw.toString();
    }

    /**
     * Return a constant pool item from this class. This reindexes the constant pool item to make sure the indices are
     * still correct.
     *
     * @param index index of the item
     *
     * @return pool item, or null if out of range
     */
    public APoolInfo getConstantPoolItem(short index) {
        APoolInfo cp;

        if ((0 >= index) || (index > (_constantPool.size() - 1))) {
            return null;
        }
        cp = _constantPool.get(index);
        cp.reindex();
        return cp;
    }

    /**
     * Add new items to the constant pool. The items must have been resolved already. The list is pruned to avoid adding
     * items to the pool that are already there.
     * <p/>
     * The function first goes through the list and identifies all those value items (int, long, float, double, ASCII,
     * and Unicode) that are already in the pool. If an item is already in the pool, it is not added again. Furthermore,
     * if an item is an ASCII or Unicode UTF item, then all the references to that item present in the list of new items
     * are changed to point to the item in pool. This is only done for ASCII and Unicode UTF items since they are the
     * only ones referenced from other items.
     * <p/>
     * This process is repeated for all reference items (class, field, method, interface method, name-and-type, and
     * string) in the list. Here, only name-and-type and class are referenced from other items, so references to them
     * are changed to point to the pool if they are already in the pool.
     * <p/>
     * Finally, items not yet in the pool are added.
     *
     * @param items array of new items
     *
     * @return array with indices of added items
     */
    public short[] addConstantPoolItems(final APoolInfo[] items) {
        final short[] indices = new short[items.length];

        // process value items
        for(int jv = 0; jv < items.length; ++jv) {
            final int j = jv;
            items[j].execute(new AValueReferencePoolInfoVisitor<Object, Object>() {
                public Object refCase(APoolInfo host, Object o) {
                    // do nothing, not interested in reference objects right now
                    return null;
                }

                public Object emptyCase(EmptyPoolInfo host, Object o) {
                    // do nothing
                    return null;
                }

                public Object valueCase(APoolInfo host, Object o) {
                    // check if this object is already in the pool
                    indices[j] = 0;
                    APoolInfo newArg = host.inPool(_constantPool);
                    if (null != newArg) {
                        // yes, mark for deletion
                        indices[j] = _constantPool.indexOf(newArg);
                    }

                    return null;
                }

                public Object asciizCase(ASCIIPoolInfo host, Object o) {
                    return utfCase(host, o);
                }

                public Object unicodeCase(UnicodePoolInfo host, Object o) {
                    return utfCase(host, o);
                }

                private Object utfCase(AUTFPoolInfo utfHost, Object o) {
                    // check if this object is already in the pool
                    indices[j] = 0;
                    final AUTFPoolInfo newArg = (AUTFPoolInfo)utfHost.inPool(_constantPool);
                    if (null != newArg) {
                        // yes, mark for deletion
                        indices[j] = _constantPool.indexOf(newArg);

                        // update references
                        for(APoolInfo cpi : items) {
                            cpi.execute(new AValueReferencePoolInfoVisitor<Object, Object>() {
                                public Object refCase(APoolInfo host, Object o) {
                                    // do nothing, these objects don't have UTF references
                                    return null;
                                }

                                public Object emptyCase(EmptyPoolInfo host, Object o) {
                                    // do nothing
                                    return null;
                                }

                                public Object valueCase(APoolInfo host, Object o) {
                                    // do nothing, value objects don't have references
                                    return null;
                                }

                                public Object classCase(ClassPoolInfo host, Object o) {
                                    // class objects have a UTF reference
                                    if (host.getName() == newArg) {
                                        // this is affected, update it
                                        host.setName(newArg);
                                    }
                                    return null;
                                }

                                public Object stringCase(StringPoolInfo host, Object o) {
                                    // string objects have a UTF reference
                                    if (host.getUtf() == newArg) {
                                        // this is affected, update it
                                        host.setUtf(newArg);
                                    }
                                    return null;
                                }

                                public Object nameAndTypeCase(NameAndTypePoolInfo host, Object o) {
                                    // name and type objects have two UTF references
                                    if (host.getName() == newArg) {
                                        // this is affected, update it
                                        host.setName(newArg);
                                    }
                                    if (host.getDescriptor() == newArg) {
                                        // this is affected, update it
                                        host.setDescriptor(newArg);
                                    }
                                    return null;
                                }
                            }, null);
                        }
                    }

                    return null;
                }
            }, null);
        }

        // process reference items
        for(int jv = 0; jv < items.length; ++jv) {
            final int j = jv;
            items[j].execute(new AValueReferencePoolInfoVisitor<Object, Object>() {
                public Object valueCase(APoolInfo host, Object o) {
                    // do nothing, not interested in reference objects right now
                    return null;
                }

                public Object emptyCase(EmptyPoolInfo host, Object o) {
                    // do nothing
                    return null;
                }

                public Object refCase(APoolInfo host, Object o) {
                    // check if this object is already in the pool
                    indices[j] = 0;
                    APoolInfo newArg = host.inPool(_constantPool);
                    if (null != newArg) {
                        // yes, mark for deletion
                        indices[j] = _constantPool.indexOf(newArg);
                    }

                    return null;
                }

                public Object classCase(ClassPoolInfo host, Object o) {
                    return referencedCase(host, o);
                }

                public Object nameAndTypeInfo(NameAndTypePoolInfo host, Object o) {
                    return referencedCase(host, o);
                }

                private Object referencedCase(APoolInfo referencedHost, Object o) {
                    // check if this object is already in the pool
                    indices[j] = 0;
                    final APoolInfo newArg = referencedHost.inPool(_constantPool);
                    if (null != newArg) {
                        // yes, mark for deletion
                        indices[j] = _constantPool.indexOf(newArg);

                        // update references
                        for(APoolInfo cpi : items) {
                            cpi.execute(new AValueReferencePoolInfoVisitor<Object, Object>() {
                                public Object valueCase(APoolInfo host, Object o) {
                                    // do nothing, value objects don't have references
                                    return null;
                                }

                                public Object emptyCase(EmptyPoolInfo host, Object o) {
                                    // do nothing
                                    return null;
                                }

                                public Object refCase(APoolInfo host, Object o) {
                                    // do nothing, these objects don't have Class or NameAndType references
                                    return null;
                                }

                                public Object fieldCase(FieldPoolInfo host, Object o) {
                                    // field objects have Class and NameAndType references
                                    return classNameTypeCase(host, o);
                                }

                                public Object methodCase(MethodPoolInfo host, Object o) {
                                    // method objects have Class and NameAndType references
                                    return classNameTypeCase(host, o);
                                }

                                public Object interfaceMethodCase(InterfaceMethodPoolInfo host, Object o) {
                                    // interface method objects have Class and NameAndType references
                                    return classNameTypeCase(host, o);
                                }

                                public Object classNameTypeCase(AClassNameTypePoolInfo host, Object o) {
                                    // ClassNameType objects have a Class and a NameAndType references
                                    if (host.getClassInfo() == newArg) {
                                        // this is affected, update it
                                        host.setClassInfo((ClassPoolInfo)newArg);
                                    }
                                    if (host.getNameAndType() == newArg) {
                                        // this is affected, update it
                                        host.setNameAndType((NameAndTypePoolInfo)newArg);
                                    }
                                    return null;
                                }
                            }, null);
                        }
                    }

                    return null;
                }
            }, null);
        }

        // add new items to the pool
        for(int i = 0; i < items.length; ++i) {
            if (0 == indices[i]) {
                _constantPool.add(items[i]);
                // add empty item for long and double
                items[i].execute(new NoOpPoolInfoVisitor<Object, Object>() {
                    public Object longCase(LongPoolInfo host, Object o) {
                        _constantPool.add(EmptyPoolInfo.singleton());
                        return null;
                    }

                    public Object doubleCase(DoublePoolInfo host, Object o) {
                        _constantPool.add(EmptyPoolInfo.singleton());
                        return null;
                    }
                }, null);
                indices[i] = _constantPool.indexOf(items[i]);
            }
        }

        return indices;
    }

    /**
     * Add an attribute to the class.
     *
     * @param newAttribute new attribute
     */
    public void addAttribute(AAttributeInfo newAttribute) {
        _attributes.add(newAttribute);
    }

    /**
     * Return the attribute with the specified name.
     *
     * @param name attribute name
     *
     * @return attribute or null if not found
     */
    public AAttributeInfo getAttribute(String name) {
        if (0 == _attributes.size()) {
            return null;
        }
        for(AAttributeInfo attr : _attributes) {
            if (0 == name.compareTo(attr.getName().toString())) {
                return attr;
            }
        }
        return null;
    }

    /**
     * Accessor for the minor version.
     *
     * @return minor version
     */
    public short getMinorVersion() {
        return _minorVersion;
    }

    /**
     * Mutator for the minor version.
     *
     * @param minorVersion new minor version
     */
    public void setMinorVersion(short minorVersion) {
        _minorVersion = minorVersion;
    }

    /**
     * Accessor for the major version.
     *
     * @return major version
     */
    public short getMajorVersion() {
        return _majorVersion;
    }

    /**
     * Mutator for the major version.
     *
     * @param majorVersion new major version
     */
    public void setMajorVersion(short majorVersion) {
        _majorVersion = majorVersion;
    }

    /**
     * Accessor for the class access flags.
     *
     * @return access flags
     */
    public short getClassAccessFlags() {
        return _classAccessFlags;
    }

    /**
     * Mutator for the class access flags.
     *
     * @param classAccessFlags new flags
     */
    public void setClassAccessFlags(short classAccessFlags) {
        _classAccessFlags = classAccessFlags;
    }

    /**
     * Accessor for the superclass.
     *
     * @return superclass information
     */
    public ClassPoolInfo getSuperClass() {
        return _superClass;
    }

    /**
     * Mutator for the superclass.
     * @param cpi superclass information
     */
    public void setSuperClass(ClassPoolInfo cpi) {
        _superClass = cpi;
    }

    /**
     * Accessor for this class.
     *
     * @return this class
     */
    public ClassPoolInfo getThisClass() {
        return _thisClass;
    }
    /**
     * Mutator for the this class.
     * @param cpi this class information
     */
    public void setThisClass(ClassPoolInfo cpi) {
        _thisClass = cpi;
    }


    /**
     * Accessor for the interface list.
     *
     * @return interface list
     */
    public ArrayList<ClassPoolInfo> getInterfaces() {
        return _interfaces;
    }

    /**
     * Accessor for the fields list.
     *
     * @return fields list
     */
    public ArrayList<FieldInfo> getFields() {
        return _fields;
    }

    /**
     * Accessor for the methods list.
     *
     * @return methods list
     */
    public ArrayList<MethodInfo> getMethods() {
        return _methods;
    }

    /**
     * Accessor for the attributes list.
     *
     * @return methods list
     */
    public ArrayList<AAttributeInfo> getAttributes() {
        return _attributes;
    }

    /**
     * Accessor for the constant pool.
     *
     * @return methods list
     */
    public ConstantPool getConstantPool() {
        return _constantPool;
    }

    /**
     * Add the constant pool items for a method "methodName:methodDescriptor" in class "className".
     *
     * @param className        class name
     * @param methodName       method name
     * @param methodDescriptor method descriptor
     *
     * @return constant pool index of method info
     */
    public short addMethodToConstantPool(String className, String methodName, String methodDescriptor) {
        ConstantPool cp = getConstantPool();

        short[] l;

        // add a new name for the monitor class
        AUTFPoolInfo monitorClassName = new ASCIIPoolInfo(className, cp);
        l = addConstantPoolItems(new APoolInfo[]{monitorClassName});
        // Debug.out.println("monitorClassName index = " + l[0]);
        monitorClassName = getConstantPoolItem(l[0]).execute(CheckUTFVisitor.singleton(), null);

        // add a new class for the monitor class
        ClassPoolInfo monitorClass = new ClassPoolInfo(monitorClassName, cp);
        l = addConstantPoolItems(new APoolInfo[]{monitorClass});
        // Debug.out.println("monitorClass index = " + l[0]);
        monitorClass = getConstantPoolItem(l[0]).execute(CheckClassVisitor.singleton(), null);

        // add a new method name for the method
        AUTFPoolInfo methodNameInfo = new ASCIIPoolInfo(methodName, cp);
        l = addConstantPoolItems(new APoolInfo[]{methodNameInfo});
        // Debug.out.println("methodNameInfo index = " + l[0]);
        methodNameInfo = getConstantPoolItem(l[0]).execute(CheckUTFVisitor.singleton(), null);

        // add a new type name for the method
        AUTFPoolInfo methodTypeName = new ASCIIPoolInfo(methodDescriptor, cp);
        l = addConstantPoolItems(new APoolInfo[]{methodTypeName});
        // Debug.out.println("methodTypeName index = " + l[0]);
        methodTypeName = getConstantPoolItem(l[0]).execute(CheckUTFVisitor.singleton(), null);

        // add a new name-and-type for the method
        NameAndTypePoolInfo methodNaT = new NameAndTypePoolInfo(methodNameInfo, methodTypeName, cp);
        l = addConstantPoolItems(new APoolInfo[]{methodNaT});
        // Debug.out.println("methodNaT index = " + l[0]);
        methodNaT = getConstantPoolItem(l[0]).execute(CheckNameAndTypeVisitor.singleton(), null);

        // add a new method info for the method
        MethodPoolInfo method = new MethodPoolInfo(monitorClass, methodNaT, cp);
        l = addConstantPoolItems(new APoolInfo[]{method});
        // Debug.out.println("method index = " + l[0]);
        method = getConstantPoolItem(l[0]).execute(CheckMethodVisitor.singleton(), null);

        return l[0];
    }

    /**
     * Add the constant pool items for a field "fieldName:fieldDescriptor" in class "className".
     *
     * @param className        class name
     * @param fieldName        field name
     * @param fieldDescriptor  field descriptor
     * @param addToFields      true if a new field should be added to the fields list as well
     * @param accessFlags      access flags (only if addToFields is true)
     *
     * @return constant pool index of field info
     */
    public short addField(String className, String fieldName, String fieldDescriptor, boolean addToFields,
                          short accessFlags) {
        ConstantPool cp = getConstantPool();

        short[] l;

        // add a new name
        AUTFPoolInfo fieldClassName = new ASCIIPoolInfo(className, cp);
        l = addConstantPoolItems(new APoolInfo[]{fieldClassName});
        fieldClassName = getConstantPoolItem(l[0]).execute(CheckUTFVisitor.singleton(), null);

        // add a new class
        ClassPoolInfo fieldClass = new ClassPoolInfo(fieldClassName, cp);
        l = addConstantPoolItems(new APoolInfo[]{fieldClass});
        fieldClass = getConstantPoolItem(l[0]).execute(CheckClassVisitor.singleton(), null);

        // add a new field name for the field
        AUTFPoolInfo fieldNameInfo = new ASCIIPoolInfo(fieldName, cp);
        l = addConstantPoolItems(new APoolInfo[]{fieldNameInfo});
        fieldNameInfo = getConstantPoolItem(l[0]).execute(CheckUTFVisitor.singleton(), null);

        // add a new type name for the field
        AUTFPoolInfo fieldTypeName = new ASCIIPoolInfo(fieldDescriptor, cp);
        l = addConstantPoolItems(new APoolInfo[]{fieldTypeName});
        fieldTypeName = getConstantPoolItem(l[0]).execute(CheckUTFVisitor.singleton(), null);

        // add a new name-and-type for the field
        NameAndTypePoolInfo fieldNaT = new NameAndTypePoolInfo(fieldNameInfo, fieldTypeName, cp);
        l = addConstantPoolItems(new APoolInfo[]{fieldNaT});
        fieldNaT = getConstantPoolItem(l[0]).execute(CheckNameAndTypeVisitor.singleton(), null);

        // add a new field info for the field
        FieldPoolInfo field = new FieldPoolInfo(fieldClass, fieldNaT, cp);
        l = addConstantPoolItems(new APoolInfo[]{field});
        field = getConstantPoolItem(l[0]).execute(new ADefaultPoolInfoVisitor<FieldPoolInfo, Object>() {
                    public FieldPoolInfo defaultCase(APoolInfo host, Object param) {
                        throw new ClassFormatError("Info is of type " + host.getClass().getName() + ", needs to be FieldPoolInfo");
                    }

                    public FieldPoolInfo fieldCase(FieldPoolInfo host, Object param) {
                        return host;
                    }
                }, null);

        if (addToFields) {
            FieldInfo fi = new FieldInfo(accessFlags, fieldNameInfo, fieldTypeName, new SourceFileAttributeInfo[] {});
            getFields().add(fi);
        }

        return l[0];
    }

    /**
     * Find the method info in the constant pool for a method "methodName:methodDescriptor" in class "className".
     *
     * @param className        class name
     * @param methodName       method name
     * @param methodDescriptor method descriptor
     *
     * @return constant pool index of the method, or 0 if not found
     */
    public short findMethodInConstantPool(String className, String methodName, String methodDescriptor) {
        ConstantPool cp = getConstantPool();

        short[] l;
        APoolInfo newArg;

        // add a new name for the monitor class
        AUTFPoolInfo monitorClassName = new ASCIIPoolInfo(className, cp);
        newArg = (AUTFPoolInfo)monitorClassName.inPool(cp);
        if (null == newArg) {
            // not found
            return 0;
        }
        monitorClassName = newArg.execute(CheckUTFVisitor.singleton(), null);

        // add a new class for the monitor class
        ClassPoolInfo monitorClass = new ClassPoolInfo(monitorClassName, cp);
        newArg = monitorClass.inPool(cp);
        if (null == newArg) {
            // not found
            return 0;
        }
        monitorClass = newArg.execute(CheckClassVisitor.singleton(), null);

        // add a new method name for the method
        AUTFPoolInfo methodNameInfo = new ASCIIPoolInfo(methodName, cp);
        newArg = methodNameInfo.inPool(cp);
        if (null == newArg) {
            // not found
            return 0;
        }
        methodNameInfo = newArg.execute(CheckUTFVisitor.singleton(), null);

        // add a new type name for the method
        AUTFPoolInfo methodTypeName = new ASCIIPoolInfo(methodDescriptor, cp);
        newArg = methodTypeName.inPool(cp);
        if (null == newArg) {
            // not found
            return 0;
        }
        methodTypeName = newArg.execute(CheckUTFVisitor.singleton(), null);

        // add a new name-and-type for the method
        NameAndTypePoolInfo methodNaT = new NameAndTypePoolInfo(methodNameInfo, methodTypeName, cp);
        newArg = methodNaT.inPool(cp);
        if (null == newArg) {
            // not found
            return 0;
        }
        methodNaT = newArg.execute(CheckNameAndTypeVisitor.singleton(), null);

        // add a new method info for the method
        MethodPoolInfo method = new MethodPoolInfo(monitorClass, methodNaT, cp);
        newArg = method.inPool(cp);
        if (null == newArg) {
            // not found
            return 0;
        }
        method = newArg.execute(CheckMethodVisitor.singleton(), null);

        return cp.indexOf(method);
    }
}
