package sysModel.classFile;

/**
 * Tools for dealing with class files.
 *
 * @author Mathias Ricken
 */
public class ClassFileTools {


    /**
     * Return a string that represents the access flags set.
     *
     * @param flags access flags
     *
     * @return string with access flags as words
     */
    public static String getAccessString(short flags) {
        StringBuffer x = new StringBuffer();

        if (0 != (flags & ClassFile.ACC_PUBLIC)) {
            x.append("public ");
        }

        if (0 != (flags & ClassFile.ACC_PRIVATE)) {
            x.append("private ");
        }

        if (0 != (flags & ClassFile.ACC_PROTECTED)) {
            x.append("protected ");
        }

        if (0 != (flags & ClassFile.ACC_STATIC)) {
            x.append("static ");
        }

        if (0 != (flags & ClassFile.ACC_FINAL)) {
            x.append("final ");
        }

        if (0 != (flags & ClassFile.ACC_SYNCHRONIZED)) {
            x.append("synchronized ");
        }

        if (0 != (flags & ClassFile.ACC_VOLATILE)) {
            x.append("volatile ");
        }

        if (0 != (flags & ClassFile.ACC_TRANSIENT)) {
            x.append("transient ");
        }

        if (0 != (flags & ClassFile.ACC_NATIVE)) {
            x.append("native ");
        }

        if (0 != (flags & ClassFile.ACC_INTERFACE)) {
            x.append("interface ");
        }

        if (0 != (flags & ClassFile.ACC_ABSTRACT)) {
            x.append("abstract ");
        }

        if (0 != (flags & ClassFile.ACC_STRICT)) {
            x.append("strictfp ");
        }

        // TODO: Handle ACC_SYNTHETIC, ACC_ANNOTATION, ACC_ENUM, ACC_BRIDGE, and ACC_VARARGS here?

        return x.toString();
    }

    /**
     * Translate a type descriptor descriptor and a variable name into a Java declaration.
     *
     * @param typeString type descriptor descriptor
     * @param varName    variable name
     *
     * @return declaration
     */
    public static String getTypeString(String typeString, String varName) {
        int isArray = 0;
        int ndx = 0;
        StringBuffer x = new StringBuffer();

        while('[' == typeString.charAt(ndx)) {
            ++isArray;
            ++ndx;
        }

        switch(typeString.charAt(ndx)) {
            case 'B':
                x.append("byte");
                break;
            case 'C':
                x.append("char");
                break;
            case 'D':
                x.append("double");
                break;
            case 'F':
                x.append("float");
                break;
            case 'I':
                x.append("int");
                break;
            case 'J':
                x.append("long");
                break;
            case 'L':
                for(int i = ndx + 1; i < typeString.indexOf((int)';'); ++i) {
                    if ('/' != typeString.charAt(i)) {
                        x.append(typeString.charAt(i));
                    }
                    else {
                        x.append('.');
                    }
                }
                break;
            case 'V':
                x.append("void");
                break;
            case 'S':
                x.append("short");
                break;
            case 'Z':
                x.append("boolean");
                break;
        }
        while(0 < isArray) {
            x.append("[]");
            isArray--;
        }
        x.append(' ');
        x.append(varName);
        return x.toString();
    }

    /**
     * Return true if this is a primitive type or an array of primitive types.
     *
     * @param typeString type descriptor descriptor
     *
     * @return true if primitive type or array of primitive types
     */
    public static boolean isPrimitive(String typeString) {
        int ndx = 0;

        while('[' == typeString.charAt(ndx)) {
            ++ndx;
        }

        switch(typeString.charAt(ndx)) {
            case 'B':
            case 'C':
            case 'D':
            case 'F':
            case 'I':
            case 'J':
            case 'V':
            case 'S':
            case 'Z':
                return true;
            default:
                return false;
        }
    }

    /**
     * Return the next descriptor from a string of concatenated signatures. For example, if the descriptor was "[BII",
     * this method would return "II".
     *
     * @param sig concatenated signatures
     *
     * @return next descriptor
     */
    public static String getNextSignature(String sig) {
        int ndx = 0;
        String x;

        while('[' == sig.charAt(ndx)) {
            ++ndx;
        }

        if ('L' == sig.charAt(ndx)) {
            while(';' != sig.charAt(ndx)) {
                ++ndx;
            }
        }
        ++ndx;
        x = (sig.substring(ndx));
        return (x);
    }

    /**
     * Return class name in Java form.
     *
     * @param s mangled class name
     *
     * @return Java class name
     */
    public static String getClassName(String s) {
        if ('[' == s.charAt(0)) {
            return getTypeString(s, "");
        }

        return s.replace('/','.');
    }
}
