package sysModel.classFile;

import sysModel.classFile.attributes.AAttributeInfo;
import sysModel.classFile.attributes.SourceFileAttributeInfo;
import sysModel.classFile.constantPool.AUTFPoolInfo;
import sysModel.classFile.constantPool.ConstantPool;
import sysModel.classFile.constantPool.visitors.CheckUTFVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;

/**
 * Represents a field in a class file.
 *
 * @author Mathias Ricken
 */

public final class FieldInfo {
    /**
     * Field access flags.
     */
    private short _accessFlags;

    /**
     * Name information.
     */
    private AUTFPoolInfo _name;

    /**
     * Type descriptor information.
     */
    private AUTFPoolInfo _descriptor;

    /**
     * Attributes.
     */
    private ArrayList<AAttributeInfo> _attributes = new ArrayList<AAttributeInfo>();

    /**
     * Constructor.
     *
     * @param accessFlags field access flags
     * @param name        field name
     * @param descriptor  field descriptor
     * @param attributes  array of attributes
     */
    public FieldInfo(short accessFlags,
                     AUTFPoolInfo name,
                     AUTFPoolInfo descriptor,
                     SourceFileAttributeInfo[] attributes) {
        _accessFlags = accessFlags;
        _name = name;
        _descriptor = descriptor;
        if (null != attributes) {
            for(AAttributeInfo attr : attributes) {
                _attributes.add(attr);
            }
        }
    }

    /**
     * Constructor.
     *
     * @param di   input stream
     * @param pool constant pool
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    public FieldInfo(DataInputStream di, ConstantPool pool) throws IOException, ClassFormatError {
        _accessFlags = di.readShort();
        _name = pool.get(di.readShort()).execute(CheckUTFVisitor.singleton(), null);
        _descriptor = pool.get(di.readShort()).execute(CheckUTFVisitor.singleton(), null);

        int count = di.readShort();
        for(int i = 0; i < count; i++) {
            _attributes.add(AAttributeInfo.read(di, pool));
        }
    }

    /**
     * Write this field into a stream.
     *
     * @param dos  output stream
     * @param pool constant pool
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos, ConstantPool pool) throws IOException {
        dos.writeShort(_accessFlags);
        dos.writeShort(pool.indexOf(_name));
        dos.writeShort(pool.indexOf(_descriptor));
        dos.writeShort(_attributes.size());
        for(AAttributeInfo attr : _attributes) {
            attr.write(dos);
        }
    }

    /**
     * Return a human-readable version of this field.
     *
     * @return string
     */
    public String toString() {
        StringBuffer x = new StringBuffer();

        x.append(ClassFileTools.getAccessString(_accessFlags));
        x.append(ClassFileTools.getTypeString(_descriptor.toStringVerbose(), _name.toStringVerbose()));
        if (0 != _attributes.size()) {
            x.append(" = " + _attributes.get(0).toString());
        }
        return x.toString();
    }

    /**
     * Return a human-readable version of this field and correctly handle booleans and strings.
     *
     * @param pool constant pool
     *
     * @return string
     */
    public final String toString(ConstantPool pool) {
        StringBuffer x = new StringBuffer();

        x.append(ClassFileTools.getAccessString(_accessFlags));
        String t = ClassFileTools.getTypeString(_descriptor.toString(), _name.toString());
        x.append(t);
        for(AAttributeInfo attr : _attributes) {
            x.append(", " + attr);
        }
        return x.toString();
    }

    /**
     * Accessor for access flags.
     *
     * @return access flags
     */
    public short getAccessFlags() {
        return _accessFlags;
    }

    /**
     * Mutator for access flags.
     *
     * @param accessFlags new access flags
     */
    public void setAccessFlags(short accessFlags) {
        _accessFlags = accessFlags;
    }

    /**
     * Accessor for field name.
     *
     * @return field name
     */
    public AUTFPoolInfo getName() {
        return _name;
    }

    /**
     * Mutator for field name.
     *
     * @param name new field name
     */
    public void setName(AUTFPoolInfo name) {
        _name = name;
    }

    /**
     * Accessor for descriptor.
     *
     * @return descriptor
     */
    public AUTFPoolInfo getDescriptor() {
        return _descriptor;
    }

    /**
     * Mutator for descriptor.
     *
     * @param descriptor new descriptor
     */
    public void setDescriptor(AUTFPoolInfo descriptor) {
        _descriptor = descriptor;
    }

    /**
     * Accessor for attributes.
     *
     * @return attributes
     */
    public ArrayList<AAttributeInfo> getAttributes() {
        return _attributes;
    }

    /**
     * Mutator for attributes.
     *
     * @param attributes new attributes
     */
    public void setAttributes(ArrayList<AAttributeInfo> attributes) {
        _attributes = attributes;
    }
}
