package sysModel.classFile;

import sysModel.classFile.attributes.AAttributeInfo;
import sysModel.classFile.attributes.CodeAttributeInfo;
import sysModel.classFile.attributes.visitors.ADefaultAttributeVisitor;
import sysModel.classFile.code.InstructionList;
import sysModel.classFile.constantPool.AUTFPoolInfo;
import sysModel.classFile.constantPool.ConstantPool;
import sysModel.classFile.constantPool.visitors.CheckUTFVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.util.ArrayList;

/**
 * Represents a method in a class file.
 *
 * @author Mathias Ricken
 */

public final class MethodInfo {
    /**
     * Method access flags.
     */
    private short _accessFlags;

    /**
     * Name information.
     */
    private AUTFPoolInfo _name;

    /**
     * Type descriptor information.
     */
    private AUTFPoolInfo _descriptor;

    /**
     * Attributes.
     */
    private ArrayList<AAttributeInfo> _attributes = new ArrayList<AAttributeInfo>();

    /**
     * Constructor.
     *
     * @param accessFlags method access flags
     * @param name        method name
     * @param signature   method descriptor
     * @param attributes  array of attributes
     */
    public MethodInfo(short accessFlags,
                      AUTFPoolInfo name,
                      AUTFPoolInfo signature,
                      AAttributeInfo[] attributes) {
        _accessFlags = accessFlags;
        _name = name;
        _descriptor = signature;
        if (null != attributes) {
            for(AAttributeInfo attr : attributes) {
                _attributes.add(attr);
            }
        }
    }

    /**
     * Constructor.
     *
     * @param di   input stream
     * @param pool constant pool
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    public MethodInfo(DataInputStream di, ConstantPool pool) throws IOException, ClassFormatError {
        _accessFlags = di.readShort();
        _name = pool.get(di.readShort()).execute(CheckUTFVisitor.singleton(), null);
        _descriptor = pool.get(di.readShort()).execute(CheckUTFVisitor.singleton(), null);

        int count = di.readShort();
        for(int i = 0; i < count; i++) {
            _attributes.add(AAttributeInfo.read(di, pool));
        }
    }

    /**
     * Write this method into the stream.
     *
     * @param dos  output stream
     * @param pool constant pool
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos, ConstantPool pool) throws IOException {
        dos.writeShort(_accessFlags);
        dos.writeShort(pool.indexOf(_name));
        dos.writeShort(pool.indexOf(_descriptor));
        dos.writeShort(_attributes.size());
        for(AAttributeInfo attr : _attributes) {
            attr.write(dos);
        }
    }

    /**
     * Return a human-readable version of this method.
     *
     * @return string
     */
    public String toString() {
        String s = _descriptor.toString();
        String paramSig = s.substring(s.indexOf('(') + 1, s.indexOf(')'));
        String returnSig = s.substring(s.indexOf(')') + 1);

        // handle constructor correctly
        StringBuffer parameterList = new StringBuffer();
        parameterList.append(_name + "(");

        char initialParameter = 'a';
        if ((0 < paramSig.length()) && 'V' != paramSig.charAt(0)) {
            StringBuffer varName = new StringBuffer();
            while(0 < paramSig.length()) {
                varName.setLength(0);
                varName.append(initialParameter);
                initialParameter++;
                parameterList.append(ClassFileTools.getTypeString(paramSig, varName.toString()));
                paramSig = ClassFileTools.getNextSignature(paramSig);
                if (0 < paramSig.length()) {
                    parameterList.append(", ");
                }
            }

        }
        parameterList.append(')');

        StringBuffer x = new StringBuffer();
        x.append(ClassFileTools.getAccessString(_accessFlags));
        x.append(ClassFileTools.getTypeString(returnSig, parameterList.toString()));

        return x.toString();
    }

    /**
     * Return a human-readable version of this method.
     *
     * @param pool constant pool
     *
     * @return string
     */
    public String toString(ConstantPool pool) {
        return toString(pool, true, true);
    }

    /**
     * Return a human-readable version of this method.
     *
     * @param pool constant pool
     * @param lineNumbers print line numbers
     * @param PCs print PC values
     *
     * @return string
     */
    public String toString(ConstantPool pool, boolean lineNumbers, boolean PCs) {
        String s = _descriptor.toString();
        String paramSig = s.substring(s.indexOf('(') + 1, s.indexOf(')'));
        String returnSig = s.substring(s.indexOf(')') + 1);

        // handle constructor correctly
        StringBuffer parameterList = new StringBuffer();
        parameterList.append(_name + "(");

        char initialParameter = 'a';
        if ((0 < paramSig.length()) && 'V' != paramSig.charAt(0)) {
            StringBuffer varName = new StringBuffer();
            while(0 < paramSig.length()) {
                varName.setLength(0);
                varName.append(initialParameter);
                initialParameter++;
                parameterList.append(ClassFileTools.getTypeString(paramSig, varName.toString()));
                paramSig = ClassFileTools.getNextSignature(paramSig);
                if (0 < paramSig.length()) {
                    parameterList.append(", ");
                }
            }

        }
        parameterList.append(')');

        StringBuffer x = new StringBuffer();
        x.append(ClassFileTools.getAccessString(_accessFlags));
        x.append(ClassFileTools.getTypeString(returnSig, parameterList.toString()));

        for(AAttributeInfo attr : _attributes) {
            x.append("\n\t").append(attr.toString());
        }

        if (0 == (_accessFlags & (ClassFile.ACC_NATIVE | ClassFile.ACC_ABSTRACT))) {
            CodeAttributeInfo ca = getCodeAttributeInfo();
            x.append('\n').append((new InstructionList(ca.getCode())).toString(lineNumbers, PCs));
        }

        return x.toString();
    }

    /**
     * Accessor for access flags.
     *
     * @return access flags
     */
    public short getAccessFlags() {
        return _accessFlags;
    }

    /**
     * Mutator for access flags.
     *
     * @param accessFlags new access flags
     */
    public void setAccessFlags(short accessFlags) {
        _accessFlags = accessFlags;
    }

    /**
     * Accessor for field name.
     *
     * @return field name
     */
    public AUTFPoolInfo getName() {
        return _name;
    }

    /**
     * Mutator for field name.
     *
     * @param name new field name
     */
    public void setName(AUTFPoolInfo name) {
        _name = name;
    }

    /**
     * Accessor for descriptor.
     *
     * @return descriptor
     */
    public AUTFPoolInfo getDescriptor() {
        return _descriptor;
    }

    /**
     * Mutator for descriptor.
     *
     * @param descriptor new descriptor
     */
    public void setDescriptor(AUTFPoolInfo descriptor) {
        _descriptor = descriptor;
    }

    /**
     * Accessor for attributes.
     *
     * @return attributes
     */
    public ArrayList<AAttributeInfo> getAttributes() {
        return _attributes;
    }

    /**
     * Return this method's code attribute info.
     *
     * @return code attribute info
     */
    public CodeAttributeInfo getCodeAttributeInfo() {
        ADefaultAttributeVisitor<CodeAttributeInfo, Object> getCodeAttributeVisitor
            = new ADefaultAttributeVisitor<CodeAttributeInfo, Object>() {
            public CodeAttributeInfo defaultCase(AAttributeInfo host, Object o) {
                return null;
            }

            public CodeAttributeInfo codeCase(CodeAttributeInfo host, Object o) {
                return host;
            }
        };
        for(AAttributeInfo attr : _attributes) {
            CodeAttributeInfo code = attr.execute(getCodeAttributeVisitor, null);
            if (null != code) {
                return code;
            }

        }
        throw new ClassFormatError("Method has no Code attribute");
    }
}
