package sysModel.classFile;

/**
 * Type conversions.
 *
 * @author Mathias Ricken
 */
public class Types {
    /**
     * Convert a signed int into an unsigned integer.
     *
     * @param i signed int
     *
     * @return unsigned integer
     */
    public static final long unsigned(int i) {
        return i & 0xffffffff;
    }

    /**
     * Convert a signed short into an unsigned short.
     *
     * @param s signed short
     *
     * @return unsigned short
     */
    public static final int unsigned(short s) {
        return s & 0xffff;
    }

    /**
     * Convert a signed byte into an unsigned byte.
     *
     * @param b signed byte
     *
     * @return unsigned byte
     */
    public static final int unsigned(byte b) {
        return b & 0xff;
    }

    /**
     * Construct a short from two bytes in a byte array.
     *
     * @param b     byte array
     * @param start index in byte array
     *
     * @return unsigned short
     */
    public static final short ushortFromBytes(byte[] b, int start) {
        return (short)(((b[start] & 0xff) << 8) | ((b[start + 1] & 0xff) << 0));
    }

    /**
     * Construct a short from two bytes in a byte array.
     *
     * @param b     byte array
     * @param start index in byte array
     *
     * @return short
     */
    public static final short shortFromBytes(byte[] b, int start) {
        return (short)((b[start] << 8) | ((b[start + 1] & 0xff) << 0));
    }

    /**
     * Construct an array of two individual bytes from a short.
     *
     * @param s short
     *
     * @return array of two bytes
     */
    public static final byte[] bytesFromShort(short s) {
        return new byte[]{(byte)(s >> 8), (byte)(s & 0xff)};
    }

    /**
     * Construct two individual bytes from a short and place them in the byte array at offset start.
     *
     * @param s     short
     * @param b     byte array
     * @param start offset
     */
    public static final void bytesFromShort(short s, byte[] b, int start) {
        b[start] = (byte)(s >> 8);
        b[start + 1] = (byte)(s & 0xff);
    }

    /**
     * Construct an int from four bytes in a byte array.
     *
     * @param b     byte array
     * @param start index in byte array
     *
     * @return unsigned int
     */
    public static final int uintFromBytes(byte[] b, int start) {
        return (int)(((b[start] & 0xff) << 24) | ((b[start + 1] & 0xff) << 16) | ((b[start + 2] & 0xff) << 8) | ((b[start + 3] & 0xff) << 0));
    }

    /**
     * Construct an int from four bytes in a byte array.
     *
     * @param b     byte array
     * @param start index in byte array
     *
     * @return signed int
     */
    public static final int intFromBytes(byte[] b, int start) {
        return (int)((b[start] << 24) | ((b[start + 1] & 0xff) << 16) | ((b[start + 2] & 0xff) << 8) | ((b[start + 3] & 0xff) << 0));
    }

    /**
     * Construct an array of four individual bytes from an int.
     *
     * @param i int
     *
     * @return array of four bytes
     */
    public static final byte[] bytesFromInt(int i) {
        return new byte[]{(byte)(i >> 24), (byte)((i >> 16) & 0xff), (byte)((i >> 8) & 0xff), (byte)(i & 0xff)};
    }

    /**
     * Construct four individual bytes from an int and place them in the byte array starting at offset start.
     *
     * @param i     int
     * @param b     byte array
     * @param start offset
     */
    public static final void bytesFromInt(int i, byte[] b, int start) {
        b[start] = (byte)(i >> 24);
        b[start + 1] = (byte)((i >> 16) & 0xff);
        b[start + 2] = (byte)((i >> 8) & 0xff);
        b[start + 3] = (byte)(i & 0xff);
    }
}
