package sysModel.classFile.attributes;

import sysModel.classFile.attributes.visitors.IAttributeVisitor;
import sysModel.classFile.code.instructions.LineNumberTable;
import sysModel.classFile.constantPool.AUTFPoolInfo;
import sysModel.classFile.constantPool.ConstantPool;
import sysModel.classFile.constantPool.visitors.CheckUTFVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;
import java.lang.reflect.Method;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Constructor;

/**
 * Abstract class file attribute.
 * @author Mathias Ricken
 */
public abstract class AAttributeInfo implements Cloneable {
    /**
     * Attribute _name information.
     */
    protected AUTFPoolInfo _name;

    /**
     * Attribute _data.
     */
    protected byte[] _data;

    /**
     * Constant pool.
     */
    protected ConstantPool _constantPool;

    /**
     * Array with registered attributes.
     */
    private static Class[] _knownAttributes = new Class[] {
                                                                           SourceFileAttributeInfo.class,
                                                                           CodeAttributeInfo.class,
                                                                    };

    /**
     * Read from stream and return unresolved constant pool object.
     *
     * @param di   stream
     * @param pool constant pool
     *
     * @return attribute
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    public static AAttributeInfo read(DataInputStream di, ConstantPool pool) throws IOException, ClassFormatError {
        AUTFPoolInfo name = pool.get(di.readShort()).execute(CheckUTFVisitor.singleton(), null);
        String attrName = name.toString();

        int len = di.readInt();
        byte[] data = new byte[len];
        int offset = 0, bytesRead;
        while((offset < data.length) && (-1 != (bytesRead = di.read(data, offset, data.length - offset)))) {
            offset += bytesRead;
        }

        if ((len != data.length) && ((-1 != len) || (0 != data.length))) {
            throw new ClassFormatError(
                "Attribute data of " + attrName + " has wrong length, actual=" + data.length + ", expected=" + len);
        }

        for (Class c: _knownAttributes) {
            try {
                Method m = c.getMethod("getAttributeName");
                String knownName = (String)m.invoke(null);
                if (knownName.equals(name.toString())) {
                    Constructor ctor = c.getConstructor(AUTFPoolInfo.class, byte[].class, ConstantPool.class);
                    return (AAttributeInfo)ctor.newInstance(name, data, pool);
                }
            }
            catch(NoSuchMethodException e) {
                throw new Error("Error creating attribute info", e);
            }
            catch(IllegalAccessException e) {
                throw new Error("Error creating attribute info", e);
            }
            catch(InvocationTargetException e) {
                throw new Error("Error creating attribute info", e);
            }
            catch(InstantiationException e) {
                e.printStackTrace();
            }
        }

        return new UnknownAttributeInfo(name, data, pool);
    }


    /**
     * Constructor.
     *
     * @param name attrobite name
     * @param data attribute data
     * @param cp constant pool
     */
    public AAttributeInfo(AUTFPoolInfo name, byte[] data, ConstantPool cp) {
        _name = name;
        _data = data;
        _constantPool = cp;
    }

    /**
     * Accessor for _name information.
     *
     * @return _name information
     */
    public AUTFPoolInfo getName() {
        return _name;
    }

    /**
     * Mutator for _name information.
     *
     * @param name _name information
     */
    public void setName(AUTFPoolInfo name) {
        _name = name;
    }

    /**
     * Accessor for data.
     *
     * @return data data
     */
    public byte[] getData() {
        return _data;
    }

    /**
     * Mutator for data.
     *
     * @param data data
     */
    public void setData(byte[] data) {
        _data = data;
    }

    /**
     * Write this attribute into a stream
     *
     * @param dos output stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        dos.writeShort(_constantPool.indexOf(_name));
        dos.writeInt(_data.length);
        dos.write(_data, 0, _data.length);
    }

    /**
     * Return a human-readable version of this attribute.
     *
     * @return string
     */
    public String toString() {
        String hexChars = "0123456789abcdef";
        StringBuffer sb = new StringBuffer();
        sb.append(_name);
        sb.append(" <");
        sb.append(_data.length);
        sb.append(" bytes:");
        for (byte b:_data) {
            sb.append(' ');
            sb.append(hexChars.charAt((b>>>4)&0x0F));
            sb.append(hexChars.charAt(b&0x0F));
        }
        sb.append('>');
        return sb.toString();
    }

    /**
     * Execute a visitor on this attribute.
     *
     * @param visitor visitor
     * @param param   visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public abstract <R, D> R execute(IAttributeVisitor<R, D> visitor, D param);

    /**
     * Adjust program counter values contained in this attribute, starting at startPC, by adding deltaPC to them.
     *
     * @param startPC program counter to start at
     * @param deltaPC change in program counter values
     */
    public abstract void adjustPC(short startPC, short deltaPC);

    /**
     * Translate the program counter values contained in this attribute from an old line number table to a new one.
     *
     * @param index      critical point (insertion or deletion point)
     * @param deltaIndex delta value to add to all old line numbers greater than the critical point
     * @param oldLnt     old line number table
     * @param newLnt     new line number table
     */
    public abstract void translatePC(short index, short deltaIndex, LineNumberTable oldLnt, LineNumberTable newLnt);

    /**
     * Creates and returns a copy of this object.
     * @return a clone of this instance.
     * @throws CloneNotSupportedException if the object's class does not support the <code>Cloneable</code> interface.
     */
    public Object lone() throws CloneNotSupportedException {
        AAttributeInfo a = (AAttributeInfo)super.clone();

        a._data = new byte[_data.length];
        System.arraycopy(_data, 0, a._data, 0, _data.length);

        return a;
    }
}
