package sysModel.classFile.attributes;

import sysModel.classFile.attributes.visitors.IAttributeVisitor;
import sysModel.classFile.code.instructions.LineNumberTable;
import sysModel.classFile.constantPool.AUTFPoolInfo;
import sysModel.classFile.constantPool.ConstantPool;
import sysModel.classFile.Types;

import java.io.*;

/**
 * Represents the Code attribute in a class file.
 *
 * @author Mathias Ricken
 */
public class CodeAttributeInfo extends AAttributeInfo {
    /**
     * Storage class for code properties.
     */
    public static class CodeProperties {
        public short maxStack;
        public short maxLocals;
        public int codeLength;
        public short exceptionTableLength;
        public short attributesCount;

        public CodeProperties(short maxStack,
                              short maxLocals,
                              int codeLength,
                              short exceptionTableLength,
                              short attributesCount) {
            this.maxStack = maxStack;
            this.maxLocals = maxLocals;
            this.codeLength = codeLength;
            this.exceptionTableLength = exceptionTableLength;
            this.attributesCount = attributesCount;
        }
    }

    public static class ExceptionTableEntry {
        public short startPC;
        public short endPC;
        public short handlerPC;
        public short catchType;

        public ExceptionTableEntry(short startPC, short endPC, short handlerPC, short catchType) {
            this.startPC = startPC;
            this.endPC = endPC;
            this.handlerPC = handlerPC;
            this.catchType = catchType;
        }
    }

    /**
     * Properties read out from _data field.
     */
    protected CodeProperties _props = null;

    /**
     * Constructor.
     *
     * @param name attribute name
     * @param data attribute data
     * @param cp   constant pool
     *
     * @throws ClassFormatError
     */
    public CodeAttributeInfo(AUTFPoolInfo name, byte[] data, ConstantPool cp) throws ClassFormatError {
        super(name, data, cp);
    }

    /**
     * Constructor.
     *
     * @param name                  attribute name, must be a UTF item "Code"
     * @param maxStack              maximum state size
     * @param maxLocals             maximum number of locals
     * @param code                  bytecode
     * @param exceptionTableEntries exception table entries
     * @param attributes            attributes
     * @param cp                    constant pool
     *
     * @throws ClassFormatError
     * @throws IOException
     */
    public CodeAttributeInfo(AUTFPoolInfo name, short maxStack, short maxLocals, byte[] code,
                             ExceptionTableEntry[] exceptionTableEntries, AAttributeInfo[] attributes,
                             ConstantPool cp) throws ClassFormatError, IOException {
        super(name, null, cp);
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        DataOutputStream dos = new DataOutputStream(bos);
        dos.writeShort(maxStack);
        dos.writeShort(maxLocals);
        dos.writeInt(0); // no code
        dos.writeShort(0); // no exceptions
        dos.writeShort(0); // no attributes
        setData(bos.toByteArray());

        setCode(code);
        setExceptionTableEntries(exceptionTableEntries);
        setAttributes(attributes);
    }

    /**
     * Mutator for data.
     *
     * @param data data
     */
    public void setData(byte[] data) {
        _props = null;
        super.setData(data);
    }

    /**
     * Return a copy of the code properties.
     *
     * @return code properties
     *
     * @throws ClassFormatError
     */
    public CodeProperties getProperties() throws ClassFormatError {
        if (null == _props) {
            short maxStack = Types.ushortFromBytes(_data, 0);
            short maxLocals = Types.ushortFromBytes(_data, 2);
            int codeLength = Types.uintFromBytes(_data, 4);
            short exceptionTableLength = Types.ushortFromBytes(_data, 8 + codeLength);
            short attributesCount = Types.ushortFromBytes(_data, 10 + codeLength + 8 * exceptionTableLength);
            _props = new CodeProperties(maxStack, maxLocals, codeLength, exceptionTableLength, attributesCount);
        }

        return new CodeProperties(_props.maxStack, _props.maxLocals, _props.codeLength, _props.exceptionTableLength,
            _props.attributesCount);
    }

    /**
     * Set the code properties.
     *
     * @param maxStack  new maximum state
     * @param maxLocals new maximum locals
     *
     * @throws ClassFormatError
     */
    public void setProperties(short maxStack, short maxLocals) {
        getProperties();

        Types.bytesFromShort(_props.maxStack = maxStack, _data, 0);
        Types.bytesFromShort(_props.maxLocals = maxLocals, _data, 2);
    }

    /**
     * Return a copy of the code bytes.
     *
     * @return code bytes
     */
    public byte[] getCode() {
        getProperties();

        byte[] b = new byte[_props.codeLength];
        System.arraycopy(_data, 8, b, 0, _props.codeLength);

        return b;
    }

    /**
     * Set the code bytes.
     *
     * @param code new code bytes
     */
    public void setCode(byte[] code) throws IllegalArgumentException {
        if (0 == code.length) {
            throw new IllegalArgumentException("Code block cannot be empty");
        }

        getProperties();

        byte[] newData = new byte[_data.length - _props.codeLength + code.length];
        System.arraycopy(_data, 0, newData, 0, 4);
        Types.bytesFromInt(code.length, newData, 4);
        System.arraycopy(code, 0, newData, 8, code.length);
        System.arraycopy(_data, 8 + _props.codeLength, newData, 8 + code.length, _data.length - 8 - _props.codeLength);
        setData(newData);
    }

    /**
     * Return a copy of the exception table entries.
     *
     * @return entries
     */
    public ExceptionTableEntry[] getExceptionTableEntries() {
        getProperties();

        ExceptionTableEntry[] e = new ExceptionTableEntry[_props.exceptionTableLength];
        for(int i = 0, index = 10 + _props.codeLength; i < _props.exceptionTableLength; ++i, index += 8) {
            short startPC = Types.ushortFromBytes(_data, index);
            short endPC = Types.ushortFromBytes(_data, index + 2);
            short handlerPC = Types.ushortFromBytes(_data, index + 4);
            short catchType = Types.ushortFromBytes(_data, index + 6);
            e[i] = new ExceptionTableEntry(startPC, endPC, handlerPC, catchType);
        }

        return e;
    }

    /**
     * Set the exception table entries.
     *
     * @param e entries
     *
     * @throws IllegalArgumentException
     */
    public void setExceptionTableEntries(ExceptionTableEntry[] e) throws IllegalArgumentException {
        if (0xffff < e.length) {
            throw new IllegalArgumentException("Too many exception table entries, max=0xffff");
        }

        getProperties();

        byte[] newData = new byte[_data.length - 8 * _props.exceptionTableLength + 8 * e.length];
        System.arraycopy(_data, 0, newData, 0, 8 + _props.codeLength);
        Types.bytesFromShort((short)e.length, newData, 8 + _props.codeLength);

        for(int i = 0; i < e.length; ++i) {
            Types.bytesFromShort(e[i].startPC, newData, 10 + _props.codeLength + i * 8);
            Types.bytesFromShort(e[i].endPC, newData, 12 + _props.codeLength + i * 8);
            Types.bytesFromShort(e[i].handlerPC, newData, 14 + _props.codeLength + i * 8);
            Types.bytesFromShort(e[i].catchType, newData, 16 + _props.codeLength + i * 8);
        }

        System.arraycopy(_data,
            10 + _props.codeLength + 8 * _props.exceptionTableLength,
            newData,
            10 + _props.codeLength + 8 * e.length,
            _data.length - 10 - _props.codeLength - 8 * _props.exceptionTableLength);
        setData(newData);
    }

    /**
     * Return a copy of the attributes.
     *
     * @return attributes
     *
     * @throws ClassFormatError
     */
    public AAttributeInfo[] getAttributes() throws ClassFormatError {
        getProperties();

        AAttributeInfo[] a = new AAttributeInfo[_props.attributesCount];
        byte[] b = new byte[_data.length - 12 - _props.codeLength - 8 * _props.exceptionTableLength];
        System.arraycopy(_data,
            12 + _props.codeLength + 8 * _props.exceptionTableLength,
            b,
            0,
            b.length);
        try {
            DataInputStream dis = new DataInputStream(new ByteArrayInputStream(b));
            for(int i = 0; i < _props.attributesCount; ++i) {
                a[i] = AAttributeInfo.read(dis, _constantPool);
            }
        }
        catch(IOException e) {
            throw new ClassFormatError("Could not get code attributes: "+e);
        }

        return a;
    }

    /**
     * Sets the attribute list.
     *
     * @param a attributes
     *
     * @throws ClassFormatError
     * @throws IllegalArgumentException
     */
    public void setAttributes(AAttributeInfo[] a) throws ClassFormatError, IllegalArgumentException {
        if (0xffff < a.length) {
            throw new IllegalArgumentException("Too many atttributes, max=0xffff");
        }

        getProperties();

        byte[] newData = new byte[0];
        try {
            ByteArrayOutputStream baos = new ByteArrayOutputStream();
            DataOutputStream dos = new DataOutputStream(baos);
            for(AAttributeInfo attr : a) {
                attr.write(dos);
            }

            byte[] attr = baos.toByteArray();
            newData = new byte[12 + _props.codeLength + 8 * _props.exceptionTableLength + attr.length];
            System.arraycopy(_data, 0, newData, 0, 10 + _props.codeLength + 8 * _props.exceptionTableLength);
            Types.bytesFromShort((short)a.length, newData, 10 + _props.codeLength + 8 * _props.exceptionTableLength);
            System.arraycopy(attr, 0, newData, newData.length - attr.length, attr.length);
        }
        catch(IOException e) {
            throw new ClassFormatError("Cannot set code attributes: "+e);
        }

        setData(newData);
    }

    /**
     * Execute a visitor on this attribute.
     *
     * @param visitor visitor
     * @param param   visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public <R, D> R execute(IAttributeVisitor<R, D> visitor, D param) {
        return visitor.codeCase(this, param);
    }

    /**
     * Return a human-readable version of this attribute.
     *
     * @return string
     */
    public String toString() {
        getProperties();
        StringBuffer x = new StringBuffer();
        x.append("Code <" + _data.length + " bytes, " + _props.maxStack + " max stack, " +
            _props.maxLocals + " max locals, " + _props.codeLength + " code bytes, " +
            _props.exceptionTableLength + " exception table entries { ");

        for (ExceptionTableEntry e: getExceptionTableEntries()) {
            x.append("(pc="+e.startPC+".."+e.endPC+" handler="+e.handlerPC+" type="+e.catchType+") ");
        }

        x.append("}, " + _props.attributesCount + " attributes = { ");

        boolean first = true;
        for(AAttributeInfo a : getAttributes()) {
            if (first) {
                first = false;
            }
            else {
                x.append(", ");
            }
            x.append(a.toString());
        }

        x.append("} >");

        return x.toString();
    }

    /**
     * Adjust program counter values contained in this attribute, starting at startPC, by adding deltaPC to them.
     *
     * NOTE: This does not transform branch targets in the code.
     *
     * @param startPC program counter to start at
     * @param deltaPC change in program counter values
     */
    public void adjustPC(short startPC, short deltaPC) {
        // adjust PCs in exception table
        ExceptionTableEntry[] exceptions = getExceptionTableEntries();
        for(ExceptionTableEntry exc : exceptions) {
            if (exc.startPC >= startPC) {
                // Debug.out.println("Adjusted CodeAttribute.exceptions[].startPC: "+exc.startPC+" --> "+(exc.startPC+deltaPC));
                exc.startPC += deltaPC;
            }
            if (exc.endPC >= startPC) {
                // Debug.out.println("Adjusted CodeAttribute.exceptions[].endPC: "+exc.endPC+" --> "+(exc.endPC+deltaPC));
                exc.endPC += deltaPC;
            }
            if (exc.handlerPC >= startPC) {
                // Debug.out.println("Adjusted CodeAttribute.exceptions[].handlerPC: "+exc.handlerPC+" --> "+(exc.handlerPC+deltaPC));
                exc.handlerPC += deltaPC;
            }
        }
        setExceptionTableEntries(exceptions);

        // recursively adjust PC in all contained attributes
        AAttributeInfo[] attributes = getAttributes();
        for(AAttributeInfo attr : attributes) {
            attr.adjustPC(startPC, deltaPC);
        }
        setAttributes(attributes);
    }

    /**
     * Translate the program counter values contained in this attribute from an old line number table to a new one.
     *
     * @param index      critical point (insertion or deletion point)
     * @param deltaIndex delta value to add to all old line numbers greater than the critical point
     * @param oldLnt     old line number table
     * @param newLnt     new line number table
     */
    public void translatePC(short index, short deltaIndex, LineNumberTable oldLnt, LineNumberTable newLnt) {
        // adjust PCs in exception table
        ExceptionTableEntry[] exceptions = getExceptionTableEntries();
        for(ExceptionTableEntry exc : exceptions) {
            int oldLineNo = oldLnt.getLineNumber(exc.startPC);
            oldLineNo += (oldLineNo > index) ? deltaIndex : 0;
            exc.startPC = (short)newLnt.getPC(oldLineNo);

            oldLineNo = oldLnt.getLineNumber(exc.endPC);
            oldLineNo += (oldLineNo > index) ? deltaIndex : 0;
            exc.endPC = (short)newLnt.getPC(oldLineNo);

            oldLineNo = oldLnt.getLineNumber(exc.handlerPC);
            oldLineNo += (oldLineNo > index) ? deltaIndex : 0;
            exc.handlerPC = (short)newLnt.getPC(oldLineNo);
        }
        setExceptionTableEntries(exceptions);

        // recursively adjust PC in all contained attributes
        AAttributeInfo[] attributes = getAttributes();
        for(AAttributeInfo attr : attributes) {
            attr.translatePC(index, deltaIndex, oldLnt, newLnt);
        }
        setAttributes(attributes);
    }

    /**
     * Creates and returns a copy of this object.
     */
    public Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

    /**
     * Returns the name of the attribute as it appears in the class file.
     * @return name of the attribute.
     */
    public static String getAttributeName() {
        return "Code";
    }
}
