package sysModel.classFile.attributes;

import sysModel.classFile.attributes.visitors.IAttributeVisitor;
import sysModel.classFile.code.instructions.LineNumberTable;
import sysModel.classFile.constantPool.AUTFPoolInfo;
import sysModel.classFile.constantPool.ConstantPool;
import sysModel.classFile.constantPool.visitors.CheckUTFVisitor;
import sysModel.classFile.Types;

/**
 * Represents the SourceFile attribute in a class file.
 *
 * @author Mathias Ricken
 */
public class SourceFileAttributeInfo extends AAttributeInfo {
    /**
     * Constructor.
     *
     * @param name attribute name
     * @param data attribute data
     * @param cp   constant pool
     *
     * @throws ClassFormatError
     */
    public SourceFileAttributeInfo(AUTFPoolInfo name, byte[] data, ConstantPool cp) throws ClassFormatError {
        super(name, data, cp);
    }

    /**
     * Return the source file name information.
     *
     * @return source file name information
     *
     * @throws ClassFormatError
     */
    public AUTFPoolInfo getSourceFileName() throws ClassFormatError {
        return _constantPool.get(Types.ushortFromBytes(_data, 0)).execute(CheckUTFVisitor.singleton(), null);
    }

    /**
     * Set the source file name information.
     *
     * @param newFileName constant pool
     */
    public void setSourceFileName(AUTFPoolInfo newFileName) {
        _data = new byte[2];
        Types.bytesFromShort(_constantPool.indexOf(newFileName), _data, 0);
    }

    /**
     * Execute a visitor on this attribute.
     *
     * @param visitor visitor
     * @param param   visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public <R, D> R execute(IAttributeVisitor<R, D> visitor, D param) {
        return visitor.sourceFileCase(this, param);
    }

    /**
     * Adjust program counter values contained in this attribute, starting at startPC, by adding deltaPC to them.
     *
     * @param startPC program counter to start at
     * @param deltaPC change in program counter values
     */
    public void adjustPC(short startPC, short deltaPC) {
        // nothing to do
    }

    /**
     * Translate the program counter values contained in this attribute from an old line number table to a new one.
     *
     * @param index      critical point (insertion or deletion point)
     * @param deltaIndex delta value to add to all old line numbers greater than the critical point
     * @param oldLnt     old line number table
     * @param newLnt     new line number table
     */
    public void translatePC(short index, short deltaIndex, LineNumberTable oldLnt, LineNumberTable newLnt) {
        // nothing to do
    }

    /**
     * Creates and returns a copy of this object.
     */
    public Object clone() throws CloneNotSupportedException {
        return super.clone();
    }

    /**
     * Return a human-readable version of this attribute.
     *
     * @return string
     */
    public String toString() {
        return "Source File = "+getSourceFileName().toString();
    }

    /**
     * Returns the name of the attribute as it appears in the class file.
     *
     * @return name of the attribute.
     */
    public static String getAttributeName() {
        return "SourceFile";
    }
}
