package sysModel.classFile.code;

import sysModel.classFile.attributes.CodeAttributeInfo;
import sysModel.classFile.code.instructions.AInstruction;
import sysModel.classFile.code.instructions.LineNumberTable;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.LinkedList;

/**
 * Represents a list of Java instructions.
 *
 * @author Mathias Ricken
 */
public class InstructionList {
    LinkedList<AInstruction> _instructionList = new LinkedList<AInstruction>();
    protected short _index;
    LineNumberTable _lnt;

    /**
     * Constructor.
     *
     * @param code bytecode
     */
    public InstructionList(byte[] code) {
        setCode(code);
        _index = 0;
    }

    /**
     * Copy constructor.
     *
     * @param original original
     */
    public InstructionList(InstructionList original) {
        _instructionList.addAll(original._instructionList);
        _index = original._index;
        _lnt = original._lnt;
    }

    /**
     * Constructor.
     *
     * @param code  bytecode
     * @param index program index.
     *
     * @throws IndexOutOfBoundsException
     */
    public InstructionList(byte[] code, short index) throws IndexOutOfBoundsException {
        this(code);
        setIndex(index);
    }

    /**
     * Return a human-readable version of the code.
     *
     * @return mnemonics
     */
    public String toString() {
        return toString(true, true);
    }

    /**
     * Return a human-readable version of the code.
     *
     * @param lineNumbers print line numbers
     * @param PCs print PC values
     * @return mnemonics
     */
    public String toString(boolean lineNumbers, boolean PCs) {
        StringBuffer x = new StringBuffer();

        short index = 0;
        short pc = 0;
        for(AInstruction instr : _instructionList) {
            if (lineNumbers) {
                x.append(String.format("%5d", new Object[]{new Integer(index)}));
                if (PCs) {
                    x.append(' ');
                }
                else {
                    x.append(": ");
                }
            }
            if (PCs) {
                x.append(String.format("(pc %5d): ", new Object[]{new Integer(pc)}));
            }
            if ((!lineNumbers) && (!PCs)) {
                x.append("        ");
            }
            x.append(instr);
            x.append('\n');
            pc += instr.getBytecodeLength(pc);
            ++index;
        }

        return x.toString();
    }

    /**
     * Accessor for the bytecode.
     *
     * @return bytecode
     *
     * @throws IllegalStateException
     */
    public byte[] getCode() throws IllegalStateException {
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        short pc = 0;
        for(AInstruction instr : _instructionList) {
            byte[] b = instr.getBytecode(pc, _lnt);
            try {
                baos.write(b);
            }
            catch(IOException e) {
                throw new IllegalStateException("Could not generate bytecode", e);
            }

            pc += b.length;
        }

        return baos.toByteArray();
    }

    /**
     * Mutator for the bytecode. This also resets the program counter to 0.
     *
     * @param code bytecode
     */
    public void setCode(byte[] code) {
        _lnt = new LineNumberTable(code);

        short pc = 0;
        _instructionList = new LinkedList<AInstruction>();
        while(pc < code.length) {
            // make instruction and add it
            _instructionList.addLast(AInstruction.makeInstruction(code, pc, pc, _lnt));

            pc += Opcode.getInstrSize(code, pc);
        }
        if (pc != code.length) {
            throw new IllegalArgumentException("Invalid bytecode length");
        }
        _index = 0;
    }

    /**
     * Accessor for the program index.
     *
     * @return program index
     */
    public short getIndex() {
        return _index;
    }

    /**
     * Calculate the PC for an index.
     *
     * @param index index
     *
     * @return PC
     */
    public short getPCFromIndex(short index) {
        short pc = 0;
        for(int i = 0; i < index; ++i) {
            pc += _instructionList.get(i).getBytecodeLength(pc);
        }

        return pc;
    }

    /**
     * Get the length of the instruction list.
     *
     * @return length
     */
    public short getLength() {
        return (short)_instructionList.size();
    }

    /**
     * Mutator for the program index.
     *
     * @param index new program index
     *
     * @throws IndexOutOfBoundsException
     */
    public void setIndex(int index) throws IndexOutOfBoundsException, IllegalArgumentException {
        if ((0 > index) || (index >= _instructionList.size())) {
            throw new IndexOutOfBoundsException(
                "The program index (" + index + ") is not within the instruction list (length=" + _instructionList.size() + ')');
        }

        _index = (short)index;
    }

    /**
     * Advance the program index by multiple instructions.
     * @param count number of instructions
     * @return true while the end of the instruction list has not been reached
     */
    public boolean advanceIndex(int count) {
        while(0 < count) {
            if (!advanceIndex()) {
                return false;
            }
            --count;
        }
        return true;
    }

    /**
     * Advance the program index to the next instruction.
     *
     * @return true while the end of the instruction list has not been reached
     */
    public boolean advanceIndex() {
        if (0 > _index) {
            throw new IndexOutOfBoundsException(
                "The program index (" + _index + ") is not within the instruction list (length=" + _instructionList.size() + ')');
        }
        if (_index < _instructionList.size()) {
            ++_index;
        }
        return (_index < _instructionList.size());
    }

    /**
     * Rewind the program index by multiple instructions.
     * @param count number of instructions
     * @return true while the beginning of the instruction list has not been reached
     */
    public boolean rewindIndex(int count) {
        while(0 < count) {
            if (!rewindIndex()) {
                return false;
            }
            --count;
        }
        return true;
    }

    /**
     * Rewind the program index to the previous instruction.
     *
     * @return true while the beginning of the instruction list has not been reached
     */
    public boolean rewindIndex() {
        if (0 > _index) {
            throw new IndexOutOfBoundsException(
                "The program index (" + _index + ") is not within the instruction list (length=" + _instructionList.size() + ')');
        }
        if (0 < _index) {
            --_index;
        }
        return (0 < _index);
    }

    /**
     * Return the opcode at the program index.
     *
     * @return opcode at program index
     */
    public byte getOpcode() {
        if ((0 > _index) || (_index >= _instructionList.size())) {
            throw new IndexOutOfBoundsException(
                "The program index (" + _index + ") is not within the instruction list (length=" + _instructionList.size() + ')');
        }
        return _instructionList.get(_index).getOpcode();
    }

    /**
     * Return the instruction at the program index. This includes the code and the operands.
     *
     * @return instruction at program index
     */
    public AInstruction getInstr() {
        if ((0 > _index) || (_index >= _instructionList.size())) {
            throw new IndexOutOfBoundsException(
                "The program index (" + _index + ") is not within the instruction list (length=" + _instructionList.size() + ')');
        }

        return _instructionList.get(_index);
    }

    /**
     * This method deletes the instruction at the program index. It also relocates all jumps. All jumps to places after
     * the deletion point will be changed so that they still point to the same instruction. Jumps to the deletion point
     * or to places before it are not changed. The index will remain unchanged. In case the program counter is past the
     * end of the code block, false will be returned. If codeAttribute is non-null, the PCs in the CodeAttribute will be
     * adjusted.
     *
     * @param codeAttribute CodeAttribute that contains this code block, or null if none
     *
     * @return true while the end of the code block has not been reached
     */
    public boolean deleteInstr(CodeAttributeInfo codeAttribute) {
        if ((0 > _index) || (_index >= _instructionList.size())) {
            throw new IndexOutOfBoundsException(
                "The program index (" + _index + ") is not within the instruction list (length=" + _instructionList.size() + ')');
        }

        byte[] oldCode = getCode();
        short deletionPoint = _index;

        // walk through all instructions and relocate jumps
        InstructionList oldCodeBlock = new InstructionList(oldCode);
        do {
            if (Opcode.isBranch(oldCodeBlock.getOpcode())) {
                // this is a branch, so we might have to relocate
                short[] branchTargets = oldCodeBlock.getInstr().getBranchTargets();
                for(int i = 0; i < branchTargets.length; ++i) {
                    // Debug.out.println("insertInstr: DP = "+deletionPoint+", PC="+oldCodeBlock.getPC()+", BT="+branchTargets[i]);
                    if ((branchTargets[i] > deletionPoint) || (branchTargets[i] == oldCodeBlock.getLength() - 1)) {
                        // the target of this instruction was the insertion point or somewhere past it
                        // we need to relocate
                        --branchTargets[i];

                        // Debug.out.println("\tnew BT="+branchTargets[i]);
                    }
                }
                oldCodeBlock.getInstr().setBranchTargets(branchTargets);
            }
        } while(oldCodeBlock.advanceIndex());

        oldCodeBlock._instructionList.remove(deletionPoint);
        oldCodeBlock._lnt = new LineNumberTable(oldCodeBlock);
        byte[] newCode = oldCodeBlock.getCode();

        if (null != codeAttribute) {
            // adjust CodeAttribute PCs
            codeAttribute.translatePC(deletionPoint, (short)-1, _lnt, oldCodeBlock._lnt);
        }

        // transfer the code
        setCode(newCode);

        if (deletionPoint == getLength()) {
            _index = getLength();
            return false;
        }
        else {
            setIndex(deletionPoint);
            return true;
        }
    }


    /**
     * This method inserts the instruction at the program counter. The instruction currently at the program counter will
     * be pushed towards the end of the bytecode array. It also relocates all jumps. Jumps to the insertion point or to
     * places before it are not changed. Jumps to all other places will be changed so that they still point to the same
     * instruction. That means that jumps to the instruction previously at the insertion point will now jump to the
     * newly inserted instruction. If codeAttribute is non-null, the PCs in the CodeAttribute will be adjusted. The
     * program counter will remain unchanged. If the inserted instruction itself contains jump targets, these will be
     * relocated as well. Notice, though, that if the instruction contains a jump exactly to the insertion point, this
     * jump will be relocated so that it points to the old instruction (this is the behavior of insertBeforeInstr).
     * <p/>
     * Assume the following code:
     * 1. a
     * 2. b
     * 3. c
     * 4. goto 1 // jump to place before insertion point
     * 5. goto 2 // jump to insertion point
     * 6. goto 3 // junp to place after insertion point
     * The program counter is currently at 2, and the instruction x is inserted using insertInstr. The result is:
     * 1. a
     * 2. x // inserted
     * 3. b
     * 4. c
     * 5. goto 1 // not changed
     * 6. goto 2 // not changed
     * 7. goto 4 // changed
     * <p/>
     * Relocation of branches in the instruction inserted.
     * Type   : jump to place before insertion point
     * Action : not changed
     * Example; goto 1 --> goto 1
     *
     * Type   : jump to insertion point
     * Action : changed
     * Example; goto 2 --> goto 3

     * Type   : jump to place after instruction point
     * Action : changed
     * Example; goto 3 --> goto 4
     *
     * @param codeAttribute CodeAttribute that contains this code block, or null if none
     * @param instr         instruction to be inserted
     */
    public void insertInstr(AInstruction instr, CodeAttributeInfo codeAttribute) {
        byte[] oldCode = getCode();
        short insertionPoint = _index;

        // walk through all instructions and relocate jumps
        InstructionList oldCodeBlock = new InstructionList(oldCode);
        do {
            if (Opcode.isBranch(oldCodeBlock.getOpcode())) {
                // this is a branch, so we might have to relocate
                AInstruction relocInstr = oldCodeBlock.getInstr();
                relocateBranches(relocInstr, insertionPoint);
            }
        } while(oldCodeBlock.advanceIndex());

        // relocate instruction to be inserted
        if (Opcode.isBranch(instr.getOpcode())) {
            // this is a branch, so we might have to relocate
            relocateBranches(instr, insertionPoint-1);
        }

        oldCodeBlock._instructionList.add(insertionPoint, instr);
        oldCodeBlock._lnt = new LineNumberTable(oldCodeBlock);
        byte[] newCode = oldCodeBlock.getCode();

        if (null != codeAttribute) {
            // adjust CodeAttribute PCs
            codeAttribute.translatePC(insertionPoint, (short)1, _lnt, oldCodeBlock._lnt);
        }

        // transfer the code
        setCode(newCode);
        setIndex(insertionPoint);
    }

    /**
     * Relocate branches in the instruction. If a branch target is to somewhere past the insertion point, it is
     * increased by one.
     * @param relocInstr instruction whose branches should be relocated
     * @param ip insertion point
     */
    protected void relocateBranches(AInstruction relocInstr, int ip) {
        short[] branchTargets = relocInstr.getBranchTargets();
        for(int i = 0; i < branchTargets.length; ++i) {
            // Debug.out.println("insertInstr: DP = "+deletionPoint+", PC="+oldCodeBlock.getPC()+", BT="+branchTargets[i]);
            if (branchTargets[i] > ip) {
                // the target of this instruction was somewhere past the insertion point
                // we need to relocate
                ++branchTargets[i];

                // Debug.out.println("\tnew BT="+branchTargets[i]);
            }
        }
        relocInstr.setBranchTargets(branchTargets);
    }

    /**
     * This method inserts the instruction before the program counter. The instruction currently at the program counter
     * will be pushed towards the end of the bytecode array. It also relocates all jumps. Jumps to places before the
     * insertion point are not changed. Jumps to all other places will be changed so that they still point to the same
     * instruction. That means that jumps to the instruction previously at the insertion point will still jump to the
     * old instruction. If codeAttribute is non-null, the PCs in the CodeAttribute will be adjusted. The
     * program counter will remain unchanged. If the inserted instruction itself contains jump targets, these will be
     * relocated as well.
     * <p/>
     * Assume the following code:
     * 1. a
     * 2. b
     * 3. c
     * 4. goto 1 // jump to place before insertion point
     * 5. goto 2 // jump to insertion point
     * 6. goto 3 // junp to place after insertion point
     * The program counter is currently at 2, and the instruction x is inserted using insertBeforeInstr. The result is:
     * 1. a
     * 2. x // inserted
     * 3. b
     * 4. c
     * 5. goto 1 // not changed
     * 6. goto 3 // changed
     * 7. goto 4 // changed
     * <p/>
     * Relocation of branches in the instruction inserted.
     * Type   : jump to place before insertion point
     * Action : not changed
     * Example; goto 1 --> goto 1
     *
     * Type   : jump to insertion point
     * Action : changed
     * Example; goto 2 --> goto 3

     * Type   : jump to place after instruction point
     * Action : changed
     * Example; goto 3 --> goto 4
     *
     * @param codeAttribute CodeAttribute that contains this code block, or null if none
     * @param instr         instruction to be inserted
     */
    public void insertBeforeInstr(AInstruction instr, CodeAttributeInfo codeAttribute) {
        byte[] oldCode = getCode();
        short insertionPoint = _index;

        // walk through all instructions and relocate jumps
        InstructionList oldCodeBlock = new InstructionList(oldCode);
        do {
            if (Opcode.isBranch(oldCodeBlock.getOpcode())) {
                // this is a branch, so we might have to relocate
                AInstruction relocInstr = oldCodeBlock.getInstr();
                relocateBranches(relocInstr, insertionPoint-1);
            }
        } while(oldCodeBlock.advanceIndex());

        // relocate instruction to be inserted
        if (Opcode.isBranch(instr.getOpcode())) {
            // this is a branch, so we might have to relocate
            relocateBranches(instr, insertionPoint-1);
        }

        oldCodeBlock._instructionList.add(insertionPoint, instr);
        oldCodeBlock._lnt = new LineNumberTable(oldCodeBlock);
        byte[] newCode = oldCodeBlock.getCode();

        if (null != codeAttribute) {
            // adjust CodeAttribute PCs
            codeAttribute.translatePC((short)(insertionPoint-1), (short)1, _lnt, oldCodeBlock._lnt);
        }

        // transfer the code
        setCode(newCode);
        setIndex(insertionPoint);
    }

    /**
     * Finds the next instruction with the specified opcode and moves the program counter there. If the current
     * instruction matches the opcode, the program counter will not be changed. This means that the caller is
     * responsible for advancing the program counter after a match has been found!
     *
     * @param opcode opcode to find
     *
     * @return true while the end of the code block has not been reached
     */
    public boolean findOpcode(byte opcode) {
        if (0 > _index) {
            throw new IndexOutOfBoundsException(
                "The program index (" + _index + ") is not within the instruction list (length=" + _instructionList.size() + ')');
        }
        if (_index == _instructionList.size()) {
            // past end already
            return false;
        }
        do {
            if (getOpcode() == opcode) {
                return true;
            }
        } while(advanceIndex());
        return false;
    }

    /**
     * Finds the next instruction that matches the specified instruction completely and moves the program counter there.
     * A complete match requires that both the opcode and all the operands are identical. If the current instruction
     * matches already, the program counter will not be changed. This means that the caller is responsible for advancing
     * the program counter after a match has been found!
     *
     * @param instr instruction to find
     *
     * @return true while the end of the code block has not been reached
     */
    public boolean findInstruction(AInstruction instr) {
        if (0 > _index) {
            throw new IndexOutOfBoundsException(
                "The program index (" + _index + ") is not within the instruction list (length=" + _instructionList.size() + ')');
        }
        if (_index == _instructionList.size()) {
            // past end already
            return false;
        }
        do {
            if (getInstr().equals(instr)) {

                return true;
            }
        } while(advanceIndex());
        return false;
    }

    /**
     * Return true if this instruction list and the other object are equal.
     *
     * @param o other object
     *
     * @return true if equal
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof InstructionList)) {
            return false;
        }

        InstructionList instructionList = (InstructionList)o;
        return _instructionList.equals(instructionList._instructionList);
    }

    /**
     * Return hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _instructionList.hashCode();
    }
}
