package sysModel.classFile.code;

import sysModel.classFile.code.instructions.*;
import junit.framework.TestCase;

import java.io.IOException;
import java.util.Arrays;

/**
 * Unit tests for InstructionList class.
 *
 * @author Mathias Ricken
 */
public class InstructionListTest extends TestCase {
    protected final byte[] TEST_CODE = new byte[]{
        Opcode.ALOAD_0, // 0
        Opcode.IFEQ, 0, 0, // 1
        Opcode.GOTO, 0, 0, // 4
        Opcode.GOTO_W, 0, 0, 0, 0, // 7
        Opcode.ALOAD, 0, // 12
        Opcode.WIDE, Opcode.ALOAD, 0, 0, // 14
        Opcode.IINC, 0, 0, // 18
        Opcode.WIDE, Opcode.IINC, 0, 0, 0, 0, // 21
        Opcode.TABLESWITCH, /*default*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xf1, /*low*/0, 0, 0, 5, /*high*/0, 0, 0, 7,
        /*5*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xf3, /*6*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xf7, /*7*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xfa, // 27
        Opcode.LOOKUPSWITCH, /*pad*/ 0, 0, 0, /*default*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xe7, /*n*/0, 0, 0, 2, /*key*/0, 0, 0, 0,
        /*offset*/0, 0, 0, 0, /*key*/ 0, 0, 0, 1, /*offset*/ (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x1c, // 52
        Opcode.RETURN // 80
    };

    protected final byte[] TEST_CODE2 = new byte[]{
        Opcode.ALOAD_0,
        Opcode.ALOAD_1,
        Opcode.INVOKESTATIC, 0, 0,
        Opcode.RETURN
    };

    protected final byte[] TEST_CODE_INVALID = new byte[]{
        Opcode.ALOAD
    };

    protected final byte[] TEST_CODE_EMPTY = new byte[]{};

    protected final byte[] TEST_CODE_INSERTDELETE = new byte[]{
        Opcode.GOTO, (byte)0x00, (byte)0x0A, // 0
        Opcode.GOTO, (byte)0x00, (byte)0x03, // 3
        Opcode.NOP, // 6
        Opcode.GOTO, (byte)0xFF, (byte)0xFC, // 7
        Opcode.GOTO, (byte)0xFF, (byte)0xFD // 10
    };

    protected final byte[] TEST_CODE_DELETE_RESULT = new byte[]{
        Opcode.GOTO, (byte)0x00, (byte)0x09, // 0
        Opcode.GOTO, (byte)0x00, (byte)0x03, // 3
        Opcode.GOTO, (byte)0xFF, (byte)0xFD, // 6
        Opcode.GOTO, (byte)0xFF, (byte)0xFD // 9
    };

    protected final byte[] TEST_CODE_DELETE_RESULT2 = new byte[]{
        Opcode.GOTO, (byte)0x00, (byte)0x06, // 0
        Opcode.GOTO, (byte)0x00, (byte)0x03, // 3
        Opcode.GOTO, (byte)0xFF, (byte)0xFD // 6
    };

    protected final byte[] TEST_CODE_INSERT_RESULT = new byte[]{
        Opcode.GOTO, (byte)0x00, (byte)0x0B, // 0
        Opcode.GOTO, (byte)0x00, (byte)0x03, // 3
        Opcode.NOP, // 6
        Opcode.NOP, // 7
        Opcode.GOTO, (byte)0xFF, (byte)0xFB, // 8
        Opcode.GOTO, (byte)0xFF, (byte)0xFD // 11
    };

    protected final byte[] TEST_CODE_INSERT_RESULT2 = new byte[]{
        Opcode.GOTO, (byte)0x00, (byte)0x25, // 0
        Opcode.GOTO, (byte)0x00, (byte)0x03, // 3
        Opcode.TABLESWITCH, 0, /*default*/(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x1A, /*low*/ (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x05,
        /*high*/(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x07, /*5*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xfd,
        /*6*/(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x1B, /*7*/(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x1F, // 6
        Opcode.NOP, // 32
        Opcode.NOP, // 33
        Opcode.GOTO, (byte)0xFF, (byte)0xE1, // 34
        Opcode.GOTO, (byte)0xFF, (byte)0xFD // 37
    };

    protected final byte[] TEST_CODE_INSERT_RESULT3 = new byte[]{
        Opcode.GOTO, (byte)0x00, (byte)0x25, // 0
        Opcode.GOTO, (byte)0x00, (byte)0x03, // 3
        Opcode.NOP, // 6
        Opcode.TABLESWITCH, /*default*/(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x19, /*low*/ (byte)0x00, (byte)0x00, (byte)0x00, (byte)0x05,
        /*high*/(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x07, /*5*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xfc,
        /*6*/(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x1A, /*7*/(byte)0x00, (byte)0x00, (byte)0x00, (byte)0x1E, // 7
        Opcode.NOP, // 32
        Opcode.NOP, // 33
        Opcode.GOTO, (byte)0xFF, (byte)0xE1, // 34
        Opcode.GOTO, (byte)0xFF, (byte)0xFD // 37
    };

    public void testCtorAndGetCode() throws IOException {
        InstructionList ilist = new InstructionList(TEST_CODE);

        byte[] generatedCode = ilist.getCode();
        assertTrue("Generated bytecode not the same", Arrays.equals(TEST_CODE, generatedCode));

        ilist.setCode(TEST_CODE2);

        generatedCode = ilist.getCode();
        assertTrue("Generated bytecode not the same", Arrays.equals(TEST_CODE2, generatedCode));
    }

    public void testExceptions() {
        try {
            InstructionList ilist = new InstructionList(TEST_CODE_INVALID);
            fail("Should have thrown IllegalArgumentException, invalid block length");
        }
        catch(IllegalArgumentException e) {
            // passed
        }
        catch(Throwable t) {
            fail("Should have thrown IllegalArgumentException, invalid block length");
        }

        InstructionList ilist = new InstructionList(TEST_CODE_EMPTY);
        try {
            ilist.setCode(TEST_CODE_INVALID);
            fail("Should have thrown IllegalArgumentException, invalid block length");
        }
        catch(IllegalArgumentException e) {
            // passed
        }
        catch(Throwable t) {
            fail("Should have thrown IllegalArgumentException, invalid block length");
        }

        ilist = new InstructionList(TEST_CODE);
        try {
            ilist.setIndex(-1);
            fail("Should have thrown IndexOutOfBoundsException, invalid index");
        }
        catch(IndexOutOfBoundsException e) {
            // passed
        }
        catch(Throwable t) {
            fail("Should have thrown IndexOutOfBoundsException, invalid block length");
        }
        try {
            ilist.setIndex(11);
            fail("Should have thrown IndexOutOfBoundsException, invalid index");
        }
        catch(IndexOutOfBoundsException e) {
            // passed
        }
        catch(Throwable t) {
            fail("Should have thrown IndexOutOfBoundsException, invalid block length");
        }
    }

    public void testIndex() {
        InstructionList ilist = new InstructionList(TEST_CODE);

        assertEquals("Wrong index", 0, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.ALOAD_0, ilist.getOpcode());
        assertEquals("Wrong instruction", GenericInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 1, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.IFEQ, ilist.getOpcode());
        assertEquals("Wrong instruction", BranchInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 2, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.GOTO, ilist.getOpcode());
        assertEquals("Wrong instruction", BranchInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 3, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.GOTO_W, ilist.getOpcode());
        assertEquals("Wrong instruction", WideBranchInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 4, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.ALOAD, ilist.getOpcode());
        assertEquals("Wrong instruction", GenericInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 5, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.WIDE, ilist.getOpcode());
        assertEquals("Wrong instruction", WideInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 6, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.IINC, ilist.getOpcode());
        assertEquals("Wrong instruction", GenericInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 7, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.WIDE, ilist.getOpcode());
        assertEquals("Wrong instruction", WideInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 8, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.TABLESWITCH, ilist.getOpcode());
        assertEquals("Wrong instruction", TableSwitchInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 9, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.LOOKUPSWITCH, ilist.getOpcode());
        assertEquals("Wrong instruction", LookupSwitchInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", true, ilist.advanceIndex());

        assertEquals("Wrong index", 10, ilist.getIndex());
        assertEquals("Wrong opcode", Opcode.RETURN, ilist.getOpcode());
        assertEquals("Wrong instruction", GenericInstruction.class, ilist.getInstr().getClass());
        assertEquals("Problem in advanceIndex", false, ilist.advanceIndex());
    }

    public void testFindOpcode() {
        InstructionList ilist = new InstructionList(TEST_CODE);

        assertEquals("Should have found ALOAD_0", true, ilist.findOpcode(Opcode.ALOAD_0));
        assertEquals("Wrong index", 0, ilist.getIndex());

        assertEquals("Should have found ALOAD", true, ilist.findOpcode(Opcode.ALOAD));
        assertEquals("Wrong index", 4, ilist.getIndex());
        ilist.advanceIndex();

        assertEquals("Should have not have found ALOAD", false, ilist.findOpcode(Opcode.ALOAD));
        assertEquals("Wrong index", 11, ilist.getIndex());
    }

    public void testFindInstr() {
        InstructionList ilist = new InstructionList(TEST_CODE);
        LineNumberTable lnt = new LineNumberTable(TEST_CODE);

        assertEquals("Should have found ALOAD_0", true, ilist.findInstruction(
            new GenericInstruction(TEST_CODE, (short)0, (short)0, lnt)));
        assertEquals("Wrong index", 0, ilist.getIndex());

        assertEquals("Should have found ALOAD", true, ilist.findInstruction(
            new GenericInstruction(TEST_CODE, (short)12, (short)12, lnt)));
        assertEquals("Wrong index", 4, ilist.getIndex());
        ilist.advanceIndex();

        assertEquals("Should have not found ALOAD", false, ilist.findInstruction(
            new GenericInstruction(TEST_CODE, (short)12, (short)12, lnt)));
        assertEquals("Wrong index", 11, ilist.getIndex());

        ilist.setIndex(0);

        assertEquals("Should have not found ALOAD", false, ilist.findInstruction(
            new GenericInstruction(new byte[]{Opcode.ALOAD, 1}, (short)0, (short)12, lnt)));
        assertEquals("Wrong index", 11, ilist.getIndex());
    }

    public void testDelete() {
        InstructionList ilist = new InstructionList(TEST_CODE_INSERTDELETE);

        ilist.setIndex(2);
        assertEquals("Should not be at end yet", true, ilist.deleteInstr(null));
        assertEquals("At wrong index", 2, ilist.getIndex());
        assertEquals("Wrong result", new InstructionList(TEST_CODE_DELETE_RESULT), ilist);

        ilist.setIndex(3);
        assertEquals("Should be at end", false, ilist.deleteInstr(null));
        assertEquals("At wrong index", 3, ilist.getIndex());
        assertEquals("Wrong result", new InstructionList(TEST_CODE_DELETE_RESULT2), ilist);
    }

    public void testInsert() {
        InstructionList ilist = new InstructionList(TEST_CODE_INSERTDELETE);

        ilist.setIndex(2);
        ilist.insertInstr(new GenericInstruction(new byte[]{Opcode.NOP}), null);
        assertEquals("At wrong index", 2, ilist.getIndex());
        assertEquals("Wrong result", new InstructionList(TEST_CODE_INSERT_RESULT), ilist);

        ilist.insertInstr(new TableSwitchInstruction(2, 5, 7, new int[]{1, 3, 5}), null);
        assertEquals("At wrong index", 2, ilist.getIndex());
        assertEquals("Wrong result", new InstructionList(TEST_CODE_INSERT_RESULT2), ilist);

        // this causes the TABLESWITCH instruction to repad
        ilist.insertInstr(new GenericInstruction(new byte[]{Opcode.NOP}), null);
        assertEquals("At wrong index", 2, ilist.getIndex());
        assertEquals("Wrong result", new InstructionList(TEST_CODE_INSERT_RESULT3), ilist);
    }
}
