package sysModel.classFile.code;

import sysModel.classFile.Types;

/**
 * Java opcode class.
 *
 * @author Mathias Ricken
 */
public abstract class Opcode {
    public static final byte NOP = (byte)0x00;
    public static final byte ACONST_NULL = (byte)0x01;
    public static final byte ICONST_M1 = (byte)0x02;
    public static final byte ICONST_0 = (byte)0x03;
    public static final byte ICONST_1 = (byte)0x04;
    public static final byte ICONST_2 = (byte)0x05;
    public static final byte ICONST_3 = (byte)0x06;
    public static final byte ICONST_4 = (byte)0x07;
    public static final byte ICONST_5 = (byte)0x08;
    public static final byte LCONST_0 = (byte)0x09;
    public static final byte LCONST_1 = (byte)0x0A;
    public static final byte FCONST_0 = (byte)0x0B;
    public static final byte FCONST_1 = (byte)0x0C;
    public static final byte FCONST_2 = (byte)0x0D;
    public static final byte DCONST_0 = (byte)0x0E;
    public static final byte DCONST_1 = (byte)0x0F;
    public static final byte BIPUSH = (byte)0x10;
    public static final byte SIPUSH = (byte)0x11;
    public static final byte LDC = (byte)0x12;
    public static final byte LDC_W = (byte)0x13;
    public static final byte LDC2_W = (byte)0x14;
    public static final byte ILOAD = (byte)0x15;
    public static final byte LLOAD = (byte)0x16;
    public static final byte FLOAD = (byte)0x17;
    public static final byte DLOAD = (byte)0x18;
    public static final byte ALOAD = (byte)0x19;
    public static final byte ILOAD_0 = (byte)0x1A;
    public static final byte ILOAD_1 = (byte)0x1B;
    public static final byte ILOAD_2 = (byte)0x1C;
    public static final byte ILOAD_3 = (byte)0x1D;
    public static final byte LLOAD_0 = (byte)0x1E;
    public static final byte LLOAD_1 = (byte)0x1F;
    public static final byte LLOAD_2 = (byte)0x20;
    public static final byte LLOAD_3 = (byte)0x21;
    public static final byte FLOAD_0 = (byte)0x22;
    public static final byte FLOAD_1 = (byte)0x23;
    public static final byte FLOAD_2 = (byte)0x24;
    public static final byte FLOAD_3 = (byte)0x25;
    public static final byte DLOAD_0 = (byte)0x26;
    public static final byte DLOAD_1 = (byte)0x27;
    public static final byte DLOAD_2 = (byte)0x28;
    public static final byte DLOAD_3 = (byte)0x29;
    public static final byte ALOAD_0 = (byte)0x2A;
    public static final byte ALOAD_1 = (byte)0x2B;
    public static final byte ALOAD_2 = (byte)0x2C;
    public static final byte ALOAD_3 = (byte)0x2D;
    public static final byte IALOAD = (byte)0x2E;
    public static final byte LALOAD = (byte)0x2F;
    public static final byte FALOAD = (byte)0x30;
    public static final byte DALOAD = (byte)0x31;
    public static final byte AALOAD = (byte)0x32;
    public static final byte BALOAD = (byte)0x33;
    public static final byte CALOAD = (byte)0x34;
    public static final byte SALOAD = (byte)0x35;
    public static final byte ISTORE = (byte)0x36;
    public static final byte LSTORE = (byte)0x37;
    public static final byte FSTORE = (byte)0x38;
    public static final byte DSTORE = (byte)0x39;
    public static final byte ASTORE = (byte)0x3A;
    public static final byte ISTORE_0 = (byte)0x3B;
    public static final byte ISTORE_1 = (byte)0x3C;
    public static final byte ISTORE_2 = (byte)0x3D;
    public static final byte ISTORE_3 = (byte)0x3E;
    public static final byte LSTORE_0 = (byte)0x3F;
    public static final byte LSTORE_1 = (byte)0x40;
    public static final byte LSTORE_2 = (byte)0x41;
    public static final byte LSTORE_3 = (byte)0x42;
    public static final byte FSTORE_0 = (byte)0x43;
    public static final byte FSTORE_1 = (byte)0x44;
    public static final byte FSTORE_2 = (byte)0x45;
    public static final byte FSTORE_3 = (byte)0x46;
    public static final byte DSTORE_0 = (byte)0x47;
    public static final byte DSTORE_1 = (byte)0x48;
    public static final byte DSTORE_2 = (byte)0x49;
    public static final byte DSTORE_3 = (byte)0x4A;
    public static final byte ASTORE_0 = (byte)0x4B;
    public static final byte ASTORE_1 = (byte)0x4C;
    public static final byte ASTORE_2 = (byte)0x4D;
    public static final byte ASTORE_3 = (byte)0x4E;
    public static final byte IASTORE = (byte)0x4F;
    public static final byte LASTORE = (byte)0x50;
    public static final byte FASTORE = (byte)0x51;
    public static final byte DASTORE = (byte)0x52;
    public static final byte AASTORE = (byte)0x53;
    public static final byte BASTORE = (byte)0x54;
    public static final byte CASTORE = (byte)0x55;
    public static final byte SASTORE = (byte)0x56;
    public static final byte POP = (byte)0x57;
    public static final byte POP2 = (byte)0x58;
    public static final byte DUP = (byte)0x59;
    public static final byte DUP_X1 = (byte)0x5A;
    public static final byte DUP_X2 = (byte)0x5B;
    public static final byte DUP2 = (byte)0x5C;
    public static final byte DUP2_X1 = (byte)0x5D;
    public static final byte DUP2_X2 = (byte)0x5E;
    public static final byte SWAP = (byte)0x5F;
    public static final byte IADD = (byte)0x60;
    public static final byte LADD = (byte)0x61;
    public static final byte FADD = (byte)0x62;
    public static final byte DADD = (byte)0x63;
    public static final byte ISUB = (byte)0x64;
    public static final byte LSUB = (byte)0x65;
    public static final byte FSUB = (byte)0x66;
    public static final byte DSUB = (byte)0x67;
    public static final byte IMUL = (byte)0x68;
    public static final byte LMUL = (byte)0x69;
    public static final byte FMUL = (byte)0x6A;
    public static final byte DMUL = (byte)0x6B;
    public static final byte IDIV = (byte)0x6C;
    public static final byte LDIV = (byte)0x6D;
    public static final byte FDIV = (byte)0x6E;
    public static final byte DDIV = (byte)0x6F;
    public static final byte IREM = (byte)0x70;
    public static final byte LREM = (byte)0x71;
    public static final byte FREM = (byte)0x72;
    public static final byte DREM = (byte)0x73;
    public static final byte INEG = (byte)0x74;
    public static final byte LNEG = (byte)0x75;
    public static final byte FNEG = (byte)0x76;
    public static final byte DNEG = (byte)0x77;
    public static final byte ISHL = (byte)0x78;
    public static final byte LSHL = (byte)0x79;
    public static final byte ISHR = (byte)0x7A;
    public static final byte LSHR = (byte)0x7B;
    public static final byte IUSHR = (byte)0x7C;
    public static final byte LUSHR = (byte)0x7D;
    public static final byte IAND = (byte)0x7E;
    public static final byte LAND = (byte)0x7F;
    public static final byte IOR = (byte)0x80;
    public static final byte LOR = (byte)0x81;
    public static final byte IXOR = (byte)0x82;
    public static final byte LXOR = (byte)0x83;
    public static final byte IINC = (byte)0x84;
    public static final byte I2L = (byte)0x85;
    public static final byte I2F = (byte)0x86;
    public static final byte I2D = (byte)0x87;
    public static final byte L2I = (byte)0x88;
    public static final byte L2F = (byte)0x89;
    public static final byte L2D = (byte)0x8A;
    public static final byte F2I = (byte)0x8B;
    public static final byte F2L = (byte)0x8C;
    public static final byte F2D = (byte)0x8D;
    public static final byte D2I = (byte)0x8E;
    public static final byte D2L = (byte)0x8F;
    public static final byte D2F = (byte)0x90;
    public static final byte I2B = (byte)0x91;
    public static final byte I2C = (byte)0x92;
    public static final byte I2S = (byte)0x93;
    public static final byte LCMP = (byte)0x94;
    public static final byte FCMPL = (byte)0x95;
    public static final byte FCMPG = (byte)0x96;
    public static final byte DCMPL = (byte)0x97;
    public static final byte DCMPG = (byte)0x98;
    public static final byte IFEQ = (byte)0x99;
    public static final byte IFNE = (byte)0x9A;
    public static final byte IFLT = (byte)0x9B;
    public static final byte IFGE = (byte)0x9C;
    public static final byte IFGT = (byte)0x9D;
    public static final byte IFLE = (byte)0x9E;
    public static final byte IF_ICMPEQ = (byte)0x9F;
    public static final byte IF_ICMPNE = (byte)0xA0;
    public static final byte IF_ICMPLT = (byte)0xA1;
    public static final byte IF_ICMPGE = (byte)0xA2;
    public static final byte IF_ICMPGT = (byte)0xA3;
    public static final byte IF_ICMPLE = (byte)0xA4;
    public static final byte IF_ACMPEQ = (byte)0xA5;
    public static final byte IF_ACMPNE = (byte)0xA6;
    public static final byte GOTO = (byte)0xA7;
    public static final byte JSR = (byte)0xA8;
    public static final byte RET = (byte)0xA9;
    public static final byte TABLESWITCH = (byte)0xAA;
    public static final byte LOOKUPSWITCH = (byte)0xAB;
    public static final byte IRETURN = (byte)0xAC;
    public static final byte LRETURN = (byte)0xAD;
    public static final byte FRETURN = (byte)0xAE;
    public static final byte DRETURN = (byte)0xAF;
    public static final byte ARETURN = (byte)0xB0;
    public static final byte RETURN = (byte)0xB1;
    public static final byte GETSTATIC = (byte)0xB2;
    public static final byte PUTSTATIC = (byte)0xB3;
    public static final byte GETFIELD = (byte)0xB4;
    public static final byte PUTFIELD = (byte)0xB5;
    public static final byte INVOKEVIRTUAL = (byte)0xB6;
    public static final byte INVOKESPECIAL = (byte)0xB7;
    public static final byte INVOKESTATIC = (byte)0xB8;
    public static final byte INVOKEINTERFACE = (byte)0xB9;
    public static final byte XXXUNUSEDXXX = (byte)0xBA;
    public static final byte NEW = (byte)0xBB;
    public static final byte NEWARRAY = (byte)0xBC;
    public static final byte ANEWARRAY = (byte)0xBD;
    public static final byte ARRAYLENGTH = (byte)0xBE;
    public static final byte ATHROW = (byte)0xBF;
    public static final byte CHECKCAST = (byte)0xC0;
    public static final byte INSTANCEOF = (byte)0xC1;
    public static final byte MONITORENTER = (byte)0xC2;
    public static final byte MONITOREXIT = (byte)0xC3;
    public static final byte WIDE = (byte)0xC4;
    public static final byte MULTIANEWARRAY = (byte)0xC5;
    public static final byte IFNULL = (byte)0xC6;
    public static final byte IFNONNULL = (byte)0xC7;
    public static final byte GOTO_W = (byte)0xC8;
    public static final byte JSR_W = (byte)0xC9;
    public static final byte BREAKPOINT = (byte)0xCA;
    public static final byte LDC_QUICK = (byte)0xCB;
    public static final byte LDC_W_QUICK = (byte)0xCC;
    public static final byte LDC2_W_QUICK = (byte)0xCD;
    public static final byte GETFIELD_QUICK = (byte)0xCE;
    public static final byte PUTFIELD_QUICK = (byte)0xCF;
    public static final byte GETFIELD2_QUICK = (byte)0xD0;
    public static final byte PUTFIELD2_QUICK = (byte)0xD1;
    public static final byte GETSTATIC_QUICK = (byte)0xD2;
    public static final byte PUTSTATIC_QUICK = (byte)0xD3;
    public static final byte GETSTATIC2_QUICK = (byte)0xD4;
    public static final byte PUTSTATIC2_QUICK = (byte)0xD5;
    public static final byte INVOKEVIRTUAL_QUICK = (byte)0xD6;
    public static final byte INVOKENONVIRTUAL_QUICK = (byte)0xD7;
    public static final byte INVOKESUPER_QUICK = (byte)0xD8;
    public static final byte INVOKESTATIC_QUICK = (byte)0xD9;
    public static final byte INVOKEINTERFACE_QUICK = (byte)0xDA;
    public static final byte INVOKEVIRTUALOBJECT_QUICK = (byte)0xDB;
    public static final byte UNKNOWN_DC = (byte)0xDC;
    public static final byte NEW_QUICK = (byte)0xDD;
    public static final byte ANEWARRAY_QUICK = (byte)0xDE;
    public static final byte MULTIANEWARRAY_QUICK = (byte)0xDF;
    public static final byte CHECKCAST_QUICK = (byte)0xE0;
    public static final byte INSTANCEOF_QUICK = (byte)0xE1;
    public static final byte INVOKEVIRTUAL_QUICK_W = (byte)0xE2;
    public static final byte GETFIELD_QUICK_W = (byte)0xE3;
    public static final byte PUTFIELD_QUICK_W = (byte)0xE4;
    public static final byte UNKNOWN_E5 = (byte)0xE5;
    public static final byte UNKNOWN_E6 = (byte)0xE6;
    public static final byte UNKNOWN_E7 = (byte)0xE7;
    public static final byte UNKNOWN_E8 = (byte)0xE8;
    public static final byte UNKNOWN_E9 = (byte)0xE9;
    public static final byte UNKNOWN_EA = (byte)0xEA;
    public static final byte UNKNOWN_EB = (byte)0xEB;
    public static final byte UNKNOWN_EC = (byte)0xEC;
    public static final byte UNKNOWN_ED = (byte)0xED;
    public static final byte UNKNOWN_EE = (byte)0xEE;
    public static final byte UNKNOWN_EF = (byte)0xEF;
    public static final byte UNKNOWN_F0 = (byte)0xF0;
    public static final byte UNKNOWN_F1 = (byte)0xF1;
    public static final byte UNKNOWN_F2 = (byte)0xF2;
    public static final byte UNKNOWN_F3 = (byte)0xF3;
    public static final byte UNKNOWN_F4 = (byte)0xF4;
    public static final byte UNKNOWN_F5 = (byte)0xF5;
    public static final byte UNKNOWN_F6 = (byte)0xF6;
    public static final byte UNKNOWN_F7 = (byte)0xF7;
    public static final byte UNKNOWN_F8 = (byte)0xF8;
    public static final byte UNKNOWN_F9 = (byte)0xF9;
    public static final byte UNKNOWN_FA = (byte)0xFA;
    public static final byte UNKNOWN_FB = (byte)0xFB;
    public static final byte UNKNOWN_FC = (byte)0xFC;
    public static final byte UNKNOWN_FD = (byte)0xFD;
    public static final byte IMPDEP1 = (byte)0xFE;
    public static final byte IMPDEP2 = (byte)0xFF;

    /**
     * Return a human-readable version of the code.
     *
     * @param opcode code
     *
     * @return mnemonic
     */
    public static final String getOpcodeName(byte opcode) {
        return NAMES[((int)opcode) & 0xFF];
    }

    /**
     * Table of mnemonics.
     */
    private static final String[] NAMES = new String[]{
        /* 0x00 */ "nop",
        /* 0x01 */ "aconst_null",
        /* 0x02 */ "iconst_m1",
        /* 0x03 */ "iconst_0",
        /* 0x04 */ "iconst_1",
        /* 0x05 */ "iconst_2",
        /* 0x06 */ "iconst_3",
        /* 0x07 */ "iconst_4",
        /* 0x08 */ "iconst_5",
        /* 0x09 */ "lconst_0",
        /* 0x0A */ "lconst_1",
        /* 0x0B */ "fconst_0",
        /* 0x0C */ "fconst_1",
        /* 0x0D */ "fconst_2",
        /* 0x0E */ "dconst_0",
        /* 0x0F */ "dconst_1",
        /* 0x10 */ "bipush",
        /* 0x11 */ "sipush",
        /* 0x12 */ "ldc",
        /* 0x13 */ "ldc_w",
        /* 0x14 */ "ldc2_w",
        /* 0x15 */ "iload",
        /* 0x16 */ "lload",
        /* 0x17 */ "fload",
        /* 0x18 */ "dload",
        /* 0x19 */ "aload",
        /* 0x1A */ "iload_0",
        /* 0x1B */ "iload_1",
        /* 0x1C */ "iload_2",
        /* 0x1D */ "iload_3",
        /* 0x1E */ "lload_0",
        /* 0x1F */ "lload_1",
        /* 0x20 */ "lload_2",
        /* 0x21 */ "lload_3",
        /* 0x22 */ "fload_0",
        /* 0x23 */ "fload_1",
        /* 0x24 */ "fload_2",
        /* 0x25 */ "fload_3",
        /* 0x26 */ "dload_0",
        /* 0x27 */ "dload_1",
        /* 0x28 */ "dload_2",
        /* 0x29 */ "dload_3",
        /* 0x2A */ "aload_0",
        /* 0x2B */ "aload_1",
        /* 0x2C */ "aload_2",
        /* 0x2D */ "aload_3",
        /* 0x2E */ "iaload",
        /* 0x2F */ "laload",
        /* 0x30 */ "faload",
        /* 0x31 */ "daload",
        /* 0x32 */ "aaload",
        /* 0x33 */ "baload",
        /* 0x34 */ "caload",
        /* 0x35 */ "saload",
        /* 0x36 */ "istore",
        /* 0x37 */ "lstore",
        /* 0x38 */ "fstore",
        /* 0x39 */ "dstore",
        /* 0x3A */ "astore",
        /* 0x3B */ "istore_0",
        /* 0x3C */ "istore_1",
        /* 0x3D */ "istore_2",
        /* 0x3E */ "istore_3",
        /* 0x3F */ "lstore_0",
        /* 0x40 */ "lstore_1",
        /* 0x41 */ "lstore_2",
        /* 0x42 */ "lstore_3",
        /* 0x43 */ "fstore_0",
        /* 0x44 */ "fstore_1",
        /* 0x45 */ "fstore_2",
        /* 0x46 */ "fstore_3",
        /* 0x47 */ "dstore_0",
        /* 0x48 */ "dstore_1",
        /* 0x49 */ "dstore_2",
        /* 0x4A */ "dstore_3",
        /* 0x4B */ "astore_0",
        /* 0x4C */ "astore_1",
        /* 0x4D */ "astore_2",
        /* 0x4E */ "astore_3",
        /* 0x4F */ "iastore",
        /* 0x50 */ "lastore",
        /* 0x51 */ "fastore",
        /* 0x52 */ "dastore",
        /* 0x53 */ "aastore",
        /* 0x54 */ "bastore",
        /* 0x55 */ "castore",
        /* 0x56 */ "sastore",
        /* 0x57 */ "pop",
        /* 0x58 */ "pop2",
        /* 0x59 */ "dup",
        /* 0x5A */ "dup_x1",
        /* 0x5B */ "dup_x2",
        /* 0x5C */ "dup2",
        /* 0x5D */ "dup2_x1",
        /* 0x5E */ "dup2_x2",
        /* 0x5F */ "swap",
        /* 0x60 */ "iadd",
        /* 0x61 */ "ladd",
        /* 0x62 */ "fadd",
        /* 0x63 */ "dadd",
        /* 0x64 */ "isub",
        /* 0x65 */ "lsub",
        /* 0x66 */ "fsub",
        /* 0x67 */ "dsub",
        /* 0x68 */ "imul",
        /* 0x69 */ "lmul",
        /* 0x6A */ "fmul",
        /* 0x6B */ "dmul",
        /* 0x6C */ "idiv",
        /* 0x6D */ "ldiv",
        /* 0x6E */ "fdiv",
        /* 0x6F */ "ddiv",
        /* 0x70 */ "irem",
        /* 0x71 */ "lrem",
        /* 0x72 */ "frem",
        /* 0x73 */ "drem",
        /* 0x74 */ "ineg",
        /* 0x75 */ "lneg",
        /* 0x76 */ "fneg",
        /* 0x77 */ "dneg",
        /* 0x78 */ "ishl",
        /* 0x79 */ "lshl",
        /* 0x7A */ "ishr",
        /* 0x7B */ "lshr",
        /* 0x7C */ "iushr",
        /* 0x7D */ "lushr",
        /* 0x7E */ "iand",
        /* 0x7F */ "land",
        /* 0x80 */ "ior",
        /* 0x81 */ "lor",
        /* 0x82 */ "ixor",
        /* 0x83 */ "lxor",
        /* 0x84 */ "iinc",
        /* 0x85 */ "i2l",
        /* 0x86 */ "i2f",
        /* 0x87 */ "i2d",
        /* 0x88 */ "l2i",
        /* 0x89 */ "l2f",
        /* 0x8A */ "l2d",
        /* 0x8B */ "f2i",
        /* 0x8C */ "f2l",
        /* 0x8D */ "f2d",
        /* 0x8E */ "d2i",
        /* 0x8F */ "d2l",
        /* 0x90 */ "d2f",
        /* 0x91 */ "i2b",
        /* 0x92 */ "i2c",
        /* 0x93 */ "i2s",
        /* 0x94 */ "lcmp",
        /* 0x95 */ "fcmpl",
        /* 0x96 */ "fcmpg",
        /* 0x97 */ "dcmpl",
        /* 0x98 */ "dcmpg",
        /* 0x99 */ "ifeq",
        /* 0x9A */ "ifne",
        /* 0x9B */ "iflt",
        /* 0x9C */ "ifge",
        /* 0x9D */ "ifgt",
        /* 0x9E */ "ifle",
        /* 0x9F */ "if_icmpeq",
        /* 0xA0 */ "if_icmpne",
        /* 0xA1 */ "if_icmplt",
        /* 0xA2 */ "if_icmpge",
        /* 0xA3 */ "if_icmpgt",
        /* 0xA4 */ "if_icmple",
        /* 0xA5 */ "if_acmpeq",
        /* 0xA6 */ "if_acmpne",
        /* 0xA7 */ "goto",
        /* 0xA8 */ "jsr",
        /* 0xA9 */ "ret",
        /* 0xAA */ "tableswitch",
        /* 0xAB */ "lookupswitch",
        /* 0xAC */ "ireturn",
        /* 0xAD */ "lreturn",
        /* 0xAE */ "freturn",
        /* 0xAF */ "dreturn",
        /* 0xB0 */ "areturn",
        /* 0xB1 */ "return",
        /* 0xB2 */ "getstatic",
        /* 0xB3 */ "putstatic",
        /* 0xB4 */ "getfield",
        /* 0xB5 */ "putfield",
        /* 0xB6 */ "invokevirtual",
        /* 0xB7 */ "invokespecial",
        /* 0xB8 */ "invokestatic",
        /* 0xB9 */ "invokeinterface",
        /* 0xBA */ "xxxunusedxxx",
        /* 0xBB */ "new",
        /* 0xBC */ "newarray",
        /* 0xBD */ "anewarray",
        /* 0xBE */ "arraylength",
        /* 0xBF */ "athrow",
        /* 0xC0 */ "checkcast",
        /* 0xC1 */ "instanceof",
        /* 0xC2 */ "monitorenter",
        /* 0xC3 */ "monitorexit",
        /* 0xC4 */ "wide",
        /* 0xC5 */ "multianewarray",
        /* 0xC6 */ "ifnull",
        /* 0xC7 */ "ifnonnull",
        /* 0xC8 */ "goto_w",
        /* 0xC9 */ "jsr_w",
        /* 0xCA */ "breakpoint",
        /* 0xCB */ "ldc_quick",
        /* 0xCC */ "ldc_w_quick",
        /* 0xCD */ "ldc2_w_quick",
        /* 0xCE */ "getfield_quick",
        /* 0xCF */ "putfield_quick",
        /* 0xD0 */ "getfield2_quick",
        /* 0xD1 */ "putfield2_quick",
        /* 0xD2 */ "getstatic_quick",
        /* 0xD3 */ "putstatic_quick",
        /* 0xD4 */ "getstatic2_quick",
        /* 0xD5 */ "putstatic2_quick",
        /* 0xD6 */ "invokevirtual_quick",
        /* 0xD7 */ "invokenonvirtual_quick",
        /* 0xD8 */ "invokesuper_quick",
        /* 0xD9 */ "invokestatic_quick",
        /* 0xDA */ "invokeinterface_quick",
        /* 0xDB */ "invokevirtualobject_quick",
        /* 0xDC */ "unknown_dc",
        /* 0xDD */ "new_quick",
        /* 0xDE */ "anewarray_quick",
        /* 0xDF */ "multianewarray_quick",
        /* 0xE0 */ "checkcast_quick",
        /* 0xE1 */ "instanceof_quick",
        /* 0xE2 */ "invokevirtual_quick_w",
        /* 0xE3 */ "getfield_quick_w",
        /* 0xE4 */ "putfield_quick_w",
        /* 0xE5 */ "unknown_e5",
        /* 0xE6 */ "unknown_e6",
        /* 0xE7 */ "unknown_e7",
        /* 0xE8 */ "unknown_e8",
        /* 0xE9 */ "unknown_e9",
        /* 0xEA */ "unknown_ea",
        /* 0xEB */ "unknown_eb",
        /* 0xEC */ "unknown_ec",
        /* 0xED */ "unknown_ed",
        /* 0xEE */ "unknown_ee",
        /* 0xEF */ "unknown_ef",
        /* 0xF0 */ "unknown_f0",
        /* 0xF1 */ "unknown_f1",
        /* 0xF2 */ "unknown_f2",
        /* 0xF3 */ "unknown_f3",
        /* 0xF4 */ "unknown_f4",
        /* 0xF5 */ "unknown_f5",
        /* 0xF6 */ "unknown_f6",
        /* 0xF7 */ "unknown_f7",
        /* 0xF8 */ "unknown_f8",
        /* 0xF9 */ "unknown_f9",
        /* 0xFA */ "unknown_fa",
        /* 0xFB */ "unknown_fb",
        /* 0xFC */ "unknown_fc",
        /* 0xFD */ "unknown_fd",
        /* 0xFE */ "impdep1",
        /* 0xFF */ "impdep2"
    };

    /**
     * Table with code offsets.
     */
    private static final byte[] OFFSET = new byte[256];

    static {
        for(int i = 0; i < OFFSET.length; ++i) {
            OFFSET[i] = 0;
        }
        OFFSET[Types.unsigned(BIPUSH)] = 1;
        OFFSET[Types.unsigned(SIPUSH)] = 2;
        OFFSET[Types.unsigned(LDC)] = 1;
        OFFSET[Types.unsigned(LDC_W)] = 2;
        OFFSET[Types.unsigned(LDC2_W)] = 2;
        for(int i = Types.unsigned(ILOAD); i <= Types.unsigned(ALOAD); ++i) {
            OFFSET[i] = 1;
        }
        for(int i = Types.unsigned(ISTORE); i <= Types.unsigned(ASTORE); ++i) {
            OFFSET[i] = 1;
        }
        OFFSET[Types.unsigned(IINC)] = 2;
        for(int i = Types.unsigned(IFEQ); i <= Types.unsigned(JSR); i++) {
            OFFSET[i] = 2;
        }
        OFFSET[Types.unsigned(RET)] = 1;
        // TABLESWITCH and LOOKUPSWITCH are highly irregular.
        for(int i = Types.unsigned(GETSTATIC); i <= Types.unsigned(INVOKESTATIC); ++i) {
            OFFSET[i] = 2;
        }
        OFFSET[Types.unsigned(INVOKEINTERFACE)] = 4;
        OFFSET[Types.unsigned(NEW)] = 2;
        OFFSET[Types.unsigned(NEWARRAY)] = 1;
        OFFSET[Types.unsigned(ANEWARRAY)] = 2;
        OFFSET[Types.unsigned(CHECKCAST)] = 2;
        OFFSET[Types.unsigned(INSTANCEOF)] = 2;
        // WIDE is irregular.
        OFFSET[Types.unsigned(MULTIANEWARRAY)] = 3;
        OFFSET[Types.unsigned(IFNULL)] = 2;
        OFFSET[Types.unsigned(IFNONNULL)] = 2;
        OFFSET[Types.unsigned(GOTO_W)] = 4;
        OFFSET[Types.unsigned(JSR_W)] = 4;
    }

    /**
     * Return the length of the instruction at the specified offset in the bytecode array. This includes the code itself
     * and its operands.
     *
     * @param code the bytecode array.
     * @param pc   offset
     *
     * @return the length of the code and operands
     */
    public static final int getInstrSize(byte[] code, int pc) {
        return getInstrSize(code, pc, pc);
    }

    /**
     * Return the length of the instruction at the specified offset in the bytecode array. This includes the code itself
     * and its operands.
     *
     * @param code      the bytecode array.
     * @param pc        offset
     * @param paddingPC offset for calculating padding
     *
     * @return the length of the code and operands
     */
    public static final int getInstrSize(byte[] code, int pc, int paddingPC) {
        byte opcode = code[pc];
        switch(opcode) {
            case LOOKUPSWITCH:
                {
                    int pad = 3 - (paddingPC % 4);
                    long npairs = Types.intFromBytes(code, pc + pad + 5);
                    assert 0 <= npairs;
                    return (int)(npairs * 8) + pad + 9;
                }
            case TABLESWITCH:
                {
                    int pad = 3 - (paddingPC % 4);
                    long low = Types.intFromBytes(code, pc + pad + 5);
                    long high = Types.intFromBytes(code, pc + pad + 9);
                    long npairs = high - low + 1;
                    assert low <= high;
                    return (int)(npairs * 4) + pad + 13;
                }
            case WIDE:
                if (IINC == code[pc + 1]) {
                    return 6;
                }
                return 4;
            default:
                return 1 + OFFSET[Types.unsigned(opcode)];
        }
    }

    /**
     * Change the padding on the inside of the instruction found at pc, so that it is padded for a PC=newPC, not for a
     * PC=paddingPC.
     *
     * @param code      bytecode
     * @param pc        start of instruction in bytecode array
     * @param paddingPC old PC used for padding
     * @param newPC     new PC used for padding
     *
     * @return repadded bytecode array
     */
    public static final byte[] repadInstr(byte[] code, int pc, int paddingPC, int newPC) {
        byte opcode = code[pc];
        switch(opcode) {
            case LOOKUPSWITCH:
                {
                    int pad = 3 - (paddingPC % 4);
                    int newPad = 3 - (newPC % 4);
                    byte[] newCode = new byte[code.length - pad + newPad];
                    System.arraycopy(code, 0, newCode, 0, pc + 1);
                    System.arraycopy(code, pc + pad + 1, newCode, pc + newPad + 1, code.length - pc - pad - 1);

                    return newCode;
                }
            case TABLESWITCH:
                {
                    int pad = 3 - (paddingPC % 4);
                    int newPad = 3 - (newPC % 4);
                    byte[] newCode = new byte[code.length - pad + newPad];
                    System.arraycopy(code, 0, newCode, 0, pc + 1);
                    System.arraycopy(code, pc + pad + 1, newCode, pc + newPad + 1, code.length - pc - pad - 1);

                    return newCode;
                }
            default:
                byte[] newCode = new byte[code.length];
                System.arraycopy(code, 0, newCode, 0, code.length);

                return newCode;
        }
    }

    /**
     * Return true if the specified code is a return instruction.
     *
     * NOTE: ATHROW is not a return instruction. It does not leave immediately if there are matching exception handlers.
     *
     * @param opcode code to check
     *
     * @return true if the code is a return instruction
     */
    public static final boolean isReturn(byte opcode) {
        switch(opcode) {
            case IRETURN:
            case LRETURN:
            case FRETURN:
            case DRETURN:
            case ARETURN:
            case RETURN:
                return true;
            default:
                return false;
        }
    }

    /**
     * Table determining if an code is a branch instruction.
     */
    private static final boolean[] isBranch = new boolean[256];

    static {
        for(int i = 0; i < isBranch.length; ++i) {
            isBranch[i] = false;
        }
        for(int i = Types.unsigned(IFEQ); i <= Types.unsigned(RETURN); i++) {
            isBranch[i] = true;
        }
        for(int i = Types.unsigned(IFNULL); i <= Types.unsigned(JSR_W); i++) {
            isBranch[i] = true;
        }
        isBranch[Types.unsigned(ATHROW)] = true;
    }

    /**
     * Return true if the specified code is a branch instruction.
     *
     * @param opcode code to check
     *
     * @return true if the code is a branch instruction
     */
    public static final boolean isBranch(byte opcode) {
        return isBranch[Types.unsigned(opcode)];
    }

    /**
     * Return true if the specified code is an unconditional branch instruction.
     *
     * @param opcode code to check
     *
     * @return true if the code is an unconditional branch instruction
     */
    public static final boolean isUnconditionalBranch(byte opcode) {
        switch(opcode) {
            case GOTO:
            case GOTO_W:
            case JSR:
            case JSR_W:
            case RET:
            case IRETURN:
            case LRETURN:
            case FRETURN:
            case DRETURN:
            case ARETURN:
            case RETURN:
            case ATHROW:
            case LOOKUPSWITCH:
            case TABLESWITCH:
                return true;
            default:
                return false;
        }
    }

    /**
     * Return true if the specified code is a subroutine call.
     *
     * @param opcode code to check
     *
     * @return true if the code is a JSR or JSR_W instruction
     */
    public static final boolean isJSR(byte opcode) {
        return ((JSR == opcode) || (JSR_W == opcode));
    }

    /**
     * Returns an array of branch targets for the instruction.
     *
     * @param code      byte code
     * @param pc        program counter
     * @param paddingPC
     *
     * @return array of branch targets
     */
    public static final int[] getBranchTargets(byte[] code, int pc, int paddingPC) {
        if (!isBranch(code[pc])) {
            return new int[0];
        }

        switch(code[pc]) {
            case LOOKUPSWITCH:
                {
                    int pad = 3 - (paddingPC % 4);
                    long deflt = Types.intFromBytes(code, pc + pad + 1);
                    long npairs = Types.intFromBytes(code, pc + pad + 5);
                    assert 0 <= npairs;

                    int[] result = new int[(int)npairs + 1];
                    result[0] = pc + ((int)deflt);
                    for(int i = 0; i < npairs; i++) {
                        long offset = Types.intFromBytes(code, pc + pad + 9 + 4 + (8 * i));
                        result[i + 1] = pc + ((int)offset);
                    }
                    return result;
                }

            case TABLESWITCH:
                {
                    int pad = 3 - (paddingPC % 4);
                    long deflt = Types.intFromBytes(code, pc + pad + 1);
                    long low = Types.intFromBytes(code, pc + pad + 5);
                    long high = Types.intFromBytes(code, pc + pad + 9);
                    long npairs = high - low + 1;
                    assert low <= high;

                    int[] result = new int[(int)npairs + 1];
                    result[0] = pc + ((int)deflt);
                    for(int i = 0; i < npairs; i++) {
                        long offset = Types.intFromBytes(code, pc + pad + 13 + (4 * i));
                        result[i + 1] = pc + ((int)offset);
                    }
                    return result;
                }

            case GOTO_W:
            case JSR_W:
                {
                    long offset = Types.intFromBytes(code, pc + 1);
                    int[] result = new int[1];
                    result[0] = pc + (int)offset;
                    return result;

                }

            case RET:
            case RETURN:
            case IRETURN:
            case LRETURN:
            case FRETURN:
            case DRETURN:
            case ARETURN:
            case ATHROW:
                {
                    return new int[0];
                }

            default:
                {
                    short offset = Types.shortFromBytes(code, pc + 1);
                    int[] result = new int[1];
                    result[0] = pc + offset;
                    return result;

                }
        }
    }


    /**
     * Sets the branch targets for an instruction.
     *
     * NOTE: The number of branch targets has to remain unchanged.
     *
     * @param code          byte code
     * @param pc            program counter
     * @param branchTargets array of branch targets
     */
    public static final void setBranchTargets(byte[] code, int pc, int[] branchTargets) {
        if (!isBranch(code[pc])) {
            throw new IllegalArgumentException("Cannot set branch targets for non-branch instruction");
        }

        switch(code[pc]) {
            case LOOKUPSWITCH:
                {
                    int pad = 3 - (pc % 4);
                    long npairs = Types.intFromBytes(code, pc + pad + 5);
                    assert 0 <= npairs;

                    if (branchTargets.length != npairs + 1) {
                        throw new IllegalArgumentException("Not allowed to change number of branch targets");
                    }

                    int delta = branchTargets[0] - pc;
                    Types.bytesFromInt(delta, code, pc + pad + 1);
                    for(int i = 0; i < npairs; i++) {
                        delta = branchTargets[1 + i] - pc;
                        Types.bytesFromInt(delta, code, pc + pad + 9 + 4 + (8 * i));
                    }
                    return;
                }

            case TABLESWITCH:
                {
                    int pad = 3 - (pc % 4);
                    long low = Types.intFromBytes(code, pc + pad + 5);
                    long high = Types.intFromBytes(code, pc + pad + 9);
                    long npairs = high - low + 1;
                    assert low <= high;

                    if (branchTargets.length != npairs + 1) {
                        throw new IllegalArgumentException("Not allowed to change number of branch targets");
                    }

                    int delta = branchTargets[0] - pc;
                    Types.bytesFromInt(delta, code, pc + pad + 1);
                    for(int i = 0; i < npairs; i++) {
                        delta = branchTargets[1 + i] - pc;
                        Types.bytesFromInt(delta, code, pc + pad + 13 + (4 * i));
                    }
                    return;
                }

            case GOTO_W:
            case JSR_W:
                {
                    if (1 != branchTargets.length) {
                        throw new IllegalArgumentException("Not allowed to change number of branch targets");
                    }
                    int delta = branchTargets[0] - pc;
                    Types.bytesFromInt(delta, code, pc + 1);
                    return;

                }

            case RET:
            case RETURN:
            case IRETURN:
            case LRETURN:
            case FRETURN:
            case DRETURN:
            case ARETURN:
            case ATHROW:
                {
                    if (0 != branchTargets.length) {
                        throw new IllegalArgumentException("Not allowed to change number of branch targets");
                    }
                    return;
                }

            default:
                {
                    if (1 != branchTargets.length) {
                        throw new IllegalArgumentException("Not allowed to change number of branch targets");
                    }
                    short delta = (short)(branchTargets[0] - pc);
                    Types.bytesFromShort(delta, code, pc + 1);
                    return;

                }
        }
    }
}
