package sysModel.classFile.code.instructions;

import sysModel.classFile.code.Opcode;

/**
 * Abstract Java instruction.
 *
 * @author Mathias Ricken
 */
public abstract class AInstruction {
    /**
     * Get the bytecode for this instruction, padded for the specified program counter value, with branch targets
     * according to the specified line number table
     *
     * @param pc  PC for padding
     * @param lnt line number table for branches
     *
     * @return bytecode
     */
    public abstract byte[] getBytecode(short pc, LineNumberTable lnt);

    /**
     * Get the opcode of this instruction.
     *
     * @return opcode
     */
    public abstract byte getOpcode();

    /**
     * Get the length bytecode for this instruction, padded for the specified program counter value.
     *
     * @param pc PC for padding
     *
     * @return bytecode length
     */
    public abstract short getBytecodeLength(short pc);

    /**
     * Return an array of target indices.
     *
     * @return array of target indices.
     */
    public abstract short[] getBranchTargets();

    /**
     * Set the branch target indices.
     *
     * @param branchTargets array of target indices
     */
    public abstract void setBranchTargets(short[] branchTargets);

    /**
     * Make an instruction from the bytecode starting at pc, using the specified padding PC and the line number table.
     *
     * @param bytecode  bytecode
     * @param pc        start PC in bytecode
     * @param paddingPC PC used for padding
     * @param lnt       line number table for branches
     *
     * @return instruction
     */
    public static AInstruction makeInstruction(byte[] bytecode, short pc, short paddingPC, LineNumberTable lnt) {
        byte o = bytecode[pc];
        switch(o) {
            case Opcode.TABLESWITCH:
                return new TableSwitchInstruction(bytecode, pc, paddingPC, lnt);

            case Opcode.LOOKUPSWITCH:
                return new LookupSwitchInstruction(bytecode, pc, paddingPC, lnt);

            case Opcode.IFEQ:
            case Opcode.IFNE:
            case Opcode.IFLT:
            case Opcode.IFGE:
            case Opcode.IFGT:
            case Opcode.IFLE:
            case Opcode.IF_ICMPEQ:
            case Opcode.IF_ICMPNE:
            case Opcode.IF_ICMPLT:
            case Opcode.IF_ICMPGE:
            case Opcode.IF_ICMPGT:
            case Opcode.IF_ICMPLE:
            case Opcode.IF_ACMPEQ:
            case Opcode.IF_ACMPNE:
            case Opcode.GOTO:
            case Opcode.JSR:
            case Opcode.IFNULL:
            case Opcode.IFNONNULL:
                return new BranchInstruction(bytecode, pc, paddingPC, lnt);

            case Opcode.GOTO_W:
            case Opcode.JSR_W:
                return new WideBranchInstruction(bytecode, pc, paddingPC, lnt);

            case Opcode.WIDE:
                return new WideInstruction(bytecode, pc, paddingPC, lnt);

            case Opcode.INVOKESTATIC:
            case Opcode.INVOKESPECIAL:
            case Opcode.INVOKEVIRTUAL:
            case Opcode.CHECKCAST:
            case Opcode.GETFIELD:
            case Opcode.GETSTATIC:
            case Opcode.INSTANCEOF:
            case Opcode.NEW:
            case Opcode.PUTFIELD:
            case Opcode.PUTSTATIC:
            case Opcode.ANEWARRAY:
            case Opcode.NEWARRAY:
            case Opcode.LDC_W:
            case Opcode.LDC2_W:
                return new ReferenceInstruction(bytecode, pc, paddingPC, lnt);

            default:
                return new GenericInstruction(bytecode, pc, paddingPC, lnt);
        }
    }

    /**
     * Return true of this instruction and the other object are equal.
     *
     * @param o other object
     *
     * @return true if equal
     */
    public abstract boolean equals(Object o);

    /**
     * Return hash code.
     *
     * @return hash code
     */
    public abstract int hashCode();

    /**
     * Returns a store instruction that corresponds to the given load instruction.
     * @param loadInstruction load instruction
     * @return corresponding store instruction
     */
    public static AInstruction getCorrespondingStore(AInstruction loadInstruction) {
        byte opcode = loadInstruction.getOpcode();
        if (Opcode.WIDE == opcode) {
            WideInstruction wi = (WideInstruction)loadInstruction;
            byte[] code = wi.getBytecode();
            byte opcode2 = code[1];
            if ((Opcode.ILOAD <= opcode2) && (Opcode.ALOAD >= opcode2)) {
                // WIDE ILOAD 0x15 -> WIDE ISTORE 0x36;
                // WIDE LLOAD 0x16 -> WIDE LSTORE 0x37;
                // WIDE FLOAD 0x17 -> WIDE FSTORE 0x38;
                // WIDE DLOAD 0x18 -> WIDE DSTORE 0x39;
                // WIDE ALOAD 0x19 -> WIDE ASTORE 0x3A;
                return new WideInstruction(new byte[] {(byte)(opcode2 + (Opcode.ISTORE-Opcode.ILOAD)), code[2], code[3]});
            }
        }
        else if ((Opcode.ILOAD <= opcode) && (Opcode.ALOAD >= opcode)) {
            // ILOAD 0x15 -> ISTORE 0x36;
            // LLOAD 0x16 -> LSTORE 0x37;
            // FLOAD 0x17 -> FSTORE 0x38;
            // DLOAD 0x18 -> DSTORE 0x39;
            // ALOAD 0x19 -> ASTORE 0x3A;
            GenericInstruction gi = (GenericInstruction)loadInstruction;
            byte[] code = gi.getBytecode();
            return new GenericInstruction(new byte[] {(byte)(opcode + (Opcode.ISTORE-Opcode.ILOAD)), code[1]});
        }
        else if ((Opcode.ILOAD_0 <= opcode) && (Opcode.ALOAD_3 >= opcode)) {
            // ILOAD_0 0x1A -> ISTORE_0 0x3B;
            // ILOAD_1 0x1B -> ISTORE_1 0x3C;
            // ILOAD_2 0x1C -> ISTORE_2 0x3D;
            // ILOAD_3 0x1D -> ISTORE_3 0x3E;
            // LLOAD_0 0x1E -> LSTORE_0 0x3F;
            // LLOAD_1 0x1F -> LSTORE_1 0x40;
            // LLOAD_2 0x20 -> LSTORE_2 0x41;
            // LLOAD_3 0x21 -> LSTORE_3 0x42;
            // FLOAD_0 0x22 -> FSTORE_0 0x43;
            // FLOAD_1 0x23 -> FSTORE_1 0x44;
            // FLOAD_2 0x24 -> FSTORE_2 0x45;
            // FLOAD_3 0x25 -> FSTORE_3 0x46;
            // DLOAD_0 0x26 -> DSTORE_0 0x47;
            // DLOAD_1 0x27 -> DSTORE_1 0x48;
            // DLOAD_2 0x28 -> DSTORE_2 0x49;
            // DLOAD_3 0x29 -> DSTORE_3 0x4A;
            // ALOAD_0 0x2A -> ASTORE_0 0x4B;
            // ALOAD_1 0x2B -> ASTORE_1 0x4C;
            // ALOAD_2 0x2C -> ASTORE_2 0x4D;
            // ALOAD_3 0x2D -> ASTORE_3 0x4E;
            return new GenericInstruction(new byte[] {(byte)(opcode + (Opcode.ISTORE_0-Opcode.ILOAD_0))});
        }
        else if ((Opcode.IALOAD <= opcode) && (Opcode.SALOAD >= opcode)) {
            // IALOAD 0x2E -> IASTORE 0x4F;
            // LALOAD 0x2F -> LASTORE 0x50;
            // FALOAD 0x30 -> FASTORE 0x51;
            // DALOAD 0x31 -> DASTORE 0x52;
            // AALOAD 0x32 -> AASTORE 0x53;
            // BALOAD 0x33 -> BASTORE 0x54;
            // CALOAD 0x34 -> CASTORE 0x55;
            // SALOAD 0x35 -> SASTORE 0x56;
            GenericInstruction gi = (GenericInstruction)loadInstruction;
            byte[] code = gi.getBytecode();
            return new GenericInstruction(new byte[] {(byte)(opcode + (Opcode.ISTORE-Opcode.ILOAD))});
        }
        throw new IllegalArgumentException("Invalid load instruction");
    }

    /**
     * Returns a load instruction that corresponds to the given store instruction.
     * @param storeInstruction store instruction
     * @return corresponding load instruction
     */
    public static AInstruction getCorrespondingLoad(AInstruction storeInstruction) {
        byte opcode = storeInstruction.getOpcode();
        if (Opcode.WIDE == opcode) {
            WideInstruction wi = (WideInstruction)storeInstruction;
            byte[] code = wi.getBytecode();
            byte opcode2 = code[1];
            if ((Opcode.ISTORE <= opcode2) && (Opcode.ASTORE >= opcode2)) {
                // WIDE ISTORE 0x36 -> WIDE ILOAD 0x15;
                // WIDE LSTORE 0x37 -> WIDE LLOAD 0x16;
                // WIDE FSTORE 0x38 -> WIDE FLOAD 0x17;
                // WIDE DSTORE 0x39 -> WIDE DLOAD 0x18;
                // WIDE ASTORE 0x3A -> WIDE ALOAD 0x19;
                return new WideInstruction(new byte[] {(byte)(opcode2 - (Opcode.ISTORE-Opcode.ILOAD)), code[2], code[3]});
            }
        }
        else if ((Opcode.ISTORE <= opcode) && (Opcode.ASTORE >= opcode)) {
            // ISTORE 0x36 -> ILOAD 0x15;
            // LSTORE 0x37 -> LLOAD 0x16;
            // FSTORE 0x38 -> FLOAD 0x17;
            // DSTORE 0x39 -> DLOAD 0x18;
            // ASTORE 0x3A -> ALOAD 0x19;
            GenericInstruction gi = (GenericInstruction)storeInstruction;
            byte[] code = gi.getBytecode();
            return new GenericInstruction(new byte[] {(byte)(opcode - (Opcode.ISTORE-Opcode.ILOAD)), code[1]});
        }
        else if ((Opcode.ISTORE_0 <= opcode) && (Opcode.ASTORE_3 >= opcode)) {
            // ISTORE_0 0x3B -> ILOAD_0 0x1A;
            // ISTORE_1 0x3C -> ILOAD_1 0x1B;
            // ISTORE_2 0x3D -> ILOAD_2 0x1C;
            // ISTORE_3 0x3E -> ILOAD_3 0x1D;
            // LSTORE_0 0x3F -> LLOAD_0 0x1E;
            // LSTORE_1 0x40 -> LLOAD_1 0x1F;
            // LSTORE_2 0x41 -> LLOAD_2 0x20;
            // LSTORE_3 0x42 -> LLOAD_3 0x21;
            // FSTORE_0 0x43 -> FLOAD_0 0x22;
            // FSTORE_1 0x44 -> FLOAD_1 0x23;
            // FSTORE_2 0x45 -> FLOAD_2 0x24;
            // FSTORE_3 0x46 -> FLOAD_3 0x25;
            // DSTORE_0 0x47 -> DLOAD_0 0x26;
            // DSTORE_1 0x48 -> DLOAD_1 0x27;
            // DSTORE_2 0x49 -> DLOAD_2 0x28;
            // DSTORE_3 0x4A -> DLOAD_3 0x29;
            // ASTORE_0 0x4B -> ALOAD_0 0x2A;
            // ASTORE_1 0x4C -> ALOAD_1 0x2B;
            // ASTORE_2 0x4D -> ALOAD_2 0x2C;
            // ASTORE_3 0x4E -> ALOAD_3 0x2D;
            return new GenericInstruction(new byte[] {(byte)(opcode - (Opcode.ISTORE_0-Opcode.ILOAD_0))});
        }
        else if ((Opcode.IASTORE <= opcode) && (Opcode.SASTORE >= opcode)) {
            // IASTORE 0x4F -> IALOAD 0x2E;
            // LASTORE 0x50 -> LALOAD 0x2F;
            // FASTORE 0x51 -> FALOAD 0x30;
            // DASTORE 0x52 -> DALOAD 0x31;
            // AASTORE 0x53 -> AALOAD 0x32;
            // BASTORE 0x54 -> BALOAD 0x33;
            // CASTORE 0x55 -> CALOAD 0x34;
            // SASTORE 0x56 -> SALOAD 0x35;
            GenericInstruction gi = (GenericInstruction)storeInstruction;
            byte[] code = gi.getBytecode();
            return new GenericInstruction(new byte[] {(byte)(opcode - (Opcode.IASTORE-Opcode.IALOAD))});
        }
        throw new IllegalArgumentException("Invalid store instruction");
    }
}

