package sysModel.classFile.code.instructions;

import sysModel.classFile.code.Opcode;
import junit.framework.TestCase;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.LinkedList;

/**
 * Unit tests for AInstruction classes.
 *
 * @author Mathias Ricken
 */
public class AInstructionTest extends TestCase {
    public void testFactory() throws IOException {
        AInstruction i;
        short pc = 0;
        LinkedList<AInstruction> list = new LinkedList<AInstruction>();
        ByteArrayOutputStream baos = new ByteArrayOutputStream();

        byte[] code = new byte[]{
            Opcode.ALOAD_0, // 0
            Opcode.IFEQ, 0, 0, // 1
            Opcode.GOTO, 0, 0, // 4
            Opcode.GOTO_W, 0, 0, 0, 0, // 7
            Opcode.ALOAD, 0, // 12
            Opcode.WIDE, Opcode.ALOAD, 0, 0, // 14
            Opcode.IINC, 0, 0, // 18
            Opcode.WIDE, Opcode.IINC, 0, 0, 0, 0, // 21
            Opcode.TABLESWITCH, /*default*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xf1, /*low*/0, 0, 0, 5, /*high*/0, 0, 0, 7,
            /*5*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xf3, /*6*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xf7, /*7*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xfa, // 27
            Opcode.LOOKUPSWITCH, /*pad*/ 0, 0, 0, /*default*/(byte)0xff, (byte)0xff, (byte)0xff, (byte)0xe7, /*n*/0, 0, 0, 2, /*key*/0, 0, 0, 0,
            /*offset*/0, 0, 0, 0, /*key*/ 0, 0, 0, 1, /*offset*/ (byte)0xff, (byte)0xff, (byte)0xff, (byte)0xcc, // 52
            Opcode.RETURN // 80
        };
        LineNumberTable lnt = new LineNumberTable(code);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", GenericInstruction.class, i.getClass());
        assertEquals("Wrong size", 1, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", BranchInstruction.class, i.getClass());
        assertEquals("Wrong size", 3, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", BranchInstruction.class, i.getClass());
        assertEquals("Wrong size", 3, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", WideBranchInstruction.class, i.getClass());
        assertEquals("Wrong size", 5, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", GenericInstruction.class, i.getClass());
        assertEquals("Wrong size", 2, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", WideInstruction.class, i.getClass());
        assertEquals("Wrong size", 4, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", GenericInstruction.class, i.getClass());
        assertEquals("Wrong size", 3, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", WideInstruction.class, i.getClass());
        assertEquals("Wrong size", 6, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", TableSwitchInstruction.class, i.getClass());
        assertEquals("Wrong size", 25, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", LookupSwitchInstruction.class, i.getClass());
        assertEquals("Wrong size", 28, i.getBytecodeLength(pc));
        list.add(i);
        pc += i.getBytecodeLength(pc);

        i = AInstruction.makeInstruction(code, pc, pc, lnt);
        assertEquals("Wrong instruction", GenericInstruction.class, i.getClass());
        assertEquals("Wrong size", 1, i.getBytecodeLength(pc));
        list.add(i);

        LineNumberTable reverseLnt = new LineNumberTable(list);
        pc = 0;
        for(AInstruction instr : list) {
            baos.write(instr.getBytecode(pc, reverseLnt));
            pc += instr.getBytecodeLength(pc);
        }

        byte[] generatedCode = baos.toByteArray();
        assertTrue("Generated bytecode not the same", Arrays.equals(code, generatedCode));
    }

    public void testCorrespondingLoad() {
        AInstruction i;

        i = new GenericInstruction(new byte[] {Opcode.ISTORE_0});
        assertEquals("Wrong load for ISTORE_0", "iload_0 ", AInstruction.getCorrespondingLoad(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.ASTORE_3});
        assertEquals("Wrong load for ASTORE_3", "aload_3 ", AInstruction.getCorrespondingLoad(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.IASTORE});
        assertEquals("Wrong load for IASTORE", "iaload ", AInstruction.getCorrespondingLoad(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.SASTORE});
        assertEquals("Wrong load for SASTORE", "saload ", AInstruction.getCorrespondingLoad(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.ISTORE, 5});
        assertEquals("Wrong load for ISTORE", "iload 05 ", AInstruction.getCorrespondingLoad(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.ASTORE, 5});
        assertEquals("Wrong load for ASTORE", "aload 05 ", AInstruction.getCorrespondingLoad(i).toString());

        i = new WideInstruction(new byte[] {Opcode.ISTORE, 0, 5});
        assertEquals("Wrong load for WIDE ISTORE", "wide iload 00 05 ", AInstruction.getCorrespondingLoad(i).toString());

        i = new WideInstruction(new byte[] {Opcode.ASTORE, 0, 5});
        assertEquals("Wrong load for WIDE ASTORE", "wide aload 00 05 ", AInstruction.getCorrespondingLoad(i).toString());
    }

    public void testCorrespondingStore() {
        AInstruction i;

        i = new GenericInstruction(new byte[] {Opcode.ILOAD_0});
        assertEquals("Wrong store for ILOAD_0", "istore_0 ", AInstruction.getCorrespondingStore(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.ALOAD_3});
        assertEquals("Wrong store for ALOAD_3", "astore_3 ", AInstruction.getCorrespondingStore(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.IALOAD});
        assertEquals("Wrong store for IALOAD", "iastore ", AInstruction.getCorrespondingStore(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.SALOAD});
        assertEquals("Wrong store for SALOAD", "sastore ", AInstruction.getCorrespondingStore(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.ILOAD, 5});
        assertEquals("Wrong store for ILOAD", "istore 05 ", AInstruction.getCorrespondingStore(i).toString());

        i = new GenericInstruction(new byte[] {Opcode.ALOAD, 5});
        assertEquals("Wrong store for ALOAD", "astore 05 ", AInstruction.getCorrespondingStore(i).toString());

        i = new WideInstruction(new byte[] {Opcode.ILOAD, 0, 5});
        assertEquals("Wrong store for WIDE ILOAD", "wide istore 00 05 ", AInstruction.getCorrespondingStore(i).toString());

        i = new WideInstruction(new byte[] {Opcode.ALOAD, 0, 5});
        assertEquals("Wrong store for WIDE ALOAD", "wide astore 00 05 ", AInstruction.getCorrespondingStore(i).toString());
    }
}
