package sysModel.classFile.code.instructions;

import sysModel.classFile.code.Opcode;

import java.util.Arrays;

/**
 * Generic Java instruction.
 *
 * @author Mathias Ricken
 */
public class GenericInstruction extends AInstruction {
    /**
     * Bytecode.
     */
    protected byte[] _code;

    /**
     * Constructor.
     *
     * @param code bytecode for generic instruction
     */
    public GenericInstruction(byte[] code) {
        switch(code[0]) {
            case Opcode.TABLESWITCH:
            case Opcode.LOOKUPSWITCH:
            case Opcode.IFEQ:
            case Opcode.IFNE:
            case Opcode.IFLT:
            case Opcode.IFGE:
            case Opcode.IFGT:
            case Opcode.IFLE:
            case Opcode.IF_ICMPEQ:
            case Opcode.IF_ICMPNE:
            case Opcode.IF_ICMPLT:
            case Opcode.IF_ICMPGE:
            case Opcode.IF_ICMPGT:
            case Opcode.IF_ICMPLE:
            case Opcode.IF_ACMPEQ:
            case Opcode.IF_ACMPNE:
            case Opcode.GOTO:
            case Opcode.JSR:
            case Opcode.IFNULL:
            case Opcode.IFNONNULL:
            case Opcode.GOTO_W:
            case Opcode.JSR_W:
            case Opcode.WIDE:
                throw new IllegalArgumentException("Invalid generic opcode");
        }
        _code = code;
    }

    /**
     * Get the opcode of this instruction.
     *
     * @return opcode
     */
    public byte getOpcode() {
        return _code[0];
    }

    /**
     * Get the length bytecode for this instruction, padded for the specified program counter value.
     *
     * @param pc PC for padding
     *
     * @return bytecode length
     */
    public short getBytecodeLength(short pc) {
        return (short)Opcode.getInstrSize(_code, 0, 0);
    }

    /**
     * Make a new generic instruction from the bytecode stating at pc, padded using paddingPC, and use the line number
     * table for branches.
     *
     * @param bytecode  bytecode
     * @param pc        starting index in bytecode
     * @param paddingPC PC for padding
     * @param lnt       line number table for branches
     */
    public GenericInstruction(byte[] bytecode, short pc, short paddingPC, LineNumberTable lnt) {
        _code = new byte[Opcode.getInstrSize(bytecode, pc, paddingPC)];
        System.arraycopy(bytecode, pc, _code, 0, _code.length);
    }

    /**
     * Get the bytecode for this instruction, padded for the specified program counter value, with branch targets
     * according to the specified line number table
     *
     * @param pc  PC for padding
     * @param lnt line number table for branches
     *
     * @return bytecode
     */
    public byte[] getBytecode(short pc, LineNumberTable lnt) {
        return getBytecode();
    }

    /**
     * Get the bytecode for this instruction.
     *
     * @return bytecode
     */
    public byte[] getBytecode() {
        byte[] b = new byte[_code.length];
        System.arraycopy(_code, 0, b, 0, _code.length);
        return b;
    }

    /**
     * Return true of this instruction and the other object are equal.
     *
     * @param o other object
     *
     * @return true if equal
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof GenericInstruction)) {
            return false;
        }

        GenericInstruction genericInstruction = (GenericInstruction)o;
        return Arrays.equals(_code, genericInstruction._code);

    }

    /**
     * Return hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _code[0];
    }

    /**
     * Return an array of target indices.
     *
     * @return array of target indices.
     */
    public short[] getBranchTargets() {
        return new short[0];
    }

    /**
     * Set the branch target indices.
     *
     * @param branchTargets array of target indices
     */
    public void setBranchTargets(short[] branchTargets) {
        if (0 != branchTargets.length) {
            throw new IllegalArgumentException("Generic instruction cannot have a target");
        }
    }

    /**
     * Return instruction in human-readable form.
     *
     * @return string representation
     */
    public String toString() {
        StringBuffer x = new StringBuffer();
        x.append(Opcode.getOpcodeName(_code[0]));
        x.append(' ');
        for(int i = 1; i < _code.length; ++i) {
            x.append(String.format("%02x", new Object[]{new Byte(_code[i])})).append(' ');
        }
        return x.toString();
    }
}

