package sysModel.classFile.code.instructions;

import sysModel.classFile.code.InstructionList;
import sysModel.classFile.code.Opcode;

import java.util.Hashtable;
import java.util.LinkedList;
import java.util.NoSuchElementException;

/**
 * A table that relates PC values to line numbers.
 *
 * @author Mathias Ricken
 */
public class LineNumberTable {
    /**
     * Hash table to convert from PC to line numbers.
     */
    Hashtable<Integer, Integer> _pcToLineNumber = new Hashtable<Integer, Integer>();

    /**
     * Hash table to convert from line numbers to PCs.
     */
    Hashtable<Integer, Integer> _lineNumberToPC = new Hashtable<Integer, Integer>();

    /**
     * PC value just after the last instruction.
     */
    int _maxPC;

    /**
     * Line number value just after the last instruction.
     */
    int _maxLineNo;

    /**
     * Put this PC-line number pair into the hash tables.
     * @param pc PC value
     * @param lineNo line number
     */
    protected void put(int pc, int lineNo) {
        _pcToLineNumber.put(pc, lineNo);
        _lineNumberToPC.put(lineNo, pc);
    }

    /**
     * Create the line number table from bytecode.
     *
     * @param bytecode bytecode
     */
    public LineNumberTable(byte[] bytecode) {
        int pc = 0;
        int lineNo = 0;
        while(pc < bytecode.length) {
            put(pc, lineNo);
            pc += Opcode.getInstrSize(bytecode, pc);
            ++lineNo;
        }
        if (pc != bytecode.length) {
            throw new IllegalArgumentException("Invalid bytecode length");
        }
        put(bytecode.length,lineNo);
    }

    /**
     * Create a line number table from a list of instructions.
     *
     * @param instrList list of instructions
     */
    public LineNumberTable(LinkedList<AInstruction> instrList) {
        int pc = 0;
        int lineNo = 0;
        for(AInstruction i : instrList) {
            put(pc, lineNo);
            pc += i.getBytecodeLength((short)pc);
            ++lineNo;
        }
        put(pc,instrList.size());
    }

    /**
     * Create a line number table from a list of instructions.
     *
     * @param ilist list of instructions
     */
    public LineNumberTable(InstructionList ilist) {
        int pc = 0;
        if (0 < ilist.getLength()) {
            InstructionList copy = new InstructionList(ilist);
            copy.setIndex(0);
            int lineNo = 0;
            do {
                put(pc, lineNo);
                pc += copy.getInstr().getBytecodeLength((short)pc);
                ++lineNo;
            } while(copy.advanceIndex());
        }
        put(pc,ilist.getLength());
    }

    /**
     * Get the line number from the PC.
     *
     * @param pc program counter
     *
     * @return line number
     */
    public int getLineNumber(int pc) throws NoSuchElementException {
        Integer i = _pcToLineNumber.get(pc);
        if (null == i) {
            throw new NoSuchElementException("No line number found for PC=" + pc);
        }
        return i;
    }

    /**
     * Get the PC from the line number.
     *
     * @param lineNo line number
     *
     * @return program counter
     */
    public int getPC(int lineNo) throws NoSuchElementException {
        Integer i = _lineNumberToPC.get(lineNo);
        if (null == i) {
            throw new NoSuchElementException("No PC found for line no=" + lineNo);
        }
        return i;
    }
}
