package sysModel.classFile.code.instructions;

import sysModel.classFile.code.Opcode;
import sysModel.classFile.Types;

import java.util.Arrays;

/**
 * LOOKUPSWITCH Java instruction.
 *
 * @author Mathias Ricken
 */
public class LookupSwitchInstruction extends AInstruction {
    /**
     * Default target.
     */
    protected int _defaultTarget;

    /**
     * Branch targets.
     */
    protected int[] _targets;

    /**
     * Branch keys.
     */
    protected int[] _keys;

    /**
     * Constructor.
     *
     * @param defaultTarget default target line number
     * @param keys          keys
     * @param targets       target line numbers
     */
    public LookupSwitchInstruction(int defaultTarget, int[] keys, int[] targets) {
        _defaultTarget = defaultTarget;
        _targets = targets;
        _keys = keys;
    }

    /**
     * Get the opcode of this instruction.
     *
     * @return opcode
     */
    public byte getOpcode() {
        return Opcode.LOOKUPSWITCH;
    }

    /**
     * Get the length bytecode for this instruction, padded for the specified program counter value.
     *
     * @param pc PC for padding
     *
     * @return bytecode length
     */
    public short getBytecodeLength(short pc) {
        int pad = 3 - (pc % 4);
        return (short)(pad + 9 + (_targets.length) * 8);
    }

    /**
     * Make a new LOOKUPSWITCH instruction from the bytecode stating at pc, padded using paddingPC, and use the line
     * number table for branches.
     *
     * @param bytecode  bytecode
     * @param pc        starting index in bytecode
     * @param paddingPC PC for padding
     * @param lnt       line number table for branches
     */
    public LookupSwitchInstruction(byte[] bytecode, short pc, short paddingPC, LineNumberTable lnt) {
        int[] branchTargets = Opcode.getBranchTargets(bytecode, pc, paddingPC);

        _defaultTarget = lnt.getLineNumber((short)branchTargets[0]);

        int pad = 3 - (paddingPC % 4);
        _targets = new int[branchTargets.length - 1];
        _keys = new int[branchTargets.length - 1];
        for(int i = 1; i < branchTargets.length; ++i) {
            _targets[i - 1] = lnt.getLineNumber((short)branchTargets[i]);
            _keys[i - 1] = Types.intFromBytes(bytecode, pc + pad + 1 + i * 8);
        }
    }

    /**
     * Get the bytecode for this instruction, padded for the specified program counter value, with branch targets
     * according to the specified line number table
     *
     * @param pc  PC for padding
     * @param lnt line number table for branches
     *
     * @return bytecode
     */
    public byte[] getBytecode(short pc, LineNumberTable lnt) {
        int pad = 3 - (pc % 4);
        byte[] b = new byte[getBytecodeLength(pc)];
        b[0] = getOpcode();
        Types.bytesFromInt(lnt.getPC((short)_defaultTarget) - pc, b, pad + 1);
        Types.bytesFromInt((int)_targets.length, b, pad + 5);
        for(int i = 0; i < _targets.length; ++i) {
            Types.bytesFromInt(_keys[i], b, pad + 9 + i * 8);
            Types.bytesFromInt(lnt.getPC((short)_targets[i]) - pc, b, pad + 13 + i * 8);
        }
        return b;
    }

    /**
     * Return true of this instruction and the other object are equal.
     *
     * @param o other object
     *
     * @return true if equal
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof LookupSwitchInstruction)) {
            return false;
        }

        LookupSwitchInstruction lookupSwitchInstruction = (LookupSwitchInstruction)o;

        if (_defaultTarget != lookupSwitchInstruction._defaultTarget) {
            return false;
        }
        if (!Arrays.equals(_keys, lookupSwitchInstruction._keys)) {
            return false;
        }
        return Arrays.equals(_targets, lookupSwitchInstruction._targets);

    }

    /**
     * Return hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _defaultTarget;
    }

    /**
     * Return an array of target indices.
     *
     * @return array of target indices.
     */
    public short[] getBranchTargets() {
        short[] bt = new short[_targets.length + 1];
        bt[0] = (short)_defaultTarget;
        int i = 1;
        for(int t : _targets) {
            bt[i++] = (short)t;
        }
        return bt;
    }

    /**
     * Set the branch target indices.
     *
     * @param branchTargets array of target indices
     */
    public void setBranchTargets(short[] branchTargets) {
        if (branchTargets.length != _targets.length + 1) {
            throw new IllegalArgumentException("LOOKUPSWITCH has incorrect number of targets");
        }
        _defaultTarget = branchTargets[0];
        for(int i = 1; i < branchTargets.length; ++i) {
            _targets[i - 1] = branchTargets[i];
        }
    }

    /**
     * Return instruction in human-readable form.
     *
     * @return string representation
     */
    public String toString() {
        StringBuffer x = new StringBuffer();
        x.append(Opcode.getOpcodeName(Opcode.LOOKUPSWITCH));
        x.append(" default = ");
        x.append(_defaultTarget);
        x.append(" n = ");
        x.append(_targets.length);
        for(int i = 0; i < _targets.length; ++i) {
            x.append(" (");
            x.append(_keys[i]);
            x.append("->");
            x.append(_targets[i]);
            x.append(')');
        }
        return x.toString();
    }
}

