package sysModel.classFile.code.instructions;

import sysModel.classFile.code.Opcode;
import sysModel.classFile.Types;

/**
 * Instruction containing a reference to the constant pool.
 *
 * @author Mathias
 */
public class ReferenceInstruction extends GenericInstruction {
    /**
     * Constructor.
     *
     * @param opcode         invoke opcode
     * @param constantPoolId method id in constant pool
     */
    public ReferenceInstruction(byte opcode, short constantPoolId) {
        super(new byte[]{Opcode.NOP});

        switch(opcode) {
            case Opcode.INVOKESTATIC:
            case Opcode.INVOKESPECIAL:
            case Opcode.INVOKEVIRTUAL:
            case Opcode.CHECKCAST:
            case Opcode.GETFIELD:
            case Opcode.GETSTATIC:
            case Opcode.INSTANCEOF:
            case Opcode.NEW:
            case Opcode.PUTFIELD:
            case Opcode.PUTSTATIC:
            case Opcode.ANEWARRAY:
            case Opcode.NEWARRAY:
            case Opcode.LDC_W:
            case Opcode.LDC2_W:
                break;
            default:
                throw new IllegalArgumentException("Illegal reference instruction: " + Opcode.getOpcodeName(opcode));
        }

        _code = new byte[3];
        _code[0] = opcode;
        setReference(constantPoolId);
    }

    /**
     * Make a new generic instruction from the bytecode stating at pc, padded using paddingPC, and use the line number
     * table for branches.
     *
     * @param bytecode  bytecode
     * @param pc        starting index in bytecode
     * @param paddingPC PC for padding
     * @param lnt       line number table for branches
     */
    public ReferenceInstruction(byte[] bytecode, short pc, short paddingPC, LineNumberTable lnt) {
        super(bytecode, pc, paddingPC, lnt);
    }

    /**
     * Set the constant pool id.
     *
     * @param constantPoolId new constant pool id
     */
    public void setReference(short constantPoolId) {
        Types.bytesFromShort(constantPoolId, _code, 1);
    }

    /**
     * Get the constant pool id.
     *
     * @return constant pool id
     */
    public short getReference() {
        return Types.shortFromBytes(_code, 1);
    }
}
