package sysModel.classFile.code.instructions;

import sysModel.classFile.code.Opcode;
import sysModel.classFile.Types;

import java.util.Arrays;

/**
 * TABLESWITCH Java instruction.
 *
 * @author Mathias Ricken
 */
public class TableSwitchInstruction extends AInstruction {
    /**
     * Lower bound.
     */
    protected long _low;

    /**
     * Upper bound.
     */
    protected long _high;

    /**
     * Branch targets.
     */
    protected int[] _targets;

    /**
     * Constructor.
     *
     * @param defaultTarget default target line number
     * @param low           lower bound
     * @param high          upper bound
     * @param targets       target line nu,bers
     */
    public TableSwitchInstruction(int defaultTarget, long low, long high, int[] targets) {
        if (low > high) {
            throw new IllegalArgumentException("High cannot be greater than low.");
        }
        _low = low;
        _high = high;
        long npairs = high - low + 1;
        if (npairs != targets.length) {
            throw new IllegalArgumentException("Invalid number of targets");
        }
        _targets = new int[(int)npairs + 1];
        _targets[0] = defaultTarget;
        System.arraycopy(targets, 0, _targets, 1, (int)npairs);
    }

    /**
     * Get the opcode of this instruction.
     *
     * @return opcode
     */
    public byte getOpcode() {
        return Opcode.TABLESWITCH;
    }

    /**
     * Get the length bytecode for this instruction, padded for the specified program counter value.
     *
     * @param pc PC for padding
     *
     * @return bytecode length
     */
    public short getBytecodeLength(short pc) {
        int pad = 3 - (pc % 4);
        return (short)(pad + 9 + (_targets.length) * 4);
    }

    /**
     * Make a new TABLESWITCH instruction from the bytecode stating at pc, padded using paddingPC, and use the line
     * number table for branches.
     *
     * @param bytecode  bytecode
     * @param pc        starting index in bytecode
     * @param paddingPC PC for padding
     * @param lnt       line number table for branches
     */
    public TableSwitchInstruction(byte[] bytecode, short pc, short paddingPC, LineNumberTable lnt) {
        int[] branchTargets = Opcode.getBranchTargets(bytecode, pc, paddingPC);
        int pad = 3 - (paddingPC % 4);
        _low = Types.intFromBytes(bytecode, pc + pad + 5);
        _high = Types.intFromBytes(bytecode, pc + pad + 9);

        _targets = new int[branchTargets.length];
        int i = 0;
        for(int bt : branchTargets) {
            _targets[i++] = lnt.getLineNumber((short)bt);
        }
    }

    /**
     * Get the bytecode for this instruction, padded for the specified program counter value, with branch targets
     * according to the specified line number table
     *
     * @param pc  PC for padding
     * @param lnt line number table for branches
     *
     * @return bytecode
     */
    public byte[] getBytecode(short pc, LineNumberTable lnt) {
        int pad = 3 - (pc % 4);
        byte[] b = new byte[getBytecodeLength(pc)];
        b[0] = getOpcode();
        Types.bytesFromInt(lnt.getPC((short)_targets[0]) - pc, b, pad + 1);
        Types.bytesFromInt((int)_low, b, pad + 5);
        Types.bytesFromInt((int)_high, b, pad + 9);
        for(int i = 1; i < _targets.length; ++i) {
            Types.bytesFromInt(lnt.getPC((short)_targets[i]) - pc, b, pad + 9 + i * 4);
        }
        return b;
    }

    /**
     * Return true of this instruction and the other object are equal.
     *
     * @param o other object
     *
     * @return true if equal
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof TableSwitchInstruction)) {
            return false;
        }

        TableSwitchInstruction tableSwitchInstruction = (TableSwitchInstruction)o;

        if (_high != tableSwitchInstruction._high) {
            return false;
        }
        if (_low != tableSwitchInstruction._low) {
            return false;
        }
        return Arrays.equals(_targets, tableSwitchInstruction._targets);

    }

    /**
     * Return hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        int result;
        result = (int)(_low ^ (_low >>> 32));
        result = 29 * result + (int)(_high ^ (_high >>> 32));
        return result;
    }

    /**
     * Return an array of target indices.
     *
     * @return array of target indices.
     */
    public short[] getBranchTargets() {
        short[] bt = new short[_targets.length];
        int i = 0;
        for(int t : _targets) {
            bt[i++] = (short)t;
        }
        return bt;
    }

    /**
     * Set the branch target indices.
     *
     * @param branchTargets array of target indices
     */
    public void setBranchTargets(short[] branchTargets) {
        if (branchTargets.length != _targets.length) {
            throw new IllegalArgumentException("TABLESWITCH has incorrect number of targets");
        }
        for(int i = 0; i < branchTargets.length; ++i) {
            _targets[i] = branchTargets[i];
        }
    }

    /**
     * Return instruction in human-readable form.
     *
     * @return string representation
     */
    public String toString() {
        StringBuffer x = new StringBuffer();
        x.append(Opcode.getOpcodeName(Opcode.TABLESWITCH));
        x.append(" default = ");
        x.append(_targets[0]);
        x.append(" low = ");
        x.append(_low);
        x.append(" high = ");
        x.append(_high);
        x.append(" n = ");
        x.append(_targets.length - 1);
        for(int i = 0; i < _targets.length - 1; ++i) {
            x.append(" (");
            x.append(_low + i);
            x.append("->");
            x.append(_targets[i + 1]);
            x.append(')');
        }
        return x.toString();
    }
}

