package sysModel.classFile.code.instructions;

import sysModel.classFile.code.Opcode;
import sysModel.classFile.Types;

/**
 * Wide branch Java instruction.
 *
 * @author Mathias Ricken
 */
public class WideBranchInstruction extends AInstruction {
    /**
     * Opcode.
     */
    protected byte _opcode;

    /**
     * Branch target.
     */
    protected int _target;

    /**
     * Constructor.
     *
     * @param opcode branch opcode
     * @param target target line number
     */
    public WideBranchInstruction(byte opcode, int target) {
        switch(opcode) {
            case Opcode.GOTO_W:
            case Opcode.JSR_W:
                break;
            default:
                throw new IllegalArgumentException("Invalid wide branch opcode");
        }
        _opcode = opcode;
        _target = target;
    }

    /**
     * Get the opcode of this instruction.
     *
     * @return opcode
     */
    public byte getOpcode() {
        return _opcode;
    }

    /**
     * Get the length bytecode for this instruction, padded for the specified program counter value.
     *
     * @param pc PC for padding
     *
     * @return bytecode length
     */
    public short getBytecodeLength(short pc) {
        return 5;
    }

    /**
     * Make a new wide branch instruction from the bytecode stating at pc, padded using paddingPC, and use the line
     * number table for branches.
     *
     * @param bytecode  bytecode
     * @param pc        starting index in bytecode
     * @param paddingPC PC for padding
     * @param lnt       line number table for branches
     */
    public WideBranchInstruction(byte[] bytecode, short pc, short paddingPC, LineNumberTable lnt) {
        _opcode = bytecode[pc];
        _target = lnt.getLineNumber(pc + Types.intFromBytes(bytecode, pc + 1));
    }

    /**
     * Get the bytecode for this instruction, padded for the specified program counter value, with branch targets
     * according to the specified line number table
     *
     * @param pc  PC for padding
     * @param lnt line number table for branches
     *
     * @return bytecode
     */
    public byte[] getBytecode(short pc, LineNumberTable lnt) {
        byte[] b = new byte[getBytecodeLength(pc)];

        b[0] = getOpcode();
        Types.bytesFromInt(lnt.getPC(_target) - pc, b, 1);

        return b;
    }

    /**
     * Return true of this instruction and the other object are equal.
     *
     * @param o other object
     *
     * @return true if equal
     */
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof WideBranchInstruction)) {
            return false;
        }

        WideBranchInstruction wideBranchInstruction = (WideBranchInstruction)o;

        if (_opcode != wideBranchInstruction._opcode) {
            return false;
        }
        return _target == wideBranchInstruction._target;

    }

    /**
     * Return hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        int result;
        result = (int)_opcode;
        result = 29 * result + _target;
        return result;
    }

    /**
     * Return an array of target indices.
     *
     * @return array of target indices.
     */
    public short[] getBranchTargets() {
        return new short[]{
            (short)_target
        };
    }

    /**
     * Set the branch target indices.
     *
     * @param branchTargets array of target indices
     */
    public void setBranchTargets(short[] branchTargets) {
        if (1 != branchTargets.length) {
            throw new IllegalArgumentException("Wide branch instruction can only have one target");
        }
        _target = branchTargets[0];
    }


    /**
     * Return instruction in human-readable form.
     *
     * @return string representation
     */
    public String toString() {
        StringBuffer x = new StringBuffer();
        x.append(Opcode.getOpcodeName(_opcode));
        x.append(' ');
        x.append(_target);
        return x.toString();
    }
}

