package sysModel.classFile.code.instructions;

import sysModel.classFile.code.Opcode;

/**
 * Wide Java instruction.
 *
 * @author Mathias Ricken
 */
public class WideInstruction extends GenericInstruction {
    /**
     * Constructor.
     *
     * @param code bytecode for generic instruction, excluding the WIDE opcode
     */
    public WideInstruction(byte[] code) {
        super(new byte[]{Opcode.NOP});

        switch(code[0]) {
            case Opcode.ALOAD:
            case Opcode.DLOAD:
            case Opcode.ILOAD:
            case Opcode.FLOAD:
            case Opcode.LLOAD:
            case Opcode.ASTORE:
            case Opcode.DSTORE:
            case Opcode.ISTORE:
            case Opcode.FSTORE:
            case Opcode.LSTORE:
            case Opcode.RET:
            case Opcode.IINC:
                break;
            default:
                throw new IllegalArgumentException("Illegal wide instruction");
        }

        _code = new byte[code.length + 1];
        _code[0] = Opcode.WIDE;
        System.arraycopy(code, 0, _code, 1, code.length);
    }

    /**
     * Get the opcode of this instruction.
     *
     * @return opcode
     */
    public byte getOpcode() {
        return Opcode.WIDE;
    }

    /**
     * Get the length bytecode for this instruction, padded for the specified program counter value.
     *
     * @param pc PC for padding
     *
     * @return bytecode length
     */
    public short getBytecodeLength(short pc) {
        switch(_code[1]) {
            case Opcode.ALOAD:
            case Opcode.DLOAD:
            case Opcode.ILOAD:
            case Opcode.FLOAD:
            case Opcode.LLOAD:
            case Opcode.ASTORE:
            case Opcode.DSTORE:
            case Opcode.ISTORE:
            case Opcode.FSTORE:
            case Opcode.LSTORE:
            case Opcode.RET:
                return 4;
            case Opcode.IINC:
                return 6;
            default:
                throw new IllegalArgumentException("Illegal wide instruction");
        }
    }

    /**
     * Make a new generic instruction from the bytecode stating at pc, padded using paddingPC, and use the line number
     * table for branches.
     *
     * @param bytecode  bytecode
     * @param pc        starting index in bytecode
     * @param paddingPC PC for padding
     * @param lnt       line number table for branches
     */
    public WideInstruction(byte[] bytecode, short pc, short paddingPC, LineNumberTable lnt) {
        super(bytecode, pc, paddingPC, lnt);
    }

    /**
     * Return hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _code[2];
    }

    /**
     * Set the branch target indices.
     *
     * @param branchTargets array of target indices
     */
    public void setBranchTargets(short[] branchTargets) {
        if (0 != branchTargets.length) {
            throw new IllegalArgumentException("Wide instruction cannot have a target");
        }
    }

    /**
     * Return instruction in human-readable form.
     *
     * @return string representation
     */
    public String toString() {
        StringBuffer x = new StringBuffer();
        x.append(Opcode.getOpcodeName(Opcode.WIDE));
        x.append(' ');
        x.append(Opcode.getOpcodeName(_code[1]));
        x.append(' ');
        for(int i = 2; i < _code.length; ++i) {
            x.append(String.format("%02x", new Object[]{new Byte(_code[i])})).append(' ');
        }
        return x.toString();
    }
}
