package sysModel.classFile.constantPool;

import sysModel.classFile.constantPool.visitors.ADefaultPoolInfoVisitor;
import sysModel.classFile.constantPool.visitors.CheckClassVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents an object in the constant pool described by class and name-and-type.
 *
 * @author Mathias Ricken
 */
public abstract class AClassNameTypePoolInfo extends APoolInfo {
    /**
     * Class information.
     */
    protected ClassPoolInfo _classInfo;

    /**
     * NameAndType information.
     */
    protected NameAndTypePoolInfo _nameAndType;

    /**
     * Class index.
     */
    protected short _classInfoIndex;

    /**
     * NameAndType index.
     */
    protected short _nameAndTypeIndex;

    /**
     * Create a new object.
     *
     * @param type        type, either FIELD, METHOD or CONSTANT_InterfaceMethodref
     * @param clas        class information
     * @param nameAndType NameAndType information
     * @param cp          constant pool
     */
    public AClassNameTypePoolInfo(int type,
                                  ClassPoolInfo clas,
                                  NameAndTypePoolInfo nameAndType,
                                  ConstantPool cp) {
        super(type, cp);
        _classInfo = clas;
        _nameAndType = nameAndType;
        reindex();
    }

    /**
     * Constructor reading from a stream.
     *
     * @param type type, either FIELD, METHOD or CONSTANT_InterfaceMethodref
     * @param dis  input stream
     * @param cp   constant pool
     *
     * @throws IOException
     */
    public AClassNameTypePoolInfo(int type, DataInputStream dis, ConstantPool cp) throws IOException {
        super(type, cp);
        _classInfoIndex = dis.readShort();
        _nameAndTypeIndex = dis.readShort();
    }

    /**
     * Accessor for the class information.
     *
     * @return class information
     */
    public ClassPoolInfo getClassInfo() {
        return _classInfo;
    }

    /**
     * Mutator for the class information.
     *
     * @param classInfo new class information
     */
    public void setClassInfo(ClassPoolInfo classInfo) {
        _classInfo = classInfo;
    }

    /**
     * Accessor for the NameAndType information.
     *
     * @return NameAndType information
     */
    public NameAndTypePoolInfo getNameAndType() {
        return _nameAndType;
    }

    /**
     * Mutator for the NameAndType information.
     *
     * @param nameAndType new NameAndTypeInformation
     */
    public void setNameAndType(NameAndTypePoolInfo nameAndType) {
        _nameAndType = nameAndType;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        reindex();
        dos.writeByte(_type);
        dos.writeShort(_classInfoIndex);
        dos.writeShort(_nameAndTypeIndex);
    }

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public void resolve() {
        _classInfo = _constantPool.get(_classInfoIndex).execute(CheckClassVisitor.singleton(), null);
        _nameAndType = _constantPool.get(_nameAndTypeIndex).execute(new ADefaultPoolInfoVisitor<NameAndTypePoolInfo, Object>() {
            public NameAndTypePoolInfo defaultCase(APoolInfo host, Object o) {
                throw new ClassFormatError(
                    "Info is of type " + host.getClass().getName() + ", needs to be NameAndTypePoolInfo");
            }

            public NameAndTypePoolInfo nameAndTypeCase(NameAndTypePoolInfo host, Object o) {
                return host;
            }
        }, null);
    }

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public void reindex() {
        _classInfoIndex = _constantPool.indexOf(_classInfo);
        _nameAndTypeIndex = _constantPool.indexOf(_nameAndType);
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toStringVerbose() {
        return toString();
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toString() {
        StringBuffer s;

        s = new StringBuffer();
        s.append("FIELD: Class = #");
        s.append(_classInfoIndex);
        s.append(", Name and type = #");
        s.append(_nameAndTypeIndex);

        return s.toString();
    }

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _classInfoIndex + 6101 * _nameAndTypeIndex;
    }
}