/*
 * @(#)APoolInfo.java	1.5 95/08/16 Chuck McManis
 *
 * Copyright (c) 1996 Chuck McManis, All Rights Reserved.
 *
 * Permission to use, copy, modify, and distribute this software
 * and its documentation for NON-COMMERCIAL purposes and without
 * fee is hereby granted provided that this copyright notice
 * appears in all copies.
 *
 * CHUCK MCMANIS MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY
 * OF THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
 * TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
 * PARTICULAR PURPOSE, OR NON-INFRINGEMENT. CHUCK MCMANIS SHALL NOT BE
 * LIABLE FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
 * MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS DERIVATIVES.
 */

package sysModel.classFile.constantPool;

import sysModel.classFile.constantPool.visitors.IPoolInfoVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents an abstract constant pool object.
 *
 * @author Mathias Ricken
 */
public abstract class APoolInfo {
    // Constant pool object types
    public static final int CONSTANT_Class = 7;
    public static final int CONSTANT_Fieldref = 9;
    public static final int CONSTANT_Methodref = 10;
    public static final int CONSTANT_InterfaceMethodref = 11;
    public static final int CONSTANT_String = 8;
    public static final int CONSTANT_Integer = 3;
    public static final int CONSTANT_Float = 4;
    public static final int CONSTANT_Long = 5;
    public static final int CONSTANT_Double = 6;
    public static final int CONSTANT_NameAndType = 12;
    public static final int CONSTANT_Utf8_ASCII = 1;
    public static final int CONSTANT_Utf8_Unicode = 2;

    /**
     * Object type.
     */
    protected int _type;

    /**
     * Reference to the constant pool.
     */
    protected ConstantPool _constantPool;

    /**
     * Constructor
     *
     * @param type object type
     */
    public APoolInfo(int type, ConstantPool cp) {
        _type = type;
        _constantPool = cp;
    }

    /**
     * Read from stream and return unresolved constant pool object.
     *
     * @param dis stream
     *
     * @return unresolved constant pool object.
     *
     * @throws IOException
     * @throws ClassFormatError
     */
    public static APoolInfo read(DataInputStream dis, ConstantPool cp) throws IOException, ClassFormatError {
        int type = dis.readByte();
        APoolInfo item;
        switch(type) {
            case CONSTANT_Class:
                item = new ClassPoolInfo(dis, cp);
                break;
            case CONSTANT_Fieldref:
                item = new FieldPoolInfo(dis, cp);
                break;
            case CONSTANT_Methodref:
                item = new MethodPoolInfo(dis, cp);
                break;
            case CONSTANT_InterfaceMethodref:
                item = new InterfaceMethodPoolInfo(dis, cp);
                break;
            case CONSTANT_NameAndType:
                item = new NameAndTypePoolInfo(dis, cp);
                break;
            case CONSTANT_String:
                item = new StringPoolInfo(dis, cp);
                break;
            case CONSTANT_Integer:
                item = new IntegerPoolInfo(dis, cp);
                break;
            case CONSTANT_Float:
                item = new FloatPoolInfo(dis, cp);
                break;
            case CONSTANT_Long:
                item = new LongPoolInfo(dis, cp);
                break;
            case CONSTANT_Double:
                item = new DoublePoolInfo(dis, cp);
                break;
            case CONSTANT_Utf8_ASCII:
                item = new ASCIIPoolInfo(dis, cp);
                break;
            case CONSTANT_Utf8_Unicode:
                item = new UnicodePoolInfo(dis, cp);
                break;
            default:
                throw new ClassFormatError("Unknown constant pool item type");
        }
        return item;
    }

    /**
     * Return the reference to the constant pool item that is already in pool, that matches this one.
     *
     * @param pool constant pool
     *
     * @return matching object or null if not found
     */
    public APoolInfo inPool(ConstantPool pool) {
        for(APoolInfo cpi : pool) {
            if (equals(cpi)) {
                return cpi;
            }
        }
        return null;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public abstract void write(DataOutputStream dos) throws IOException;

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public abstract void resolve();

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public abstract void reindex();

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public abstract String toStringVerbose();

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public abstract String toString();

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public abstract int hashCode();

    /**
     * Compare this object and another one.
     *
     * @param obj other object
     *
     * @return true if the same
     */
    public abstract boolean equals(Object obj);

    /**
     * Execute a visitor.
     *
     * @param visitor visitor
     * @param data    visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public abstract <R, D> R execute(IPoolInfoVisitor<R, D> visitor, D data);
}
