package sysModel.classFile.constantPool;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents UTF data in the constant pool.
 *
 * @author Mathias Ricken
 */
public abstract class AUTFPoolInfo extends APoolInfo {
    /**
     * Data.
     */
    protected String _strValue;

    /**
     * Constructor.
     *
     * @param type constant pool object type
     * @param s    data
     */
    public AUTFPoolInfo(int type, String s, ConstantPool cp) {
        super(type, cp);
        _strValue = s;
    }

    /**
     * Constructor reading from a stream.
     *
     * @param type type, either ASCII or UNICODE
     * @param dis  input stream
     * @param cp   constant pool
     *
     * @throws IOException
     */
    public AUTFPoolInfo(int type, DataInputStream dis, ConstantPool cp) throws IOException {
        super(type, cp);
        StringBuffer xxBuf = new StringBuffer();

        int len = dis.readShort();
        while(0 < len) {
            char c = (char)(dis.readByte());
            xxBuf.append(c);
            --len;
        }
        _strValue = xxBuf.toString();
    }

    /**
     * Mutator for the data.
     *
     * @param strValue new data
     */
    public void setStrValue(String strValue) {
        _strValue = strValue;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        reindex();
        dos.writeByte(_type);
        dos.writeShort(_strValue.length());
        dos.writeBytes(_strValue);
    }

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public void resolve() {
        // do nothing
    }

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public void reindex() {
        // do nothing
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public abstract String toStringVerbose();

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toString() {
        return _strValue;
    }

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _strValue.hashCode();
    }

    /**
     * Compare this object and another one.
     *
     * @param obj other object
     *
     * @return true if the same
     */
    public abstract boolean equals(Object obj);
}