package sysModel.classFile.constantPool;

import sysModel.classFile.constantPool.visitors.CheckUTFVisitor;
import sysModel.classFile.constantPool.visitors.IPoolInfoVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents a class in the constant pool.
 *
 * @author Mathias Ricken
 */
public class ClassPoolInfo extends APoolInfo {
    /**
     * Class name information.
     */
    private AUTFPoolInfo _name;

    /**
     * Class name index.
     */
    private short _nameIndex;

    /**
     * Constructor.
     *
     * @param name class name information
     * @param cp   constant pool
     */
    public ClassPoolInfo(AUTFPoolInfo name, ConstantPool cp) {
        super(CONSTANT_Class, cp);
        _name = name;
        reindex();
    }

    /**
     * Constructor reading from a stream.
     *
     * @param dis input stream
     * @param cp  constant pool
     *
     * @throws IOException
     */
    public ClassPoolInfo(DataInputStream dis, ConstantPool cp) throws IOException {
        super(CONSTANT_Class, cp);
        _nameIndex = dis.readShort();
    }

    /**
     * Accessor for the name information.
     *
     * @return name information
     */
    public AUTFPoolInfo getName() {
        return _name;
    }

    /**
     * Mutator for the name information.
     *
     * @param name new name information
     */
    public void setName(AUTFPoolInfo name) {
        _name = name;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        reindex();
        dos.writeByte(_type);
        dos.writeShort(_nameIndex);
    }

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public void resolve() {
        _name = _constantPool.get(_nameIndex).execute(CheckUTFVisitor.singleton(), null);
    }

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public void reindex() {
        _nameIndex = _constantPool.indexOf(_name);
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toString() {
        StringBuffer s;

        s = new StringBuffer();
        s.append("CONSTANT_Class: Name = #");
        s.append(_nameIndex);

        return s.toString();
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toStringVerbose() {
        return toString();
    }

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _nameIndex;
    }

    /**
     * Compare this object and another one.
     *
     * @param obj other object
     *
     * @return true if the same
     */
    public boolean equals(Object obj) {
        return (obj instanceof ClassPoolInfo) && (
            (((ClassPoolInfo)obj)._nameIndex == _nameIndex) ||
            (((ClassPoolInfo)obj)._name == _name)
            );
    }

    /**
     * Execute a visitor.
     *
     * @param visitor visitor
     * @param data    visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public <R, D> R execute(IPoolInfoVisitor<R, D> visitor, D data) {
        return visitor.classCase(this, data);
    }
}