package sysModel.classFile.constantPool;

import sysModel.classFile.constantPool.visitors.IPoolInfoVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents a double in the constant pool.
 *
 * @author Mathias Ricken
 */
public class DoublePoolInfo extends APoolInfo {
    /**
     * Data.
     */
    private double _doubleValue;

    /**
     * Constructor.
     *
     * @param d data
     */
    public DoublePoolInfo(double d, ConstantPool cp) {
        super(CONSTANT_Double, cp);
        _doubleValue = d;
    }

    /**
     * Constructor reading from a stream.
     *
     * @param dis input stream
     * @param cp  constant pool
     *
     * @throws IOException
     */
    public DoublePoolInfo(DataInputStream dis, ConstantPool cp) throws IOException {
        super(CONSTANT_Double, cp);
        _doubleValue = dis.readDouble();
    }

    /**
     * Accessor for the data.
     *
     * @return data
     */
    public double getDoubleValue() {
        return _doubleValue;
    }

    /**
     * Mutator for the data.
     *
     * @param doubleValue new data
     */
    public void setDoubleValue(double doubleValue) {
        _doubleValue = doubleValue;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        dos.writeByte(_type);
        dos.writeDouble(_doubleValue);
    }

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public void resolve() {
        // do nothing
    }

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public void reindex() {
        // do nothing
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toString() {
        StringBuffer s;

        s = new StringBuffer();
        s.append("CONSTANT_Double: Value = ");
        s.append(_doubleValue);

        return s.toString();
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toStringVerbose() {
        return toString();
    }

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return new Double(_doubleValue).hashCode();
    }

    /**
     * Compare this object and another one.
     *
     * @param obj other object
     *
     * @return true if the same
     */
    public boolean equals(Object obj) {
        return (obj instanceof DoublePoolInfo) &&
            (((DoublePoolInfo)obj)._doubleValue == _doubleValue);
    }

    /**
     * Execute a visitor.
     *
     * @param visitor visitor
     * @param data    visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public <R, D> R execute(IPoolInfoVisitor<R, D> visitor, D data) {
        return visitor.doubleCase(this, data);
    }
}