package sysModel.classFile.constantPool;

import sysModel.classFile.constantPool.visitors.IPoolInfoVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents a float in the constant pool.
 *
 * @author Mathias Ricken
 */
public class FloatPoolInfo extends APoolInfo {
    /**
     * Data.
     */
    private float _floatValue;

    /**
     * Constructor.
     *
     * @param f data
     */
    public FloatPoolInfo(float f, ConstantPool cp) {
        super(CONSTANT_Float, cp);
        _floatValue = f;
    }

    /**
     * Constructor reading from a stream.
     *
     * @param dis input stream
     * @param cp  constant pool
     *
     * @throws IOException
     */
    public FloatPoolInfo(DataInputStream dis, ConstantPool cp) throws IOException {
        super(CONSTANT_Float, cp);
        _floatValue = dis.readFloat();
    }

    /**
     * Accessor for the data.
     *
     * @return data
     */
    public float getFloatValue() {
        return _floatValue;
    }

    /**
     * Mutator for the data.
     *
     * @param floatValue new data
     */
    public void setFloatValue(float floatValue) {
        _floatValue = floatValue;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        dos.writeByte(_type);
        dos.writeFloat(_floatValue);
    }

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public void resolve() {
        // do nothing
    }

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public void reindex() {
        // do nothing
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toString() {
        StringBuffer s;

        s = new StringBuffer();
        s.append("CONSTANT_Float: Value = ");
        s.append(_floatValue);

        return s.toString();
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toStringVerbose() {
        return toString();
    }

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return new Float(_floatValue).hashCode();
    }

    /**
     * Compare this object and another one.
     *
     * @param obj other object
     *
     * @return true if the same
     */
    public boolean equals(Object obj) {
        return (obj instanceof FloatPoolInfo) &&
            (((FloatPoolInfo)obj)._floatValue == _floatValue);
    }

    /**
     * Execute a visitor.
     *
     * @param visitor visitor
     * @param data    visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public <R, D> R execute(IPoolInfoVisitor<R, D> visitor, D data) {
        return visitor.floatCase(this, data);
    }
}