package sysModel.classFile.constantPool;

import sysModel.classFile.constantPool.visitors.IPoolInfoVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents an integer in the constant pool.
 *
 * @author Mathias Ricken
 */
public class IntegerPoolInfo extends APoolInfo {
    /**
     * Data.
     */
    private int _intValue;

    /**
     * Constructor.
     *
     * @param i data
     */
    public IntegerPoolInfo(int i, ConstantPool cp) {
        super(CONSTANT_Integer, cp);
        _intValue = i;
    }

    /**
     * Constructor reading from a stream.
     *
     * @param dis input stream
     * @param cp  constant pool
     *
     * @throws IOException
     */
    public IntegerPoolInfo(DataInputStream dis, ConstantPool cp) throws IOException {
        super(CONSTANT_Integer, cp);
        _intValue = dis.readInt();
    }

    /**
     * Accessor for the data.
     *
     * @return data
     */
    public int getIntValue() {
        return _intValue;
    }

    /**
     * Mutator for the data.
     *
     * @param intValue new data
     */
    public void setIntValue(int intValue) {
        _intValue = intValue;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        dos.writeByte(_type);
        dos.writeInt(_intValue);
    }

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public void resolve() {
        // do nothing
    }

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public void reindex() {
        // do nothing
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toString() {
        StringBuffer s;

        s = new StringBuffer();
        s.append("CONSTANT_Integer: Value = ");
        s.append(_intValue);

        return s.toString();
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toStringVerbose() {
        return toString();
    }

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _intValue;
    }

    /**
     * Compare this object and another one.
     *
     * @param obj other object
     *
     * @return true if the same
     */
    public boolean equals(Object obj) {
        return (obj instanceof IntegerPoolInfo) &&
            (((IntegerPoolInfo)obj)._intValue == _intValue);
    }

    /**
     * Execute a visitor.
     *
     * @param visitor visitor
     * @param data    visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public <R, D> R execute(IPoolInfoVisitor<R, D> visitor, D data) {
        return visitor.intCase(this, data);
    }
}