package sysModel.classFile.constantPool;

import sysModel.classFile.constantPool.visitors.IPoolInfoVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents a long in the constant pool.
 *
 * @author Mathias Ricken
 */
public class LongPoolInfo extends APoolInfo {
    /**
     * Data.
     */
    private long _longValue;

    /**
     * Constructor.
     *
     * @param l data
     */
    public LongPoolInfo(long l, ConstantPool cp) {
        super(CONSTANT_Long, cp);
        _longValue = l;
    }

    /**
     * Constructor reading from a stream.
     *
     * @param dis input stream
     * @param cp  constant pool
     *
     * @throws IOException
     */
    public LongPoolInfo(DataInputStream dis, ConstantPool cp) throws IOException {
        super(CONSTANT_Long, cp);
        _longValue = dis.readLong();
    }

    /**
     * Accessor for the data.
     *
     * @return data
     */
    public long getLongValue() {
        return _longValue;
    }

    /**
     * Mutator for the data
     *
     * @param longValue new data
     */
    public void setLongValue(long longValue) {
        _longValue = longValue;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        dos.writeByte(_type);
        dos.writeLong(_longValue);
    }

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public void resolve() {
        // do nothing
    }

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public void reindex() {
        // do nothing
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toString() {
        StringBuffer s;

        s = new StringBuffer();
        s.append("CONSTANT_Long: Value = ");
        s.append(_longValue);

        return s.toString();
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toStringVerbose() {
        return toString();
    }

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return new Long(_longValue).hashCode();
    }

    /**
     * Compare this object and another one.
     *
     * @param obj other object
     *
     * @return true if the same
     */
    public boolean equals(Object obj) {
        return (obj instanceof LongPoolInfo) &&
            (((LongPoolInfo)obj)._longValue == _longValue);
    }

    /**
     * Execute a visitor.
     *
     * @param visitor visitor
     * @param data    visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public <R, D> R execute(IPoolInfoVisitor<R, D> visitor, D data) {
        return visitor.longCase(this, data);
    }
}