package sysModel.classFile.constantPool;

import sysModel.classFile.constantPool.visitors.CheckUTFVisitor;
import sysModel.classFile.constantPool.visitors.IPoolInfoVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents a name-and-type object in the constant pool.
 *
 * @author Mathias Ricken
 */
public class NameAndTypePoolInfo extends APoolInfo {
    /**
     * Name information.
     */
    protected AUTFPoolInfo _name;

    /**
     * Type descriptor information.
     */
    protected AUTFPoolInfo _descriptor;

    /**
     * Name index.
     */
    protected short _nameIndex;

    /**
     * Type descriptor index.
     */
    protected short _descriptorIndex;

    /**
     * Constructor.
     *
     * @param name       name information
     * @param descriptor type descriptor information
     */
    public NameAndTypePoolInfo(AUTFPoolInfo name, AUTFPoolInfo descriptor, ConstantPool cp) {
        super(CONSTANT_NameAndType, cp);
        _name = name;
        _descriptor = descriptor;
        reindex();
    }

    /**
     * Constructor reading from a stream.
     *
     * @param dis input stream
     * @param cp  constant pool
     *
     * @throws IOException
     */
    public NameAndTypePoolInfo(DataInputStream dis, ConstantPool cp) throws IOException {
        super(CONSTANT_NameAndType, cp);
        _nameIndex = dis.readShort();
        _descriptorIndex = dis.readShort();
    }

    /**
     * Accessor for the name information.
     *
     * @return name information
     */
    public AUTFPoolInfo getName() {
        return _name;
    }

    /**
     * Mutator for the name information.
     *
     * @param name new name information
     */
    public void setName(AUTFPoolInfo name) {
        _name = name;
    }

    /**
     * Accessor for the type descriptor information.
     *
     * @return type descriptor information
     */
    public AUTFPoolInfo getDescriptor() {
        return _descriptor;
    }

    /**
     * Mutator for the type descriptor information.
     *
     * @param descriptor type descriptor information
     */
    public void setDescriptor(AUTFPoolInfo descriptor) {
        _descriptor = descriptor;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        reindex();
        dos.writeByte(_type);
        dos.writeShort(_nameIndex);
        dos.writeShort(_descriptorIndex);
    }

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public void resolve() {
        _name = _constantPool.get(_nameIndex).execute(CheckUTFVisitor.singleton(), null);
        _descriptor = _constantPool.get(_descriptorIndex).execute(CheckUTFVisitor.singleton(), null);
    }

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public void reindex() {
        _nameIndex = _constantPool.indexOf(_name);
        _descriptorIndex = _constantPool.indexOf(_descriptor);
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toString() {
        StringBuffer s;

        s = new StringBuffer();
        s.append("NAME-AND-TYPE: Name = #");
        s.append(_nameIndex);
        s.append(", Descriptor = #");
        s.append(_descriptorIndex);

        return s.toString();
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toStringVerbose() {
        return toString();
    }

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _nameIndex + 6101 * _descriptorIndex;
    }

    /**
     * Compare this object and another one.
     *
     * @param obj other object
     *
     * @return true if the same
     */
    public boolean equals(Object obj) {
        return (obj instanceof NameAndTypePoolInfo) && (
            (((NameAndTypePoolInfo)obj)._nameIndex == _nameIndex) ||
            (((NameAndTypePoolInfo)obj)._name == _name)
            ) && (
            (((NameAndTypePoolInfo)obj)._descriptorIndex == _descriptorIndex) ||
            (((NameAndTypePoolInfo)obj)._descriptor == _descriptor)
            );
    }

    /**
     * Execute a visitor.
     *
     * @param visitor visitor
     * @param data    visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public <R, D> R execute(IPoolInfoVisitor<R, D> visitor, D data) {
        return visitor.nameAndTypeCase(this, data);
    }
}