package sysModel.classFile.constantPool;

import sysModel.classFile.constantPool.visitors.CheckUTFVisitor;
import sysModel.classFile.constantPool.visitors.IPoolInfoVisitor;

import java.io.DataInputStream;
import java.io.DataOutputStream;
import java.io.IOException;

/**
 * Represents a string in the constant pool.
 *
 * @author Mathias Ricken
 */
public class StringPoolInfo extends APoolInfo {
    /**
     * Utf information.
     */
    private AUTFPoolInfo _utf;

    /**
     * Utf index.
     */
    private short _utfIndex;

    /**
     * Constructor.
     *
     * @param utf utf information
     */
    public StringPoolInfo(AUTFPoolInfo utf, ConstantPool cp) {
        super(CONSTANT_String, cp);
        _utf = utf;
        reindex();
    }

    /**
     * Constructor reading from a stream.
     *
     * @param dis input stream
     * @param cp  constant pool
     *
     * @throws IOException
     */
    public StringPoolInfo(DataInputStream dis, ConstantPool cp) throws IOException {
        super(CONSTANT_String, cp);
        _utfIndex = dis.readShort();
    }

    /**
     * Accessor for the utf information.
     *
     * @return utf information
     */
    public AUTFPoolInfo getUtf() {
        return _utf;
    }

    /**
     * Mutator for the utf information.
     *
     * @param utf new utf information
     */
    public void setUtf(AUTFPoolInfo utf) {
        _utf = utf;
    }

    /**
     * Write this constant pool object into the stream, including the type byte.
     *
     * @param dos stream
     *
     * @throws IOException
     */
    public void write(DataOutputStream dos) throws IOException {
        reindex();
        dos.writeByte(_type);
        dos.writeShort(_utfIndex);
    }

    /**
     * Resolve constant pool objects. This makes sure that the object links match the index links.
     */
    public void resolve() {
        _utf = _constantPool.get(_utfIndex).execute(CheckUTFVisitor.singleton(), null);
    }

    /**
     * Reindex constant pool indices. This makes sure the index links match the object links.
     */
    public void reindex() {
        _utfIndex = _constantPool.indexOf(_utf);
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toString() {
        StringBuffer s;

        s = new StringBuffer();
        s.append("CONSTANT_String: Name = #");
        s.append(_utfIndex);

        return s.toString();
    }

    /**
     * Return a human-readable version of this constant pool object.
     *
     * @return string
     */
    public String toStringVerbose() {
        return toString();
    }

    /**
     * Return a hash code.
     *
     * @return hash code
     */
    public int hashCode() {
        return _utfIndex;
    }

    /**
     * Compare this object and another one.
     *
     * @param obj other object
     *
     * @return true if the same
     */
    public boolean equals(Object obj) {
        return (obj instanceof StringPoolInfo) && (
            (((StringPoolInfo)obj)._utfIndex == _utfIndex) ||
            (((StringPoolInfo)obj)._utf == _utf)
            );
    }

    /**
     * Execute a visitor.
     *
     * @param visitor visitor
     * @param data    visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public <R, D> R execute(IPoolInfoVisitor<R, D> visitor, D data) {
        return visitor.stringCase(this, data);
    }
}