package sysModel.classFile.constantPool.visitors;

import sysModel.classFile.constantPool.*;

/**
 * Abstract visitor in which all value cases call a default value method, and all reference cases call a default
 * reference method.
 *
 * @author Mathias Ricken
 */
public abstract class AValueReferencePoolInfoVisitor <R, D> implements IPoolInfoVisitor<R, D> {
    /**
     * Abstract default reference case.
     *
     * @param host  referece constant pool info item
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public abstract R refCase(APoolInfo host, D param);

    /**
     * Abstract default value case.
     *
     * @param host  value constant pool info item
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public abstract R valueCase(APoolInfo host, D param);

    /**
     * Case for class info.
     *
     * @param host  class info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R classCase(ClassPoolInfo host, D param) {
        return refCase(host, param);
    }

    /**
     * Case for string info.
     *
     * @param host  string info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R stringCase(StringPoolInfo host, D param) {
        return refCase(host, param);
    }

    /**
     * Case for field info.
     *
     * @param host  field info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R fieldCase(FieldPoolInfo host, D param) {
        return refCase(host, param);
    }

    /**
     * Case for method info.
     *
     * @param host  method info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R methodCase(MethodPoolInfo host, D param) {
        return refCase(host, param);
    }

    /**
     * Case for interface method info.
     *
     * @param host  interface method info info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R interfaceMethodCase(InterfaceMethodPoolInfo host, D param) {
        return refCase(host, param);
    }

    /**
     * Case for name and type info.
     *
     * @param host  name and type info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R nameAndTypeCase(NameAndTypePoolInfo host, D param) {
        return refCase(host, param);
    }

    /**
     * Case for int info.
     *
     * @param host  int info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R intCase(IntegerPoolInfo host, D param) {
        return valueCase(host, param);
    }

    /**
     * Case for float info.
     *
     * @param host  float info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R floatCase(FloatPoolInfo host, D param) {
        return valueCase(host, param);
    }

    /**
     * Case for long info.
     *
     * @param host  long info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R longCase(LongPoolInfo host, D param) {
        return valueCase(host, param);
    }

    /**
     * Case for double info.
     *
     * @param host  double info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R doubleCase(DoublePoolInfo host, D param) {
        return valueCase(host, param);
    }

    /**
     * Case for asciiz info.
     *
     * @param host  asciiz info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R asciizCase(ASCIIPoolInfo host, D param) {
        return valueCase(host, param);
    }

    /**
     * Case for unicode info.
     *
     * @param host  unicode info
     * @param param visitor-specific parameter
     *
     * @return visitor-specific return value
     */
    public R unicodeCase(UnicodePoolInfo host, D param) {
        return valueCase(host, param);
    }
}
