package sysModel.env;

import controller.IScrollAdapter;
import controller.IDisplayAdapter;
import lrs.LRStruct;
import lrs.visitor.Apply;
import lrs.visitor.Clear;
import model.ILambda;
import sysModel.ICmdFactory;
import sysModel.ISecurityAdapter;
import sysModel.NoOpLambda;
import sysModel.fish.IFishFactory;
import sysModel.fish.AFish;
import sysModel.fish.FishException;
import sysModel.parser.Lexer;

import java.awt.*;
import java.awt.geom.AffineTransform;
import java.io.PrintWriter;

/**
 * Abstract global environment class.
 * <p/>
 * Note: All subclasses must provide a two-parameter constructor that takes in an ICmdFactory for dynamic access to
 * parse() and makeEnvFactory(), and an ISecurityAdapter.
 *
 * @author Mathias Ricken
 */
public abstract class AGlobalEnv {
    /**
     * Abstract bsae class of the environment local to a fish.
     *
     * @author Mathias Ricken
     */
    public abstract class ALocalEnv {
        /**
         * State.
         */
        ILocalEnvState _state = EmptyLocalEnvState.Singleton;

        /**
         * Attempt to move the fish forward, which may or may not be successful. The behavior in each case is defined by
         * the visitor: - If the move cannot be executed, the blockedCmd lambda is applied. The parameter is not used
         * and set to null. - If the move can be executed, the openCmd lambda is applied. The parameter is an ILambda
         * that can to be executed to actually move the fish to the target location of this move. The ILambda ignores
         * the input parameter and returns null.
         *
         * @param fish       AFish to move
         * @return return value of lambda executed
         * @param blockedCmd lambda to apply if blocked
         * @param openCmd    lambda to apply if open
         */
        public Object tryMoveFwd(AFish fish, final IBlockedCommand blockedCmd, final IOpenCommand openCmd) {
            // TODO: PART3
            return null;
        }

        /**
         * Attempt to breed the fish forward, which may or may not be successful. The behavior in each case is defined
         * by the visitor: - If the breeding cannot be executed, the blockedCmd lambda is applied. The parameter is not
         * used and set to null. - If the breeding can be executed, the openCmd lambda is applied. The parameter is an
         * ILambda that can to be executed to actually move the fish to the target location of this breeding. The
         * ILambda ignores the input parameter and returns null.
         *
         * @param fish       AFish to move
         * @return return value of lambda executed
         * @param blockedCmd lambda to apply if blocked
         * @param openCmd    lambda to apply if open
         */
        public Object tryBreedFwd(final AFish fish, final IBlockedCommand blockedCmd, final IOpenCommand openCmd) {
            // TODO: PART3
            return null;
        }

        /**
         * Draw the fish on the graphics object. The graphics object still has to be translated and rotated properly,
         *
         * @param fish AFish to drawFish
         * @param g    graphics object to drawFish on
         * @param comp component to drawFish on
         */
        public abstract void drawFish(AFish fish, Graphics2D g, Component comp);

        /**
         * Turn the fish radians to the right.
         *
         * @param fish    AFish to turn
         * @param radians radians to turn
         */
        public abstract void turnRight(AFish fish, double radians);

        /**
         * Remove the fish from the environment.
         *
         * @param fish AFish to remove
         */
        public void removeFish(AFish fish) {
            ILambda deleteLambda = AGlobalEnv.this.removeFish(this);

            // delete the fish from the simulation by executing the deleteLambda
            deleteLambda.apply(null);
        }

        /**
         * Execute a visitor on this local environment.
         *
         * @param visitor visitor to execute
         * @param param   visitor-specific parameter
         * @return visitor-specific return value
         */
        public Object execute(ILocalEnvVisitor visitor, Object param) {
            return _state.execute(this, visitor, param);
        }

        /**
         * Set state.
         *
         * @param state new state
         */
        public void setState(ILocalEnvState state) {
            _state = state;
        }

        /**
         * Factory method for a move lambda.
         *
         * @param le local environment for the target
         * @return move lambda to execute the move to the target
         */
        protected abstract ILambda makeMoveLambda(ALocalEnv le);

        /**
         * Make local environment in forward direction. Do not block yourself.
         *
         * @return new local environment in forward direction
         */
        protected abstract ALocalEnv makeMoveFwdLocalEnv();
    }

    /**
     * Lambda to execute a breed.
     */
    protected class BreedLambda implements ILambda {
        /// AFish to breed/
        private final AFish _fish;
        /// Target location.
        private final ALocalEnv _newLocalEnv;

        /**
         * Constructor.
         *
         * @param fish        fish to breed
         * @param newLocalEnv target location
         */
        public BreedLambda(AFish fish, ALocalEnv newLocalEnv) {
            _fish = fish;
            _newLocalEnv = newLocalEnv;
        }

        /**
         * Execute the breeding.
         *
         * @param param not used
         * @return null
         */
        public Object apply(Object param) {
            // execute the breeding
            AFish clone = (AFish) _fish.clone();
            ILambda addLambda = addFish(_newLocalEnv, clone);

            // add the fish to the simulation by executing the addLambda
            addLambda.apply(null);

            // deactivate all lambdas
            deactivateBreedLambdas();
            return null;
        }
    }

    /**
     * Color of the ocean.
     */
    protected static final Color OCEAN_BLUE = new Color(75, 75, 255);

    /**
     * List of move commands issued. The commands issued in a fish's step are recorded so that they can be deactivated
     * once a fish completes its step. This prevents a fish class from saving commands and executing them at a later
     * time.
     */
    protected LRStruct _moveLambdas;

    /**
     * List of breed commands issued. The commands issued in a fish's step are recorded so that they can be deactivated
     * once a fish completes its step. This prevents a fish class from saving commands and executing them at a later
     * time.
     */
    protected LRStruct _breedLambdas;

    /**
     * Command factory for lambdas given to the simulation driver.
     */
    protected ICmdFactory _cmdFactory;

    /**
     * Security manager that controls the actions of the fish.
     */
    protected ISecurityAdapter _securityAdapter;

    /**
     * Color of the water.
     */
    protected Color _waterColor = OCEAN_BLUE;

    /**
     * Visitor to opererate on a local environment. Note: should be protected, but is public for testing purposes
     */
    public interface ILocalEnvVisitor {
        /**
         * Case called if the location is empty.
         *
         * @param host  the location
         * @param param visitor-specific parameter
         * @return visitor-specific return value
         */
        public abstract Object emptyCase(ALocalEnv host, Object param);

        /**
         * Case called if the location is non-empty.
         *
         * @param host  the location
         * @param param visitor-specific parameter
         * @return visitor-specific return value
         */
        public abstract Object nonEmptyCase(ALocalEnv host, Object param);
    }

    /**
     * Construct a new abstract global environment.
     *
     * @param cmdFactory command factory to use
     * @param sm         security manager to control fish actions
     */
    public AGlobalEnv(ICmdFactory cmdFactory, ISecurityAdapter sm) {
        _breedLambdas = new LRStruct();
        _moveLambdas = new LRStruct();
        _cmdFactory = cmdFactory;
        _securityAdapter = sm;
    }

    /**
     * Deactivate move commands and clear list. If this is called after a fish has moved or after a step is complete, it
     * makes it impossible to use the commands at a later time.
     */
    protected void deactivateMoveLambdas() {
        _moveLambdas.execute(Apply.Singleton, new ILambda() {
            /**
             * Execute command.
             */
            public Object apply(Object param) {
                DeactivatableLambda cmd = (DeactivatableLambda) param;
                cmd.deactivate();
                return null;
            }
        });
        _moveLambdas.execute(Clear.Singleton, null);
    }

    /**
     * Deactivate move commands and clear list. If this is called after a fish has moved or after a step is complete, it
     * makes it impossible to use the commands at a later time.
     */
    protected void deactivateBreedLambdas() {
        _breedLambdas.execute(Apply.Singleton, new ILambda() {
            /**
             * Execute command.
             */
            public Object apply(Object param) {
                DeactivatableLambda cmd = (DeactivatableLambda) param;
                cmd.deactivate();
                return null;
            }
        });
        _breedLambdas.execute(Clear.Singleton, null);
    }

    /**
     * Edit a location.
     *
     * @param p           position
     * @param fishFactory factory to make a new fish
     * @param button
     * @return the lambda that the simulation control should execute for this action
     */
    public ILambda makeEditCmd(Point.Double p, final IFishFactory fishFactory, final int button) {
        final ALocalEnv localEnv = makeLocalEnv(p);
        return (ILambda) localEnv.execute(new ILocalEnvVisitor() {
            public Object emptyCase(ALocalEnv host, Object param) {
                // location is empty, add fish
                try {
                    return addFish(localEnv, fishFactory.createFish());
                }
                catch (Exception e) {
                    System.out.println("Factory throws: " + e);
                    e.printStackTrace();
                    return NoOpLambda.instance();
                }
            }

            public Object nonEmptyCase(ALocalEnv host, Object param) {
                // location is not empty
                return editFish(host, fishFactory, button);
            }
        }, null);
    }

    /**
     * Add the fish to the global environment.
     *
     * @param localEnv local environment
     * @param fish     fish to add
     * @return add lambda
     */
    protected ILambda addFish(ALocalEnv localEnv, AFish fish) {
        // prepare the fish
        localEnv.setState(NonEmptyLocalEnvState.Singleton);
        fish.setLocalEnvironment(localEnv);

        // add the fish to the environment's data
        addFishToInternalData(localEnv, fish);

        // set up the lambda to make the simulation control add the fish to the engine
        return _cmdFactory.makeAddCmd(fish);
    }

    /**
     * Remove the fish from the global environment.
     *
     * @param localEnv local environment
     * @return delete lambda
     */
    protected ILambda removeFish(ALocalEnv localEnv) {
        // change the state of the local environment
        localEnv.setState(EmptyLocalEnvState.Singleton);

        // remove the fish to the environment's data
        removeFishFromInternalData(localEnv);

        // set up the lambda to make the simulation control delete the fish from the engine
        return _cmdFactory.makeDeleteCmd(localEnv);
    }

    /**
     * Draw the environment in the region requested in model coordinates.
     *
     * @param g2   the Graphics2D object to use to render
     * @param comp the component to makeDrawCmd on
     * @param p1   top left corner
     * @param p2   bottom right corner
     * @return lambda for the simulation control to execute
     */
    public ILambda makeDrawCmd(final Graphics2D g2, final Component comp, final Point.Double p1, Point.Double p2) {
        // makeDrawCmd background
        drawBackground(g2, p1, p2);

        // tell the simulation control to makeDrawCmd all fish
        return _cmdFactory.makeNotifyCmd(new ILambda() {
            public Object apply(Object param) {
                // save the old transform
                AffineTransform oldTransform = g2.getTransform();

                // trams;ate view for drawing fish
                g2.translate(-(int) Math.floor(p1.x), -(int) Math.floor(p1.y));

                // makeDrawCmd the fish
                ((FishApplyParams) param).fish().draw(g2, comp);

                // change the old transform
                g2.setTransform(oldTransform);
                return null;
            }
        });
    }

    /**
     * Draw ocean background.
     *
     * @param g2 graphics object
     * @param p1 top left corner
     * @param p2 bottom right corner
     */
    protected void drawBackground(Graphics2D g2, Point.Double p1, Point.Double p2) {
        g2.setColor(_waterColor);  // fill blue background
        g2.fillRect(0, 0, (int) (p2.x - p1.x), (int) (p2.y - p1.y));

        // makeDrawCmd grid lines
        g2.setColor(Color.black);
        for (int y = 0; y <= (p2.y - p1.y); y++) {
            g2.drawLine(0, y, (int) (p2.x - p1.x), y);
        }
        for (int x = 0; x <= (p2.x - p1.x); x++) {
            g2.drawLine(x, 0, x, (int) (p2.y - p1.y));
        }
    }

    /**
     * Make the step command for the simulation driver to execute.
     *
     * @return step command
     */
    public ILambda makeStepCmd() {
        return _cmdFactory.makeNotifyCmd(new ILambda() {
            public Object apply(final Object param) {
                deactivateBreedLambdas();
                deactivateMoveLambdas();
                try {
                    Thread thread;
                    final RuntimeException potentialInternalException = new RuntimeException("Fish thread threw exception");
                    final DeactivatableLambda internalExceptionLambda = new DeactivatableLambda(new ILambda() {
                        public Object apply(Object param) {
                            throw potentialInternalException;
                        }
                    });
                    final RuntimeException potentialTimeOutException = new RuntimeException("Fish step timed out");
                    final DeactivatableLambda timeOutExceptionLambda = new DeactivatableLambda(new ILambda() {
                        @SuppressWarnings("deprecation")                      
                        public Object apply(Object param) {
                            _securityAdapter.setProtected(false);
                            // NOTE: Java issues a warning that the use of Thread.stop() is deprecated.
                            // This is the only way to kill a fish with divergent behavior, though.
                            // Please ignore this warning.
                            ((Thread)param).stop();
                            throw potentialTimeOutException;
                        }
                    });
                    thread = new Thread(_securityAdapter.getFishThreadGroup(), new Runnable() {
                        public void run() {
                            _securityAdapter.setProtected(true);
                            try {
                                ((FishApplyParams) param).fish().act();
                            }
                            finally {
                                _securityAdapter.setProtected(false);
                                timeOutExceptionLambda.deactivate();
                            }
                            internalExceptionLambda.deactivate();
                        }
                    });
                    thread.setUncaughtExceptionHandler(new Thread.UncaughtExceptionHandler() {
                        public void uncaughtException(Thread t, Throwable e) {
                            potentialInternalException.initCause(e);
                        }
                    });
                    thread.start();
                    thread.join(5000);
                    timeOutExceptionLambda.apply(thread);
                    internalExceptionLambda.apply(null);
                }
                catch (Throwable t) {
                    // since the fish is going to die anyway, it can't use the lambdas anyway
                    // we don't have to deactivate the lambdas
                    _breedLambdas = new LRStruct();
                    _moveLambdas = new LRStruct();

                    Throwable d = t;
                    while ((null != d)&&(null != d.getCause())) {
                        d = d.getCause();
                    }
                    _securityAdapter.handleException(d);

                    throw new FishException(t);
                }

                return null;
            }
        });
    }

    /**
     * Save an environment to a file.
     *
     * @param pw PrintWriter to use for output
     * @return lambda for the simulation control to execute
     */
    public ILambda save(final PrintWriter pw) {
        // print header
        printHeader(pw);

        // tell the simulation control to to save all the fish
        return new ILambda() {
            public Object apply(Object param) {
                pw.println(param);
                return null;
            }
        };
    }

    /**
     * Factory method for parsing a stream of tokens and creating a global environment from it.
     *
     * @param l lexer to use
     * @return new global environment
     */
    public AGlobalEnv parse(Lexer l) {
        AGlobalEnv env = parseEnvironment(l);
        env.parseFish(l);
        return env;
    }

    /* Abstract methods below. */

    /**
     * Edit the fish.
     *
     * @param localEnv    local environment
     * @param fishFactory
     * @param button
     * @return lambda for the simulation control to execute
     */
    protected abstract ILambda editFish(ALocalEnv localEnv, IFishFactory fishFactory, int button);

    /**
     * Add the fish to the environment's internal data structure.
     *
     * @param localEnv local environment
     * @param fish     fish to add
     */
    protected abstract void addFishToInternalData(ALocalEnv localEnv, AFish fish);

    /**
     * Remove the fish to the environment's internal data structure.
     *
     * @param localEnv local environment
     */
    protected abstract void removeFishFromInternalData(ALocalEnv localEnv);

    /**
     * Factory method for parsing the parameters for an environment and then creating it.
     *
     * @param l lexer
     * @return new global environment
     */
    protected abstract AGlobalEnv parseEnvironment(Lexer l);

    /**
     * Parse fish and add them to the environment.
     *
     * @param l parser to read from
     */
    protected abstract void parseFish(Lexer l);

    /**
     * Create a local environment for the position.
     *
     * @param p position
     * @return local environment
     */
    protected abstract ALocalEnv makeLocalEnv(Point.Double p);

    /**
     * Create the environment settings class for an environment. Factory method.
     *
     * @return environment settings class
     */
    public abstract AEnvFactory makeEnvFactory();

    /**
     * Print file header.
     *
     * @param pw PrintWriter to use
     */
    protected abstract void printHeader(PrintWriter pw);

    /**
     * Get size of the display.
     *
     * @return size of the display in model coordinate units.
     */
    public abstract Dimension getDisplaySize();

    /**
     * The action to be executed if the display should return home.
     *
     * @param sa scroll adapter
     */
    public abstract void returnHome(IScrollAdapter sa);

    /**
     * Ask the model where to scroll, given where the user has scrolled. If the environment just acts like a normal
     * panal, it should return pos without modification. If the environment recenters, it should return a position in
     * the middle of the pan area. All coordinates are in model coordinate units.
     *
     * @param pos position where the user scrolled to
     * @return position where the environment wants the view to be
     * @see IDisplayAdapter#getPanDelta
     */
    public abstract Point.Double getViewPosition(Point.Double pos);

    /**
     * Ask the model how much to pan, given where the user scrolled. If the environment just acts like a normal panal,
     * it should return (0,0). If the environment recenters, it should return delta without modification. All
     * coordinates are in model coordinate units.
     *
     * @param delta how far the user scrolled
     * @return how far the panel should scroll
     * @see IDisplayAdapter#getViewPosition
     */
    public abstract Point.Double getPanDelta(Point.Double delta);

    /**
     * Get a tool tip description for a specific place in the environment.
     *
     * @param p mouse coordinates
     * @return lambda for the simulation controller to execute. Must return the tooltip string.
     */
    public abstract ILambda getToolTipText(Point.Double p);
}
