package sysModel.env;

import model.ILambda;
import model.RandNumGenerator;
import sysModel.ICmdFactory;
import sysModel.ISecurityAdapter;
import sysModel.NoOpLambda;
import sysModel.fish.AFish;
import sysModel.fish.IFishFactory;
import sysModel.parser.DefaultTokenVisitor;
import sysModel.parser.Lexer;
import sysModel.parser.ParserException;

import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.lang.reflect.Constructor;
import java.util.Random;

/**
 * Abstract square environment class.
 *
 * @author Mathias Ricken
 */
public abstract class ASquareEnv extends AGlobalEnv {
    /**
     * Constructor.
     *
     * @param cmdFactory command factory to use
     * @param sm         security manager to control fish actions
     */
    public ASquareEnv(ICmdFactory cmdFactory, ISecurityAdapter sm) {
        super(cmdFactory, sm);
    }

    /**
     * Edit the fish.
     *
     * @param le          local environment
     * @param fishFactory
     * @param button
     * @return lambda to edit fish
     */
    public ILambda editFish(ALocalEnv le, final IFishFactory fishFactory, int button) {
        // by default, the control does not need to do anything
        ILambda lambdaForControl = NoOpLambda.instance();

        // rotate the fish
        final ASquareLocalEnvironment localEnv = (ASquareLocalEnvironment) le;
        double initialAngle = localEnv.direction().getAngle();
        localEnv.direction().turnRight(Math.PI / 2);

        // check if the fish should be removed
        if (initialAngle > localEnv.direction().getAngle()) {
            // rotated back to initial position, remove

            // set up lambda for simulation control
            lambdaForControl = _cmdFactory.makeDeleteCmd(localEnv);

            // and remove it from the environment's data
            ILambda deleteLambda = removeFish(le);

            // delete the fish from the simulation by executing the deleteLambda
            deleteLambda.apply(null);
        }

        return lambdaForControl;
    }

    /**
     * Create a local environment for the position.
     *
     * @param p position
     * @return local environment
     */
    public ALocalEnv makeLocalEnv(Point.Double p) {
        return makeLocalEnv(makeLocation(p.getX(), p.getY()), makeDirection());

    }

    /**
     * Create a local environment for the position.
     *
     * @param loc location
     * @param dir direction
     * @return local environment
     */
    protected abstract ASquareLocalEnvironment makeLocalEnv(Location loc, Direction dir);

    /**
     * Parse fish and add them to the environment.
     *
     * @param l parser to read from
     */
    protected void parseFish(final Lexer l) {
        while (l.nextToken().execute(new DefaultTokenVisitor() {
            public Object defaultCase() {
                throw new ParserException("Invalid token");
            }

            public Object endCase() {
                // end of stream
                // return false
                return Boolean.FALSE;
            }

            public Object wordCase(String className) {
                // read class name
                try {
                    // NOTE: Introduced class loader here
                    Class fishClass = null;
                    // Note: DrJava incompatibility.
                    fishClass = _securityAdapter.getClassLoader().loadClass(className);
                    Constructor fishCtor = fishClass.getConstructor(new Class[]{Color.class});
                    Random rng = RandNumGenerator.instance();
                    AFish fish = (AFish) fishCtor.newInstance(new Object[]{new Color(rng.nextInt(256), rng.nextInt(256), rng.nextInt(256))});

                    // read location
                    Location loc = new Location().parse(l);
                    Direction dir = new Direction().parse(l);
                    ASquareLocalEnvironment localEnv = makeLocalEnv(loc, dir);
                    ILambda addLambda = addFish(localEnv, fish);
                    addLambda.apply(null);

                    return Boolean.TRUE;
                }
                catch (Exception e) {
                    e.printStackTrace();
                    throw new ParserException(e.toString(),e);
                }
            }
        }) == Boolean.TRUE) {
        }
    }

    /**
     * Get a tool tip description for a specific place in the environment.
     *
     * @param p mouse coordinates
     * @return lambda for the simulation controller to execute. Must return the tooltip string.
     */
    public ILambda getToolTipText(final Point.Double p) {
        final ASquareLocalEnvironment l = (ASquareLocalEnvironment) makeLocalEnv(p);
        return ((ILambda) l.execute(new ILocalEnvVisitor() {
            public Object emptyCase(ALocalEnv host, Object param) {
                // this field is empty

                // return an ILambda that returns the string for an empty field
                return new ILambda() {
                    public Object apply(Object param) {
                        return "(" + (int) Math.floor(p.x) + ',' + (int) Math.floor(p.y) + ") is empty";
                    }
                };
            }

            public Object nonEmptyCase(ALocalEnv host, Object param) {
                // this field is occupied

                // return an ILambda that returns the string for an occupied
                return new ILambda() {
                    String _fishName;
                    ALocalEnv _localEnv;

                    public Object apply(Object param) {
                        _fishName = "";
                        _cmdFactory.makeNotifyCmd(new ILambda() {
                            public Object apply(Object param) {
                                FishApplyParams fap = (FishApplyParams) param;
                                if (((ASquareLocalEnvironment) fap.localEnv()).location().inField(l.location())) {
                                    _fishName = fap.fish().toString();
                                    _localEnv = fap.localEnv();
                                }
                                return null;
                            }
                        }).apply(null);

                        return _fishName + " at " + _localEnv;
                    }
                };
            }
        }, null));
    }

    /**
     * Factory method for Direction.
     *
     * @return new Direction facing up (0,-1)
     */
    public Direction makeDirection() {
        return new Direction();
    }

    /**
     * Factory method for Direction.
     *
     * @param dx delta x
     * @param dy delta y
     * @return new Direction facing (dx, dy)
     */
    public Direction makeDirection(double dx, double dy) {
        return new Direction(dx, dy);
    }

    /**
     * Factory method for Direction.
     *
     * @param other other direction
     * @return new Direction facing in the same direction as other
     */
    public Direction makeDirection(Direction other) {
        return new Direction(other);
    }

    /**
     * Concrete direction class.
     */
    public class Direction {
        /// floating point epsilon for comparisons
        private final double EPSILON = 1e-10;

        /**
         * Direction delta x.
         */
        private double _dx;

        /**
         * Direction delta y.
         */
        private double _dy;

        /**
         * Return the direction delta x.
         *
         * @return double direction delta x
         */
        public double getDeltaX() {
            return _dx;
        }

        /**
         * Return the direction delta y.
         *
         * @return double direction delta y
         */
        public double getDeltaY() {
            return _dy;
        }

        /**
         * Return a new object which has the same direction.
         *
         * @return new object with same direction
         */
        public Direction duplicate() {
            return makeDirection(_dx, _dy);
        }

        /**
         * Reverse this direction.
         */
        public void reverseDirection() {
            _dx = -_dx;
            _dy = -_dy;
        }

        /**
         * Return true of this direction is the same as the other.
         *
         * @param other other direction
         * @return true if the directions are the same
         */
        public boolean same(Direction other) {
            double diffx = _dx - other.getDeltaX();
            double diffy = _dy - other.getDeltaY();
            return (EPSILON > Math.abs(diffx) && EPSILON > Math.abs(diffy));
        }

        /**
         * Turn this direction to the left.
         *
         * @param radians radians to turn
         */
        public void turnLeft(double radians) {
            turnRight(-radians);
        }

        /**
         * Turn this direction PI/2 radians to the left.
         */
        public void turnLeft() {
            turnLeft(Math.PI / 2);
        }

        /**
         * Turn this direction to the right.
         *
         * @param radians radians to turn
         */
        public void turnRight(double radians) {
            double dx = _dx * Math.cos(radians) - _dy * Math.sin(radians);
            double dy = _dx * Math.sin(radians) + _dy * Math.cos(radians);
            if (EPSILON > Math.abs(dx)) {
                _dx = 0;
            }
            else {
                _dx = dx;
            }
            if (EPSILON > Math.abs(dy)) {
                _dy = 0;
            }
            else {
                _dy = dy;
            }
        }

        /**
         * Turn this direction PI/2 radians to the right.
         */
        public void turnRight() {
            turnRight(Math.PI / 2);
        }

        /**
         * Constructor.
         *
         * @param other other direction
         */
        public Direction(Direction other) {
            _dx = other.getDeltaX();
            _dy = other.getDeltaY();
        }

        /**
         * Constructor.
         *
         * @param dx delta x
         * @param dy delta y
         */
        public Direction(double dx, double dy) {
            _dx = dx;
            _dy = dy;
        }

        /**
         * Creates a new direction facing north.
         */
        public Direction() {
            _dx = 0;
            _dy = -1;
        }

        /**
         * Overridden toString method.
         *
         * @return string representation
         */
        public String toString() {
            return "(" + _dx + ", " + _dy + ')';
        }

        /**
         * Parses a direction.
         *
         * @param l parser to read from
         * @return parsed direction
         */
        public Direction parse(final Lexer l) {
            // read (
            return (Direction) l.nextToken().execute(new DefaultTokenVisitor() {
                public Object defaultCase() {
                    throw new ParserException("Invalid token");
                }

                public Object openCase() {
                    // read x
                    return l.nextToken().execute(new DefaultTokenVisitor() {
                        public Object defaultCase() {
                            throw new ParserException("Invalid token");
                        }

                        public Object numCase(final double x) {
                            // read ,
                            return l.nextToken().execute(new DefaultTokenVisitor() {
                                public Object defaultCase() {
                                    throw new ParserException("Invalid token");
                                }

                                public Object commaCase() {
                                    // read y
                                    return l.nextToken().execute(new DefaultTokenVisitor() {
                                        public Object defaultCase() {
                                            throw new ParserException("Invalid token");
                                        }

                                        public Object numCase(final double y) {
                                            // read )
                                            return l.nextToken().execute(new DefaultTokenVisitor() {
                                                public Object defaultCase() {
                                                    throw new ParserException("Invalid token");
                                                }

                                                public Object closeCase() {
                                                    return makeDirection(x, y);
                                                }
                                            });
                                        }
                                    });
                                }
                            });
                        }
                    });
                }
            });
        }

        /**
         * Rotate the graphics object by the angle between (0,-1) and this direction.
         *
         * @param g graphics object to rotate
         */
        public void rotateGraphics(Graphics2D g) {
            double theta = getAngle();

            if (null != g) {
                g.rotate(theta);
            }
        }

        /**
         * Return the angle between (0,-1) and this direction.
         *
         * @return angle in radians
         */
        public double getAngle() {
            /*
            Find the angle between the current direction dir and (1,0) in a clockwise direction.
            dir . (0, -1) = |dir|*|(0,-1)|*cos(theta). But |dir| = |(0, -1)| = 1, so
            cos(theta) = dir . (0, -1) = -dir.y
            This is always the smaller angle, though. Therefore
            theta = arccos(-dir.y)        if dir.x >= 0
                  = 2 PI - arccos(-dir.y) if dir.x < 0
            */
            double theta = Math.acos(-_dy);
            if (0 > _dx) {
                theta = 2 * Math.PI - theta;
            }
            if (2 * Math.PI == theta) {
                return 0;
            }
            else {
                return theta;
            }
        }
    }

    /**
     * Factory method for Location.
     *
     * @param x x coordinate
     * @param y y coordinate
     * @return new Location at (x, y)
     */
    public Location makeLocation(double x, double y) {
        return new Location(x, y);
    }

    /**
     * Concrete location class.
     *
     * @author Mathias G. Ricken
     */
    public class Location {
        /**
         * Column.
         */
        private double _x;

        /**
         * Row.
         */
        private double _y;

        /**
         * Return column.
         *
         * @return double column
         */
        public double getX() {
            return _x;
        }

        /**
         * Return row.
         *
         * @return double row
         */
        public double getY() {
            return _y;
        }

        /**
         * Set column.
         *
         * @param _x New column
         */
        public void setX(double _x) {
            this._x = _x;
        }

        /**
         * Set row.
         *
         * @param _y New row
         */
        public void setY(double _y) {
            this._y = _y;
        }

        /**
         * Constructor. Location is (0, 0)
         */
        public Location() {
            _x = 0;
            _y = 0;
        }

        /**
         * Constructor.
         *
         * @param x column
         * @param y row
         */
        public Location(double x, double y) {
            _x = x;
            _y = y;
        }

        /**
         * Return true of this location is the same as the other.
         *
         * @param other other location
         * @return true if the locations are the same
         */
        public boolean same(Location other) {
            return (_x == other.getX()) && (_y == other.getY());
        }

        /**
         * Return true of the other location is in the same field as this location.
         *
         * @param other location to compare to
         * @return true if the other point is in the same field
         */
        public boolean inField(Location other) {
            return (Math.floor(other.getX()) == Math.floor(getX()) && Math.floor(other.getY()) == Math.floor(getY()));
        }

        /**
         * Return the location of a neighbor in the given direction.
         *
         * @param dir the direction of the neighbor to be returned
         * @return neighbor in that direction
         */
        public Location getNeighbor(Direction dir) {
            return makeLocation(_x + dir.getDeltaX(), _y + dir.getDeltaY());
        }

        /**
         * Overridden toString method.
         *
         * @return string representation
         */
        public String toString() {
            return "(" + (int) Math.floor(_x) + ", " + (int) Math.floor(_y) + ')';
        }

        /**
         * Parses a location.
         *
         * @param l parser to read from
         * @return parsed location
         */
        public Location parse(final Lexer l) {
            // read (
            return (Location) l.nextToken().execute(new DefaultTokenVisitor() {
                public Object defaultCase() {
                    throw new ParserException("Invalid token");
                }

                public Object openCase() {
                    // read x
                    return l.nextToken().execute(new DefaultTokenVisitor() {
                        public Object defaultCase() {
                            throw new ParserException("Invalid token");
                        }

                        public Object numCase(final double x) {
                            // read ,
                            return l.nextToken().execute(new DefaultTokenVisitor() {
                                public Object defaultCase() {
                                    throw new ParserException("Invalid token");
                                }

                                public Object commaCase() {
                                    // read y
                                    return l.nextToken().execute(new DefaultTokenVisitor() {
                                        public Object defaultCase() {
                                            throw new ParserException("Invalid token");
                                        }

                                        public Object numCase(final double y) {
                                            // read )
                                            return l.nextToken().execute(new DefaultTokenVisitor() {
                                                public Object defaultCase() {
                                                    throw new ParserException("Invalid token");
                                                }

                                                public Object closeCase() {
                                                    return makeLocation(x, y);
                                                }
                                            });
                                        }
                                    });
                                }
                            });
                        }
                    });
                }
            });
        }
    }

    /**
     * Concrete local environment for the square environment.
     */
    protected abstract class ASquareLocalEnvironment extends ALocalEnv {

        /**
         * Accessor for the location.
         *
         * @return location
         */
        public abstract Location location();

        /**
         * Accessor for the direction.
         *
         * @return direction
         */
        public abstract Direction direction();

        /**
         * Draw the fish on the graphics object. The graphics object still has to be translated and rotated properly,
         *
         * @param fish AFish to drawFish
         * @param g    graphics object to drawFish on
         * @param comp component to drawFish on
         */
        public void drawFish(AFish fish, Graphics2D g, Component comp) {
            double centerX = Math.floor(location().getX()) + 1.0 / 2;
            double centerY = Math.floor(location().getY()) + 1.0 / 2;

            // save transformation
            AffineTransform oldTransform = g.getTransform();
            // translate to center of field
            g.translate(centerX, centerY);

            // set up the correct rotation
            direction().rotateGraphics(g);

            // makeDrawCmd the fish
            fish.paint(g, comp);

            // restore transformation
            g.setTransform(oldTransform);
        }

        /**
         * Turn the fish radians to the right.
         *
         * @param fish    AFish to turn
         * @param radians radians to turn
         */
        public void turnRight(AFish fish, double radians) {
            direction().turnRight(radians);
        }

        /**
         * String representation of the local environment. Should be "(x, y) (dx, dy)".
         *
         * @return string representation
         */
        public String toString() {
            return location() + " " + direction();
        }
    }
}
