package sysModel.env;

import controller.IScrollAdapter;
import controller.IDisplayAdapter;
import junit.framework.TestCase;
import lrs.IAlgo;
import lrs.LRStruct;
import lrs.visitor.GetLength;
import lrs.visitor.Remove;
import model.ILambda;
import model.RandNumGenerator;
import model.fish.GenericFish;
import sysModel.ICmdFactory;
import sysModel.ISecurityAdapter;
import sysModel.NoOpLambda;
import sysModel.fish.IFishFactory;
import sysModel.fish.AFish;
import sysModel.parser.DefaultTokenVisitor;
import sysModel.parser.Lexer;
import sysModel.parser.ParserException;

import javax.swing.*;
import java.awt.*;
import java.awt.event.MouseEvent;
import java.awt.geom.AffineTransform;
import java.lang.reflect.Constructor;
import java.util.LinkedList;
import java.util.Random;
import java.io.PrintWriter;

/**
 * An environment that does not use grids to place fish.
 *
 * @author Mathias Ricken
 */
public class NoGridEnv extends AGlobalEnv {
    /**
     * Concrete direction class.
     */
    public static class Direction {
        /// floating point epsilon for comparisons
        private static double EPSILON = 1e-10;

        /**
         * Direction delta x.
         */
        private double _dx;

        /**
         * Direction delta y.
         */
        private double _dy;

        /**
         * Return the direction delta x.
         *
         * @return double direction delta x
         */
        public double getDeltaX() {
            return _dx;
        }

        /**
         * Return the direction delta y.
         *
         * @return double direction delta y
         */
        public double getDeltaY() {
            return _dy;
        }

        /**
         * Return a new object which has the same direction.
         *
         * @return new object with same direction
         */
        public Direction duplicate() {
            return new Direction(_dx, _dy);
        }

        /**
         * Reverse this direction.
         */
        public void reverseDirection() {
            _dx = -_dx;
            _dy = -_dy;
        }

        /**
         * Return true of this direction is the same as the other.
         *
         * @param other other direction
         * @return true if the directions are the same
         */
        public boolean same(Direction other) {
            double diffx = _dx - other.getDeltaX();
            double diffy = _dy - other.getDeltaY();
            return (Math.abs(diffx) < EPSILON && Math.abs(diffy) < EPSILON);
        }

        /**
         * Turn this direction to the left.
         *
         * @param radians radians to turn
         */
        public void turnLeft(double radians) {
            turnRight(-radians);
        }

        /**
         * Turn this direction PI/2 radians to the left.
         */
        public void turnLeft() {
            turnLeft(Math.PI / 2);
        }

        /**
         * Turn this direction to the right.
         *
         * @param radians radians to turn
         */
        public void turnRight(double radians) {
            double dx = _dx * Math.cos(radians) - _dy * Math.sin(radians);
            double dy = _dx * Math.sin(radians) + _dy * Math.cos(radians);
            _dx = dx;
            _dy = dy;
        }

        /**
         * Turn this direction PI/2 radians to the right.
         */
        public void turnRight() {
            turnRight(Math.PI / 2);
        }

        /**
         * Constructor.
         *
         * @param dx delta x
         * @param dy delta y
         */
        public Direction(double dx, double dy) {
            _dx = dx;
            _dy = dy;
        }

        /**
         * Copy constructor.
         *
         * @param other other direction
         */
        public Direction(Direction other) {
            _dx = other.getDeltaX();
            _dy = other.getDeltaY();
        }

        /**
         * Creates a new direction facing north.
         */
        public Direction() {
            _dx = 0;
            _dy = -1;
        }

        /**
         * Overridden toString method.
         *
         * @return string representation
         */
        public String toString() {
            return "(" + _dx + ", " + _dy + ')';
        }

        /**
         * Parses a direction.
         *
         * @param l parser to read from
         * @return parsed direction
         */
        public static Direction parse(final Lexer l) {
            // read (
            return (Direction) l.nextToken().execute(new DefaultTokenVisitor() {
                public Object defaultCase() {
                    throw new ParserException("Invalid token");
                }

                public Object openCase() {
                    // read x
                    return l.nextToken().execute(new DefaultTokenVisitor() {
                        public Object defaultCase() {
                            throw new ParserException("Invalid token");
                        }

                        public Object numCase(final double x) {
                            // read ,
                            return l.nextToken().execute(new DefaultTokenVisitor() {
                                public Object defaultCase() {
                                    throw new ParserException("Invalid token");
                                }

                                public Object commaCase() {
                                    // read y
                                    return l.nextToken().execute(new DefaultTokenVisitor() {
                                        public Object defaultCase() {
                                            throw new ParserException("Invalid token");
                                        }

                                        public Object numCase(final double y) {
                                            // read )
                                            return l.nextToken().execute(new DefaultTokenVisitor() {
                                                public Object defaultCase() {
                                                    throw new ParserException("Invalid token");
                                                }

                                                public Object closeCase() {
                                                    return new Direction(x, y);
                                                }
                                            });
                                        }
                                    });
                                }
                            });
                        }
                    });
                }
            });
        }

        /**
         * Rotate the supplied Graphics2D object to match this direction.
         *
         * @param g graphics object to rotate
         */
        public void rotateGraphics(Graphics2D g) {
            /*
            Find the angle between the current direction dir and (1,0) in a clockwise direction.
            dir . (0, -1) = |dir|*|(0,-1)|*cos(theta). But |dir| = |(0, -1)| = 1, so
            cos(theta) = dir . (0, -1) = -dir.y
            This is always the smaller angle, though. Therefore
            theta = arccos(-dir.y)        if dir.x >= 0
                  = 2 PI - arccos(-dir.y) if dir.x < 0
            */
            double theta = getAngle();

            if (null != g) {
                g.rotate(theta);
            }
        }

        /**
         * Return the angle between (0,-1) and this direction.
         *
         * @return angle in radians
         */
        public double getAngle() {
            // keeping argument to acos in range [-1,1] because acos produced a NaN on one system
            double theta = Math.acos(Math.max(-1,Math.min(1,-_dy)));
            if (0 > _dx) {
                theta = 2 * Math.PI - theta;
            }
            return theta;
        }
    }

    /**
     * Concrete location class.
     *
     * @author Mathias G. Ricken
     */
    public static class Location {
        /**
         * Column.
         */
        private double _x;

        /**
         * Row.
         */
        private double _y;

        /**
         * Return column.
         *
         * @return double column
         */
        public double getX() {
            return _x;
        }

        /**
         * Return row.
         *
         * @return double row
         */
        public double getY() {
            return _y;
        }

        /**
         * Set column.
         *
         * @param _x New column
         */
        public void setX(double _x) {
            this._x = _x;
        }

        /**
         * Set row.
         *
         * @param _y New row
         */
        public void setY(double _y) {
            this._y = _y;
        }

        /**
         * Constructor.
         *
         * @param x column
         * @param y row
         */
        public Location(double x, double y) {
            _x = x;
            _y = y;
        }

        /**
         * Return true of this location is the same as the other.
         *
         * @param other other location
         * @return true if the locations are the same
         */
        public boolean same(Location other) {
            return (_x == other.getX()) &&
                    (_y == other.getY());
        }

        /**
         * Return true of the other location is in a square with this location at the center, facing in the given
         * direction, with sides of the specified length.
         *
         * @param other   location to compare to
         * @param forward direction the square is facing (this direction is perpendicular to two of the side walls)
         * @param side    side length
         * @return true if the other point is inside
         */
        public boolean inSquare(Location other, Direction forward, double side) {
            // distante center-to-other
            double deltaX = other.getX() - _x;
            double deltaY = other.getY() - _y;

            // point at center front of square
            double frontX = forward.getDeltaX();
            double frontY = forward.getDeltaY();

            // project center-to-other onto center-to-front
            double toFront = deltaX * frontX + deltaY * frontY;
            if (Math.abs(toFront) > side / 2) {
                // beyond front or back side
                return false;
            }

            // point at center right of square
            double rightX = -forward.getDeltaY();
            double rightY = forward.getDeltaX();

            // project center-to-other onto center-to-right
            double toRight = deltaX * rightX + deltaY * rightY;
            return Math.abs(toRight) <= side / 2;

        }

        /**
         * Return the location of a neighbor in the given direction.
         *
         * @param dir the direction of the neighbor to be returned
         * @return neighbor in that direction
         */
        public Location getNeighbor(Direction dir) {
            return new Location(_x + dir.getDeltaX(),
                                _y + dir.getDeltaY());
        }

        /**
         * Overridden toString method.
         *
         * @return string representation
         */
        public String toString() {
            return "(" + _x + ", " + _y + ')';
        }

        /**
         * Parses a location.
         *
         * @param l parser to read from
         * @return parsed location
         */
        public static Location parse(final Lexer l) {
            // read (
            return (Location) l.nextToken().execute(new DefaultTokenVisitor() {
                public Object defaultCase() {
                    throw new ParserException("Invalid token");
                }

                public Object openCase() {
                    // read x
                    return l.nextToken().execute(new DefaultTokenVisitor() {
                        public Object defaultCase() {
                            throw new ParserException("Invalid token");
                        }

                        public Object numCase(final double x) {
                            // read ,
                            return l.nextToken().execute(new DefaultTokenVisitor() {
                                public Object defaultCase() {
                                    throw new ParserException("Invalid token");
                                }

                                public Object commaCase() {
                                    // read y
                                    return l.nextToken().execute(new DefaultTokenVisitor() {
                                        public Object defaultCase() {
                                            throw new ParserException("Invalid token");
                                        }

                                        public Object numCase(final double y) {
                                            // read )
                                            return l.nextToken().execute(new DefaultTokenVisitor() {
                                                public Object defaultCase() {
                                                    throw new ParserException("Invalid token");
                                                }

                                                public Object closeCase() {
                                                    return new Location(x, y);
                                                }
                                            });
                                        }
                                    });
                                }
                            });
                        }
                    });
                }
            });
        }
    }

    /**
     * Concrete local environment for the square unbounded environment.
     */
    protected class LocalEnvironment extends ALocalEnv {
        /**
         * Location.
         */
        Location _loc;

        /**
         * Direction.
         */
        Direction _dir;

        /**
         * State.
         */
        ILocalEnvState _state = EmptyLocalEnvState.Singleton;

        /**
         * Lambda to execute a move.
         */
        private class MoveLambda implements ILambda {
            /// target direction
            private Direction _newDir;
            /// target location
            private Location _newLoc;

            /**
             * Constructor.
             *
             * @param le target local environment
             */
            public MoveLambda(LocalEnvironment le) {
                _newLoc = new Location(le._loc.getX(), le._loc.getY());
                _newDir = new Direction(le._dir);
            }

            /**
             * Execute the move.
             *
             * @param param not used
             * @return null
             */
            public Object apply(Object param) {
                // execute the movement
                _loc = _newLoc;
                _dir = _newDir;

                // deactivate all lambdas
                deactivateMoveLambdas();
                return null;
            }
        }

        /**
         * Construct a new local environment.
         *
         * @param loc location
         * @param dir direction
         */
        public LocalEnvironment(Location loc, Direction dir) {
            _loc = loc;
            _dir = dir;
        }

        /**
         * Accessor for the location.
         *
         * @return location
         */
        public Location location() {
            return _loc;
        }

        /**
         * Accessor for the direction.
         *
         * @return direction
         */
        public Direction direction() {
            return _dir;
        }

        /**
         * Make local environment in forward direction. Do not block yourself.
         *
         * @return new local environment in forward direction
         */
        protected ALocalEnv makeMoveFwdLocalEnv() {
            // remove this local environment to prevent collision with itself
            _localEnvList.execute(Remove.Singleton, this);
            ALocalEnv le = makeLocalEnv(_loc.getNeighbor(_dir), _dir);
            // add this local environment back in
            _localEnvList.insertFront(this);
            return le;
        }

        /**
         * Factory method for a move lambda.
         *
         * @param le local environment for the target
         * @return move lambda to execute the move to the target
         */
        protected ILambda makeMoveLambda(ALocalEnv le) {
            return new MoveLambda((LocalEnvironment) le);
        }

        /**
         * Draw the fish on the graphics object. The graphics object still has to be translated and rotated properly,
         *
         * @param fish AFish to drawFish
         * @param g    graphics object to drawFish on
         * @param comp component to drawFish on
         */
        public void drawFish(AFish fish, Graphics2D g, Component comp) {
            double centerX = _loc.getX();
            double centerY = _loc.getY();

            // save transformation
            AffineTransform oldTransform = g.getTransform();
            // translate to center of field
            g.translate(centerX, centerY);

            // set up the correct rotation
            _dir.rotateGraphics(g);

            // makeDrawCmd the fish
            fish.paint(g, comp);

            // restore transformation
            g.setTransform(oldTransform);
        }

        /**
         * Turn the fish radians to the right.
         *
         * @param fish    AFish to turn
         * @param radians radians to turn
         */
        public void turnRight(AFish fish, double radians) {
            _dir.turnRight(radians);
        }

        /**
         * String representation of the local environment. Should be "(x, y) (dx, dy)".
         *
         * @return string representation
         */
        public String toString() {
            return _loc.toString() + ' ' + _dir.toString();
        }
    }

    private static int PAN_SIZE = 2000;
    private static Point.Double PAN_CENTER = new Point.Double(PAN_SIZE / 2, PAN_SIZE / 2);

    /**
     * List of local environments in this global environment.
     */
    private LRStruct _localEnvList;

    /**
     * Number of steps for complete rotation.
     */
    private int _rotSteps;

    /**
     * Create a new environment without grids. Note: This constructor needs to exist and be public for the "environment
     * selection" dialog to work.
     *
     * @param cmdFactory command factory to use
     * @param sm         security manager to control fish actions
     */
    public NoGridEnv(ICmdFactory cmdFactory, ISecurityAdapter sm) {
        super(cmdFactory, sm);
        _localEnvList = new LRStruct();
    }

    /**
     * Create a new environment without grids.
     *
     * @param cmdFactory command factory to use
     * @param sm         security manager to control fish actions
     * @param rotSteps   rotation steps
     * @param waterColor color of the water
     */
    public NoGridEnv(ICmdFactory cmdFactory, ISecurityAdapter sm, int rotSteps, Color waterColor) {
        super(cmdFactory, sm);
        _localEnvList = new LRStruct();
        _rotSteps = rotSteps;
        _waterColor = waterColor;
    }

    /**
     * Add the fish to the global environment.
     *
     * @param localEnv local environment
     * @param fish     fish to add
     */
    protected void addFishToInternalData(ALocalEnv localEnv, AFish fish) {
        _localEnvList.insertFront(localEnv);
    }

    /**
     * Remove the fish from the global environment.
     *
     * @param localEnv local environment
     */
    protected void removeFishFromInternalData(ALocalEnv localEnv) {
        _localEnvList.execute(Remove.Singleton, localEnv);
    }

    /**
     * Create a local environment for the position.
     *
     * @param p position
     * @return local environment
     */
    public ALocalEnv makeLocalEnv(Point.Double p) {
        return makeLocalEnv(new Location(p.getX(), p.getY()), new Direction());
    }

    /**
     * Create a local environment for the position.
     *
     * @param loc location
     * @param dir direction
     * @return local environment
     */
    private ALocalEnv makeLocalEnv(final Location loc, final Direction dir) {
        return (ALocalEnv) _localEnvList.execute(new IAlgo() {
            /**
             * Operates on a non-empty LRStruct host, given an input object.
             *
             * @param host a non-empty LRStruct.
             * @param inp  input object needed by this IAlgo.
             * @return an appropriate output object.
             */
            public Object nonEmptyCase(LRStruct host, Object inp) {
                LocalEnvironment localEnv = (LocalEnvironment) host.getFirst();
                if (localEnv.location().inSquare(loc, localEnv.direction(), 1)) {
                    return localEnv;
                }
                else {
                    return host.getRest().execute(this, inp);
                }
            }

            /**
             * Operates on an empty LRStruct host, given an input object.
             *
             * @param host an empty LRStruct.
             * @param inp  input object needed by this IAlgo.
             * @return an appropriate output object.
             */
            public Object emptyCase(LRStruct host, Object inp) {
                return new LocalEnvironment(loc, dir);
            }
        }, null);
    }

    /**
     * Create a local environment with the given data.
     *
     * @param loc location
     * @param dir direction
     * @return new local environment
     */
    protected LocalEnvironment createLocalEnvironment(Location loc, Direction dir) {
        return new LocalEnvironment(loc, dir);
    }

    /**
     * Edit the fish.
     *
     * @param le          local environment
     * @param fishFactory
     * @param button
     * @return lambda to edit the fish
     */
    public ILambda editFish(ALocalEnv le, final IFishFactory fishFactory, int button) {
        // by default, the control does not need to do anything
        ILambda lambdaForControl = NoOpLambda.instance();

        // rotate the fish
        final LocalEnvironment localEnv = (LocalEnvironment) le;

        double initialAngle = localEnv.direction().getAngle();
        localEnv.direction().turnRight(2 * Math.PI / _rotSteps);

        // check if the fish should be removed
        if (initialAngle > localEnv.direction().getAngle()) {
            // rotated back to initial position, remove

            // set up lambda for simulation control
            lambdaForControl = _cmdFactory.makeDeleteCmd(localEnv);

            // and remove it from the environment's data
            ILambda deleteLambda = removeFish(le);

            // delete the fish from the simulation by executing the deleteLambda
            deleteLambda.apply(null);
        }

        return lambdaForControl;
    }

    /**
     * Factory method for parsing a stream of tokens and creating a global environment from it.
     *
     * @param l lexer to use
     * @return new global environment
     */
    protected AGlobalEnv parseEnvironment(final Lexer l) {
        // have to read number of steps
        return (AGlobalEnv) l.nextToken().execute(new DefaultTokenVisitor() {
            public Object defaultCase() {
                throw new ParserException("Invalid token");
            }

            public Object numCase(final double rotSteps) {
                // rotation steps was read
                return (AGlobalEnv) l.nextToken().execute(new DefaultTokenVisitor() {
                    public Object defaultCase() {
                        throw new ParserException("Invalid token");
                    }

                    public Object numCase(final double red) {
                        // red was read
                        return (AGlobalEnv) l.nextToken().execute(new DefaultTokenVisitor() {
                            public Object defaultCase() {
                                throw new ParserException("Invalid token");
                            }

                            public Object numCase(final double green) {
                                // green was read
                                return (AGlobalEnv) l.nextToken().execute(new DefaultTokenVisitor() {
                                    public Object defaultCase() {
                                        throw new ParserException("Invalid token");
                                    }

                                    public Object numCase(double blue) {
                                        // blue was read
                                        NoGridEnv env = new NoGridEnv(_cmdFactory, _securityAdapter, (int) rotSteps,
                                                                      new Color((int) red, (int) green, (int) blue));

                                        // parse fish
                                        env.parseFish(l);

                                        return env;
                                    }
                                });

                            }
                        });
                    }
                });
            }
        });

    }

    /**
     * Parse fish and add them to the environment.
     *
     * @param l parser to read from
     */
    protected void parseFish(final Lexer l) {
        while (l.nextToken().execute(new DefaultTokenVisitor() {
            public Object defaultCase() {
                throw new ParserException("Invalid token");
            }

            public Object endCase() {
                // end of stream
                // return false
                return Boolean.FALSE;
            }

            public Object wordCase(String className) {
                // read class name
                try {
                    // NOTE: Introduced class loader here
                    // Class fishClass = Class.forName(className);
                    Class fishClass = _securityAdapter.getClassLoader().loadClass(className);
                    Constructor fishCtor = fishClass.getConstructor(new Class[]{Color.class});
                    Random rng = RandNumGenerator.instance();
                    AFish fish = (AFish) fishCtor.newInstance(new Object[]{new Color(rng.nextInt(256), rng.nextInt(256), rng.nextInt(256))});

                    // read location
                    Location loc = Location.parse(l);
                    Direction dir = Direction.parse(l);
                    LocalEnvironment localEnv = createLocalEnvironment(loc, dir);
                    ILambda addLambda = addFish(localEnv, fish);
                    addLambda.apply(null);

                    return Boolean.TRUE;
                }
                catch (Exception e) {
                    e.printStackTrace();
                    throw new ParserException(e.toString(),e);
                }
            }
        }) == Boolean.TRUE) {
        }
    }

    /**
     * Get the environment settings class.
     *
     * @return environment settings class
     */
    public AEnvFactory makeEnvFactory() {
        return new AEnvFactory() {
            private JTextField _rotStepsField;
            private JColorChooser _colorChooser;

            {
                setLayout(new BoxLayout(this, BoxLayout.Y_AXIS));
                add(new JLabel("rotation steps: "));
                add(_rotStepsField = new JTextField("24"));
                add(new JLabel("water color: "));
                _colorChooser = new JColorChooser(OCEAN_BLUE);
                add(_colorChooser);
            }

            public AGlobalEnv create() {
                return new NoGridEnv(_cmdFactory,
                                     _securityAdapter,
                                     Integer.parseInt(_rotStepsField.getText()),
                                     _colorChooser.getColor());
            }

            public String toString() {
                return NoGridEnv.class.getName();
            }
        };
    }

    /**
     * Print file header.
     *
     * @param pw PrintWriter to use
     */
    protected void printHeader(PrintWriter pw) {
        pw.println(getClass().getName() + ' ' + _rotSteps + ' ' +
                   _waterColor.getRed() + ' ' + _waterColor.getGreen() + ' ' + _waterColor.getBlue());
    }

    /**
     * Get size of the display.
     *
     * @return size of the display in model coordinate units.
     */
    public Dimension getDisplaySize() {
        return new Dimension(PAN_SIZE, PAN_SIZE);
    }

    /**
     * The action to be executed if the display should return home.
     *
     * @param sa scroll adapter
     */
    public void returnHome(IScrollAdapter sa) {
        sa.setCorner((int) PAN_CENTER.x, (int) PAN_CENTER.y);
        sa.resetScrolling();
    }

    /**
     * Ask the model where to scroll, given where the user has scrolled. If the environment just acts like a normal
     * panal, it should return pos without modification. If the environment recenters, it should return a position in
     * the middle of the pan area. All coordinates are in model coordinate units.
     *
     * @param pos position where the user scrolled to
     * @return position where the environment wants the view to be
     * @see IDisplayAdapter#getPanDelta
     */
    public Point.Double getViewPosition(Point.Double pos) {
        // the panel always gets recentered after moving it, so return the center position
        return PAN_CENTER;
    }

    /**
     * Ask the model how much to pan, given where the user scrolled. If the environment just acts like a normal panal,
     * it should return (0,0). If the environment recenters, it should return delta without modification. All
     * coordinates are in model coordinate units.
     *
     * @param delta how far the user scrolled
     * @return how far the panel should scroll
     * @see IDisplayAdapter#getViewPosition
     */
    public Point.Double getPanDelta(Point.Double delta) {
        // we want the panel to keep track of the position, so return the delta as pan value
        return delta;
    }

    /**
     * Get a tool tip description for a specific place in the environment.
     *
     * @param p mouse coordinates
     * @return lambda for the simulation controller to execute. Must return the tooltip string.
     */
    public ILambda getToolTipText(final Point.Double p) {
        final LocalEnvironment l = createLocalEnvironment(new Location(p.x, p.y), new Direction());
        return ((ILambda) l.execute(new ILocalEnvVisitor() {
            public Object emptyCase(ALocalEnv host, Object param) {
                // this field is empty

                // return an ILambda that returns the string for an empty field
                return new ILambda() {
                    public Object apply(Object param) {
                        return "(" + p.x + ',' + p.y + ") is empty";
                    }
                };
            }

            public Object nonEmptyCase(ALocalEnv host, Object param) {
                // this field is occupied

                // return an ILambda that returns the string for an occupied
                return new ILambda() {
                    String _fishName;
                    ALocalEnv _localEnv;

                    public Object apply(Object param) {
                        _fishName = "";
                        _cmdFactory.makeNotifyCmd(new ILambda() {
                            public Object apply(Object param) {
                                FishApplyParams fap = (FishApplyParams) param;
                                LocalEnvironment lenv = (LocalEnvironment) fap.localEnv();
                                if (lenv.location().inSquare(l.location(), lenv.direction(), 1)) {
                                    _fishName = fap.fish().toString();
                                    _localEnv = fap.localEnv();
                                }
                                return null;
                            }
                        }).apply(null);

                        return _fishName + " at " + _localEnv;
                    }
                };
            }
        }, null));
    }


    /*****************************************************************************************************************
     * Tests follow
     *****************************************************************************************************************/

    /**
     * Test cases for NoGridEnv.
     *
     * @author Mathias Ricken
     */
    public static class Test_NoGridEnv extends TestCase {
        private ICmdFactory _cmdFactory;
        private ISecurityAdapter _sm;
        private NoGridEnv _env;
        private IFishFactory _fishFactory;

        private static final ILambda _notify = new ILambda() {
            public Object apply(Object param) {
                return "notifyCmd";
            }
        };
        private static final ILambda _delete = new ILambda() {
            public Object apply(Object param) {
                return "deleteCmd";
            }
        };
        private static final ILambda _add = new ILambda() {
            public Object apply(Object param) {
                return "addCmd";
            }
        };

        public void setUp() throws Exception {
            super.setUp();
            _cmdFactory = new ICmdFactory() {
                public ILambda makeNotifyCmd(ILambda lambda) {
                    return _notify;
                }

                public ILambda makeDeleteCmd(ALocalEnv env) {
                    return _delete;
                }

                public ILambda makeAddCmd(AFish fish) {
                    return _add;
                }
            };
            _sm = new ISecurityAdapter() {
                public void setProtected(boolean _protected) {
                }
                public ThreadGroup getFishThreadGroup() {
                    return null;
                }
                public ClassLoader getClassLoader() {
                    return null;
                }
                public void handleException(Throwable t) {
                }
            };

            _env = new NoGridEnv(_cmdFactory, _sm, 24, OCEAN_BLUE);


            _fishFactory = new IFishFactory() {
                /**
                 * Create a new fish.
                 *
                 * @return new fish
                 */
                public AFish createFish() {
                    return new GenericFish(Color.RED);
                }
            };
        }

        /**
         * Test addFishToInternalData.
         */
        public void testAddFish() {
            LRStruct lrs = _env._localEnvList;

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ALocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFishToInternalData(lTop, fTop);

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ALocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFishToInternalData(lBottom, fBottom);

            assertEquals(2, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
        }

        /**
         * Test editFish.
         */
        public void testEditFish() {
            LRStruct lrs = _env._localEnvList;

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            LocalEnvironment l = (LocalEnvironment) _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFishToInternalData(l, f);
            NoGridEnv.Direction d = l.direction();

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
            assertEquals(0.0, d.getAngle(), 0.01);

            ILambda lambda;
            for (int i = 1; 24 >= i; ++i) {
                lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
                assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
                assertEquals(i * Math.PI / 12.0, d.getAngle(), 0.01);
                assertEquals(NoOpLambda.instance(), lambda);
            }

            lambda = _env.editFish(l, _fishFactory, MouseEvent.BUTTON1);
            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
            assertEquals(Math.PI / 12.0, d.getAngle(), 0.01);
            assertEquals(_delete, lambda);
        }

        /**
         * Test getViewPosition.
         */
        public void testGetViewPosition() {
            Point.Double panCenter = PAN_CENTER;

            assertTrue(_env.getViewPosition(new Point.Double(0, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(1.0, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(1.2, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(0, 1.0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(0, 1.3)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(-2.5, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(-3.0, 0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(0, -2.5)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(0, -3.0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(2.0, 1.0)).equals(panCenter));
            assertTrue(_env.getViewPosition(new Point.Double(-4.0, -2.3)).equals(panCenter));
        }

        /**
         * Test getPanDelta.
         */
        public void testGetPanDelta() {
            assertTrue(_env.getPanDelta(new Point.Double(0, 0)).equals(new Point.Double(0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(1.0, 0)).equals(new Point.Double(1.0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(1.2, 0)).equals(new Point.Double(1.2, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, 1.0)).equals(new Point.Double(0, 1.0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, 1.3)).equals(new Point.Double(0, 1.3)));
            assertTrue(_env.getPanDelta(new Point.Double(-2.5, 0)).equals(new Point.Double(-2.5, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(-3.0, 0)).equals(new Point.Double(-3.0, 0)));
            assertTrue(_env.getPanDelta(new Point.Double(0, -2.5)).equals(new Point.Double(0, -2.5)));
            assertTrue(_env.getPanDelta(new Point.Double(0, -3.0)).equals(new Point.Double(0, -3.0)));
            assertTrue(_env.getPanDelta(new Point.Double(2.0, 1.0)).equals(new Point.Double(2.0, 1.0)));
            assertTrue(_env.getPanDelta(new Point.Double(-4.0, -2.3)).equals(new Point.Double(-4.0, -2.3)));
        }
    }

    /**
     * Test cases for NoGridEnv.LocalEnv.
     *
     * @author Mathias Ricken
     */
    public static class Test_NoGridEnv_LocalEnv extends TestCase {
        private ICmdFactory _cmdFactory;
        private ISecurityAdapter _sm;
        private NoGridEnv _env;

        private static class SuccessException extends RuntimeException {
            public SuccessException() {
                super();
            }
        }

        private static final ILambda _notify = new ILambda() {
            public Object apply(Object param) {
                return "notifyCmd";
            }
        };
        private static final ILambda _delete = new ILambda() {
            public Object apply(Object param) {
                return "deleteCmd";
            }
        };
        private static final ILambda _add = new ILambda() {
            public Object apply(Object param) {
                return "addCmd";
            }
        };

        public void setUp() throws Exception {
            super.setUp();
            _cmdFactory = new ICmdFactory() {
                public ILambda makeNotifyCmd(ILambda lambda) {
                    return _notify;
                }

                public ILambda makeDeleteCmd(ALocalEnv env) {
                    return _delete;
                }

                public ILambda makeAddCmd(AFish fish) {
                    return _add;
                }
            };
            _sm = new ISecurityAdapter() {
                public void setProtected(boolean _protected) {
                }
                public ThreadGroup getFishThreadGroup() {
                    return null;
                }
                public ClassLoader getClassLoader() {
                    return null;
                }
                public void handleException(Throwable t) {
                }
            };

            _env = new NoGridEnv(_cmdFactory, _sm);
        }

        /**
         * Test local environment's execute.
         */
        public void testExecute() {
            ALocalEnv l = _env.makeLocalEnv(new Point.Double(1, 1));

            try {
                l.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ALocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }

                    public Object nonEmptyCase(ALocalEnv host, Object param) {
                        throw new RuntimeException("Should be empty");
                    }
                }, null);
                fail("emptyCase should have been called --");
            }
            catch (SuccessException e) {
            }


            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);
            try {
                l.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ALocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ALocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch (SuccessException e) {
            }

            ALocalEnv l2 = _env.makeLocalEnv(new Point.Double(1, 1));
            try {
                l2.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ALocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ALocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch (SuccessException e) {
            }

            ALocalEnv l3 = _env.makeLocalEnv(new Point.Double(1.4, 1.4));
            try {
                l3.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ALocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ALocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch (SuccessException e) {
            }


            ALocalEnv l3b = _env.makeLocalEnv(new Point.Double(1.4, 1.6));
            try {
                l3b.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ALocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }

                    public Object nonEmptyCase(ALocalEnv host, Object param) {
                        throw new RuntimeException("Should be empty");
                    }
                }, null);
                fail("emptyCase should have been called --");
            }
            catch (SuccessException e) {
            }


            ALocalEnv l4 = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            try {
                l4.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ALocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }

                    public Object nonEmptyCase(ALocalEnv host, Object param) {
                        throw new RuntimeException("Should be empty");
                    }
                }, null);
                fail("emptyCase should have been called --");
            }
            catch (SuccessException e) {
            }


            GenericFish f4 = new GenericFish(Color.RED);
            f4.setLocalEnvironment(l4);
            _env.addFish(l4, f4);
            try {
                l4.execute(new AGlobalEnv.ILocalEnvVisitor() {
                    public Object emptyCase(ALocalEnv host, Object param) {
                        throw new RuntimeException("Should be non-empty");
                    }

                    public Object nonEmptyCase(ALocalEnv host, Object param) {
                        // ok
                        throw new SuccessException();
                    }
                }, null);
                fail("nonEmptyCase should have been called --");
            }
            catch (SuccessException e) {
            }
        }

        /**
         * Test local environment's tryMoveFwd.
         */
        public void testTryMoveFwd() {
            ALocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);

            ALocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);

            // move lBottom into lTop --> blocked
            Integer i = (Integer) lBottom.tryMoveFwd(fBottom, new IBlockedCommand() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",
                         new Integer(456),
                         i);

            // move lTop --> open, don't move
            i = (Integer) lTop.tryMoveFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --", param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",
                                 DeactivatableLambda.class,
                                 param.getClass());
                    // ok
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --", new Integer(123), i);

            // move lBottom into lTop --> blocked
            i = (Integer) lBottom.tryMoveFwd(fBottom, new IBlockedCommand() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(789);
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",
                         new Integer(789),
                         i);

            // move lTop --> open, move
            i = (Integer) lTop.tryMoveFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --", param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",
                                 DeactivatableLambda.class,
                                 param.getClass());
                    // ok, make move
                    ((ILambda) param).apply(null);
                    return new Integer(111);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --", new Integer(111), i);

            // move lBottom --> open
            i = (Integer) lBottom.tryMoveFwd(fBottom, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --", param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",
                                 DeactivatableLambda.class,
                                 param.getClass());
                    // ok
                    return new Integer(222);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --", new Integer(222), i);

            // move lTop --> open, don't move
            i = (Integer) lTop.tryMoveFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --", param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",
                                 DeactivatableLambda.class,
                                 param.getClass());
                    // ok
                    return new Integer(333);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --", new Integer(333), i);

            // turn and move lTop --> open, don't move
            lTop.turnRight(fTop, Math.PI / 2.0);
            i = (Integer) lTop.tryMoveFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable move lambda needs to be passed to openCmd --", param);
                    assertEquals("Error, deactivatable move lambda needs to be passed to openCmd --",
                                 DeactivatableLambda.class,
                                 param.getClass());
                    // ok
                    return new Integer(444);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --", new Integer(444), i);
        }

        /**
         * Test local environment's tryBreedFwd.
         */
        public void testTryBreedFwd() {
            LRStruct lrs = _env._localEnvList;

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ALocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ALocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);

            assertEquals(2, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // breed lBottom into lTop --> blocked
            Integer i = (Integer) lBottom.tryBreedFwd(fBottom, new IBlockedCommand() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",
                         new Integer(456),
                         i);

            // breed lTop --> open, don't breed
            i = (Integer) lTop.tryBreedFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --", param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",
                                 DeactivatableLambda.class,
                                 param.getClass());
                    // ok
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --", new Integer(123), i);

            assertEquals(2, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // breed lBottom into lTop --> blocked
            i = (Integer) lBottom.tryBreedFwd(fBottom, new IBlockedCommand() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",
                         new Integer(456),
                         i);

            // breed lTop --> open, breed
            i = (Integer) lTop.tryBreedFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --", param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",
                                 DeactivatableLambda.class,
                                 param.getClass());
                    // ok, breed
                    ((ILambda) param).apply(null);
                    return new Integer(123);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --", new Integer(123), i);

            assertEquals(3, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // breed lBottom into lTop --> blocked
            i = (Integer) lBottom.tryBreedFwd(fBottom, new IBlockedCommand() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",
                         new Integer(456),
                         i);

            // breed lTop --> blocked
            i = (Integer) lTop.tryBreedFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(456);
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",
                         new Integer(456),
                         i);

            // turn and breed lTop --> open, don't breed
            lTop.turnRight(fTop, Math.PI / 2.0);
            i = (Integer) lTop.tryBreedFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --", param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",
                                 DeactivatableLambda.class,
                                 param.getClass());
                    // ok
                    return new Integer(789);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --", new Integer(789), i);

            assertEquals(3, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // turn and breed lTop --> open, breed
            i = (Integer) lTop.tryBreedFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    assertNotNull("Error, deactivatable breed lambda needs to be passed to openCmd --", param);
                    assertEquals("Error, deactivatable breed lambda needs to be passed to openCmd --",
                                 DeactivatableLambda.class,
                                 param.getClass());
                    // ok, breed
                    ((ILambda) param).apply(null);
                    return new Integer(789);
                }
            });
            assertEquals("Error in delegation, openCmd not called, or incorrect return value --", new Integer(789), i);

            assertEquals(4, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            // turn and breed lTop --> blocked
            i = (Integer) lTop.tryBreedFwd(fTop, new IBlockedCommand() {
                public Object apply(Object param) {
                    // ok
                    return new Integer(789);
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be blocked");
                }
            });
            assertEquals("Error in delegation, blockedCmd not called, or incorrect return value --",
                         new Integer(789),
                         i);

            assertEquals(4, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
        }

        /**
         * Test local environment's turnRight.
         */
        public void testTurnRight() {
            LocalEnvironment l = (LocalEnvironment) _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);
            NoGridEnv.Direction d = l.direction();

            assertEquals(0.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(Math.PI / 2.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(Math.PI, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(3 * Math.PI / 2.0, d.getAngle(), 0.01);
            l.turnRight(f, Math.PI / 2);
            assertEquals(2 * Math.PI, d.getAngle(), 0.01);
        }

        /**
         * Test local environment's removeFish.
         */
        public void testRemoveFish() {
            LRStruct lrs = _env._localEnvList;

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ALocalEnv lTop = _env.makeLocalEnv(new Point.Double(1.0, 1.0));
            GenericFish fTop = new GenericFish(Color.RED);
            fTop.setLocalEnvironment(lTop);
            _env.addFish(lTop, fTop);

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            ALocalEnv lBottom = _env.makeLocalEnv(new Point.Double(1.0, 2.0));
            GenericFish fBottom = new GenericFish(Color.RED);
            fBottom.setLocalEnvironment(lBottom);
            _env.addFish(lBottom, fBottom);

            assertEquals(2, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            lTop.removeFish(fTop);

            assertEquals(1, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());

            lBottom.removeFish(fBottom);

            assertEquals(0, ((Integer) lrs.execute(GetLength.Singleton, null)).intValue());
        }

        /**
         * Test to make sure only one move lambda can be executed.
         */
        public void testOnlyOneMove() {
            LocalEnvironment l = (LocalEnvironment) _env.makeLocalEnv(new Point.Double(5.0, 5.0));
            GenericFish f = new GenericFish(Color.RED);
            f.setLocalEnvironment(l);
            _env.addFish(l, f);

            final LinkedList<ILambda> lambdas = new LinkedList<ILambda>();
            l.tryMoveFwd(f, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    lambdas.add((ILambda)param);
                    return null;
                }
            });
            f.turnRight();
            l.tryMoveFwd(f, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    lambdas.add((ILambda)param);
                    return null;
                }
            });

            assertEquals(2, lambdas.size());

            lambdas.get(0).apply(null);
            NoGridEnv.Location loc = l.location();
            assertTrue(loc.same(new NoGridEnv.Location(5.0, 4.0)));

            lambdas.get(1).apply(null);
            loc = l.location();
            assertTrue(loc.same(new NoGridEnv.Location(5.0, 4.0)));

            lambdas.clear();
            l.tryMoveFwd(f, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    lambdas.add((ILambda)param);
                    return null;
                }
            });
            f.turnRight();
            l.tryMoveFwd(f, new IBlockedCommand() {
                public Object apply(Object param) {
                    throw new RuntimeException("Should be open");
                }
            }, new IOpenCommand() {
                public Object apply(Object param) {
                    lambdas.add((ILambda)param);
                    return null;
                }
            });

            assertEquals(2, lambdas.size());

            lambdas.get(1).apply(null);
            loc = l.location();
            assertTrue(loc.same(new NoGridEnv.Location(6.0, 4.0)));

            lambdas.get(0).apply(null);
            loc = l.location();
            assertTrue(loc.same(new NoGridEnv.Location(6.0, 4.0)));
        }
    }

}
