package view;

import controller.IEnvAdapter;
import sysModel.ICmdFactory;
import sysModel.ISecurityAdapter;
import sysModel.env.AEnvFactory;
import sysModel.env.AGlobalEnv;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.lang.reflect.Constructor;

/**
 * Dialog to create new environment.
 *
 * @author Mathias Ricken
 */
public class CreateEnvDialog {
    /**
     * Environment factory to be used for creating the environment. This acts both as the factory and as the GUI panel
     * with the environment settings.
     */
    private AEnvFactory _envFactory;

    /**
     * Combobox containing the different choices for the environment.
     */
    private JComboBox _envChooser;

    /**
     * Environment dialog. This component contains everything else.
     */
    private JDialog _dialog;

    /**
     * Array of buttons on the dialog.
     */
    private JButton[] _optButtons;

    /**
     * Constant to access the create button in _optButtons.
     */
    private static final int CREATE_BUTTON = 0;

    /**
     * Constant to access the cancel button in _optButtons.
     */
    private static final int CANCEL_BUTTON = 1;

    /**
     * Panel for the environment options. Contains the environment factory.
     */
    private JPanel _optionsPanel;

    /**
     * Adpapter to the model for dealing with environments.
     */
    private IEnvAdapter _envAdapter;

    /**
     * Creates a new CreateEnvDialog and all its controls.
     *
     * @param parent parent frame for the dialog
     * @param ea     environment adapter
     */
    public CreateEnvDialog(JFrame parent, IEnvAdapter ea) {
        _envAdapter = ea;
        _dialog = new JDialog(parent, "Create new environment", true);
        _dialog.setContentPane(makeOptionsPanel());
        _dialog.pack();
        _dialog.setResizable(false);
    }


    /**
     * Show the modal dialog that allows the user to create a new environment.  If the dialog is dismissed by clicking
     * the "OK" button, a new environment is created to the user's specification and returned. If "Cancel" is chosen or
     * there is an error constructing the environment, <code>null</code> is returned.
     *
     * @return the newly created Environment or <code>null</code>
     */
    public AEnvFactory showDialog() {
        _envChooser.setSelectedIndex(0);
        envChosen();

        // center dialog in middle of parent frame
        Component parent = _dialog.getParent();
        _dialog.setLocation(parent.getX() + parent.getWidth() / 2 - _dialog.getSize().width / 2,
            parent.getY() + parent.getHeight() / 2 - _dialog.getSize().height / 2);
        _dialog.setVisible(true); // Modal dialog will block until setVisible(false), see ok/cancel methods

        return _envFactory;
    }

    /**
     * Remove environment settings from dialog.
     */
    private void removeEnvSettings() {
        if (null != _envFactory) {
            _optionsPanel.remove(_envFactory);
            _envFactory = null;
        }
    }

    /**
     * Add environment factory to dialog.
     *
     * @param factory factory to add
     */
    private void addEnvSettings(AEnvFactory factory) {
        _envFactory = factory;
        _optionsPanel.add(_envFactory);
    }

    /**
     * Callback when user selects new environment class choice from combo box.
     */
    private void envChosen() {
        IEnvChoice envChoice = (IEnvChoice)_envChooser.getSelectedItem();
        envChoice.select();
        _dialog.pack();
        _dialog.getContentPane().validate(); // force immediate validation
    }

    /**
     * Callback when user clicks cancel button.
     */
    private void cancelClicked() {
        _dialog.setVisible(false);
        removeEnvSettings();
    }

    /**
     * Callback when user clicks ok button.
     */
    private void okClicked() {
        _dialog.setVisible(false);
    }

    /**
     * Build options dialog.
     *
     * @return options pane
     */
    private JOptionPane makeOptionsPanel() {
        JPanel myControls = new JPanel();
        myControls.setLayout(new BoxLayout(myControls, BoxLayout.Y_AXIS));

        JLabel lab;
        myControls.add(lab = new JLabel("Choose environment type: "));
        lab.setAlignmentX(Component.CENTER_ALIGNMENT);

        _envChooser = new JComboBox();

        // add environments
        String[] envClassNames = _envAdapter.getEnvironmentClassNames();
        Class[] parameterTypes = new Class[]{ICmdFactory.class, ISecurityAdapter.class};
        Object[] initArgs = new Object[]{_envAdapter.getCmdFactory(), _envAdapter.getSecurityAdapter()};
        for(int i = 0; i < envClassNames.length; i++) {
            try {
                Class envClass = Class.forName(envClassNames[i]);
                Constructor envCtor = envClass.getConstructor(parameterTypes);
                AGlobalEnv env = (AGlobalEnv)envCtor.newInstance(initArgs);
                AEnvFactory factory = env.makeEnvFactory();

                ConcreteEnvChoice choice = new ConcreteEnvChoice(this, factory);
                _envChooser.addItem(choice);
            }
            catch(Exception e) {
                System.err.println(e);
            }
        }

        // add "Add..." choice
        _envChooser.addItem(new AddEnvChoice(this));

        _envChooser.setBorder(BorderFactory.createEmptyBorder(5, 0, 10, 0));
        _envChooser.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent evt) {
                envChosen();
            }
        });
        myControls.add(_envChooser);

        _optionsPanel = new JPanel();
        myControls.add(_optionsPanel);

        JOptionPane optPane = new JOptionPane(myControls, JOptionPane.QUESTION_MESSAGE);
        _optButtons = new JButton[]{new JButton("Create"), new JButton("Cancel")};
        optPane.setOptions(_optButtons);
        optPane.setInitialValue(_optButtons[CREATE_BUTTON]);
        _optButtons[CREATE_BUTTON].addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                okClicked();
            }
        });
        _optButtons[CANCEL_BUTTON].addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                cancelClicked();
            }
        });

        return optPane;
    }

    /**
     * Nested interface for environment choices.
     */
    private interface IEnvChoice {
        /**
         * Select this interface.
         */
        public void select();
    }

    /**
     * Nested class for choosing an environment.
     */
    private static class ConcreteEnvChoice implements IEnvChoice {
        /**
         * CreateEnvDialog used.
         */
        private CreateEnvDialog _envDialog;
        /**
         * Environment settings.
         */
        private AEnvFactory _envFactory;

        /**
         * Make a new concrete environment choice.
         *
         * @param envDialog the create environment dialog
         * @param factory   environment factory
         */
        public ConcreteEnvChoice(CreateEnvDialog envDialog, AEnvFactory factory) {
            _envDialog = envDialog;
            _envFactory = factory;
        }

        /**
         * Select this choice.
         */
        public void select() {
            _envDialog._optButtons[CREATE_BUTTON].setEnabled(true);
            _envDialog.removeEnvSettings();
            _envDialog.addEnvSettings(_envFactory);
        }

        /**
         * Return string representation.
         *
         * @return string representation
         */
        public String toString() {
            return _envFactory.toString();
        }
    }

    /**
     * Nested class for adding an environment.
     */
    private class AddEnvChoice implements IEnvChoice {
        /**
         * CreateEnvDialog used.
         */
        private CreateEnvDialog _envDialog;

        /**
         * Make a new add environment choice.
         *
         * @param envDialog the create environment dialog
         */
        public AddEnvChoice(CreateEnvDialog envDialog) {
            _envDialog = envDialog;
        }

        public void select() {
            _envDialog._optButtons[CREATE_BUTTON].setEnabled(false);
            _envDialog.removeEnvSettings();

            String className = (new InputStringDialog(_envDialog._dialog, "Add environment class")).showDialog();
            if (className==null) { return; }
            try {
                Class envClass = Class.forName(className);
                Constructor envCtor = envClass.getConstructor(new Class[]{ICmdFactory.class, ISecurityAdapter.class});
                AGlobalEnv env = (AGlobalEnv)envCtor.newInstance(
                    new Object[]{_envAdapter.getCmdFactory(), _envAdapter.getSecurityAdapter()});
                AEnvFactory envFactory = env.makeEnvFactory();

                ConcreteEnvChoice choice = new ConcreteEnvChoice(_envDialog, envFactory);
                _envDialog._envChooser.insertItemAt(choice, _envDialog._envChooser.getItemCount() - 1);
                _envDialog._envChooser.setSelectedItem(choice);
                choice.select();
            }
            catch(Exception e) {
                System.err.println(e);
                _envDialog._envChooser.setSelectedIndex(0);
                ((IEnvChoice)_envDialog._envChooser.getSelectedItem()).select();
            }
        }

        public String toString() {
            return "Add ...";
        }
    }
}

