// This class is based on the MBSGUIFrame class, version 1 August 2002
// by Julie Zenekski

// Original copyright notice:

// AP(r) Computer Science Marine Biology Simulation:
// The PseudoInfiniteViewport class is copyright(c) 2002 College Entrance
// Examination Board (www.collegeboard.com).
//
// This class is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation.
//
// This class is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

package view;

import controller.IDisplayAdapter;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

/**
 * The viewport for the display panel.
 *
 * @author Mathias Ricken
 */

public class DisplayViewport extends JViewport {
    /**
     * The Pannable interface contains those methods the view installed in a PseudoInfiniteViewport needs to support to
     * enable panning behavior along with scrolling.
     */
    public interface Pannable {
        /**
         * Pan by the specified amount.
         *
         * @param dx x-delta
         * @param dy y delta
         */
        void pan(double dx, double dy);

        /**
         * Reset the pan.
         */
        void resetPan();

        /**
         * Get the size of the cells.
         *
         * @return size of cells
         */
        int getCellSize();

        /**
         * Get pan tooltip text.
         *
         * @return pan tip text.
         */
        String getPanTipText();
    }

    private static final int ORIGIN_TIP_DELAY = 1000;

    /**
     * Scroll pane controlled by this viewport.
     */
    private JScrollPane _scrollParent;

    /**
     * Panel for the tool tip.
     */
    private JPanel _glassPane;

    /**
     * Origin tool tip.
     */
    private JToolTip _originTip;

    /**
     * Tool tip timer.
     */
    private Timer _originTipTimer;

    /**
     * Display adapter.
     */
    private IDisplayAdapter _displayAdapter;

    /**
     * Center before last movement.
     */
    private Point.Double _lastOrigin = new Point.Double();

    /**
     * Make a new display viewport for the given scroll pane.
     *
     * @param parent the JScrollPane for which this will be the viewport
     * @param da     display adapter to connect to the model
     */
    public DisplayViewport(JScrollPane parent, IDisplayAdapter da) {
        _scrollParent = parent;
        _displayAdapter = da;
        setBackground(Color.lightGray);
    }

    /**
     * Reset the viewport.
     */
    public void resetViewport() {
        Pannable p = getPannableView();
        if (null != p) {
            _lastOrigin = _displayAdapter.getViewPosition(new Point.Double(0, 0));
        }
    }

    /**
     * Set the old position of the view.
     *
     * @param pt old position of view
     */
    public void oldSetViewPosition(Point pt) {
        super.setViewPosition(pt);
    }

    /**
     * Sets the view position (upper left) to a new point. Overridden from JViewport.
     *
     * @param pt the Point to become the upper left
     */
    public void setViewPosition(Point pt) {
        boolean isAdjusting = _scrollParent.getVerticalScrollBar().getValueIsAdjusting() || _scrollParent.getHorizontalScrollBar().getValueIsAdjusting();
        Pannable p = getPannableView();
        Point.Double delta;

        boolean changed = !getViewPosition().equals(pt);
        super.setViewPosition(pt);

        if (null != p) {
            if (!isAdjusting) {
                // the user let go of the scrollbars
                int cellSize = p.getCellSize();

                // figure out how far the user scrolled
                delta = new Point.Double((double)(pt.x) / cellSize - _lastOrigin.x,
                    (double)(pt.y) / cellSize - _lastOrigin.y);

                // ask the environment how much to pan
                delta = _displayAdapter.getPanDelta(delta);
                // and pan
                p.pan(delta.x * cellSize, delta.y * cellSize);

                // convert position into model coordinate units
                Point.Double modelPos = new Point.Double(((double)pt.x) / cellSize, ((double)pt.y) / cellSize);
                // and ask the environment where to scroll
                modelPos = _displayAdapter.getViewPosition(modelPos);
                Point pixelPos = new Point((int)(modelPos.x * cellSize), (int)(modelPos.y * cellSize));
                // then scroll there
                super.setViewPosition(pixelPos);

                _lastOrigin = new Point.Double((double)(pixelPos.x) / cellSize, (double)(pixelPos.y) / cellSize);

                // update scrollbars and display
                fireStateChanged();
                repaint();
            }
        }

        if (isAdjusting || changed) {
            showOriginTip();
        }
    }

    /**
     * Return pannable view.
     *
     * @return pannable view
     */
    private Pannable getPannableView() {
        return (Pannable)getView();
    }

    /**
     * Show a tool tip over the upper left corner of the viewport with the contents of the pannable view's pannable tip
     * text (typically a string identifiying the corner point). Tip is removed after a short delay.
     */
    public void showOriginTip() {
        if (null == getRootPane()) {
            return;
        }
        // drawFish in glass pane to appear on top of other components
        if (null == _glassPane) {
            getRootPane().setGlassPane(_glassPane = new JPanel());
            _glassPane.setOpaque(false);
            _glassPane.setLayout(null);	// will control layout manually
            _glassPane.add(_originTip = new JToolTip());
            _originTipTimer = new Timer(ORIGIN_TIP_DELAY, new ActionListener() {
                public void actionPerformed(ActionEvent evt) {
                    _glassPane.setVisible(false);
                }
            });
            _originTipTimer.setRepeats(false);
        }
        String tipText = getPannableView().getPanTipText();
        if (null == tipText) {
            return;
        }

        // set tip text to identify current origin of pannable view
        _originTip.setTipText(tipText);

        // position tip to appear at upper left corner of viewport
        _originTip.setLocation(SwingUtilities.convertPoint(this, getLocation(), _glassPane));
        _originTip.setSize(_originTip.getPreferredSize());

        // show glass pane (it contains tip)
        _glassPane.setVisible(true);

        // this timer will hide the glass pane after a short delay
        _originTipTimer.restart();
    }
}
