// This class is based on the FishToolbar class, version 1 August 2002
// by Julie Zenekski

// Original copyright notice:

// AP(r) Computer Science Marine Biology Simulation:
// The FishToolbar class is copyright(c) 2002 College Entrance
// Examination Board (www.collegeboard.com).
//
// This class is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation.
//
// This class is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

package view;

import controller.IDisplayAdapter;
import controller.IEditAdapter;
import controller.IEnvAdapter;
import model.RandNumGenerator;
import sysModel.ISecurityAdapter;
import sysModel.fish.AFish;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.AffineTransform;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.util.Random;

/**
 * Code for the edit toolbar.
 *
 * @author Mathias Ricken
 */
public class EditToolbar extends JToolBar implements IEditAdapter {
    /**
     * Combobox to choose color.
     */
    private JComboBox _colorComboBox;

    /**
     * Combobox to choose fish class.
     */
    private JComboBox _fishComboBox;

    /**
     * The frame that contains this edit toolbar.
     */
    private JFrame _parentFrame;

    /**
     * Create a new EditToolbar and all its tools.
     *
     * @param parentFrame parent frame
     * @param ea          environment adapter
     * @param da          display adapter
     */
    public EditToolbar(JFrame parentFrame, IEnvAdapter ea, IDisplayAdapter da) {
        super(SwingConstants.VERTICAL);
        setFloatable(false);
        makeTools(ea, da);
        _parentFrame = parentFrame;
    }

    /**
     * Return the currently selected AFish class.
     *
     * @return name of a AFish class (or AFish subclass)
     */
    public String getCurrentFish() {
        return ((FishChoice)_fishComboBox.getSelectedItem()).getFishClassName();
    }

    /**
     * Return the currently selected color. If random color is selected, a new randomly generated color is returned.
     *
     * @return the color to use for a new fish object
     */
    public Color getCurrentColor() {
        ColorChoice cc = (ColorChoice)_colorComboBox.getSelectedItem();
        return cc.getColor();
    }

    /**
     * Create the UI components for the toolbar.
     *
     * @param ea environment adapter
     * @param da display adapter
     */
    private void makeTools(final IEnvAdapter ea, final IDisplayAdapter da) {
        setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder("Configure new fish"),
            BorderFactory.createEmptyBorder(4, 4, 4, 4)));

        add(new JLabel(" Type: "));

        _fishComboBox = new JComboBox();

        // add initial fish choices
        String[] fishClassNames = ea.getFishClassNames();
        for(int i = 0; i < fishClassNames.length; i++) {
            String className = fishClassNames[i];
            try {
                _fishComboBox.addItem(new FishChoice(className, da, ea.getSecurityAdapter()));
            }
            catch(Exception e) {
                System.out.println(className + " throws: " + e);
                e.printStackTrace();
            }
        }
        ColorChoice addNew = new ColorChoice("Add ...", Color.white) {
            public void select() {
                String className = (new InputStringDialog(JOptionPane.getFrameForComponent(EditToolbar.this),
                    "Add fish class")).showDialog();
                if (className==null) { return; }
                try {
                    FishChoice choice = new FishChoice(className, da, ea.getSecurityAdapter());
                    _fishComboBox.insertItemAt(choice, _fishComboBox.getItemCount() - 1);
                    _fishComboBox.setSelectedItem(choice);
                    choice.select();
                }
                catch(Throwable t) {
                    _fishComboBox.setSelectedIndex(0);
                    Object[] options = { "OK" };
                    StringBuffer message = new StringBuffer(t.toString());
                    if (null != t.getMessage()) {
                        message.append(':');
                        message.append(t.getMessage());
                    }
                    if (null != t.getCause()) {
                        message.append('\n');
                        message.append(t.getCause().toString());
                    }
                    JOptionPane.showOptionDialog(JOptionPane.getFrameForComponent(EditToolbar.this),
                                                 message,
                                                 "Error Loading Fish",
                                                 JOptionPane.DEFAULT_OPTION,
                                                 JOptionPane.WARNING_MESSAGE,
                                                 null,
                                                 options,
                                                 options[0]);
                }
            }
        };
        _fishComboBox.addItem(addNew);

        _fishComboBox.setRenderer(new ChoiceWithIconRenderer(_fishComboBox));
        _fishComboBox.setAlignmentX(LEFT_ALIGNMENT);
        _fishComboBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                ChoiceWithIcon c = (ChoiceWithIcon)_fishComboBox.getSelectedItem();
                c.select();
            }
        });
        add(_fishComboBox);

        addSeparator();

        add(new JLabel(" Color: "));

        ColorChoice random = new ColorChoice("Random", ColorIcon.RANDOM_COLOR) {
            public Color getColor() {
                Random rng = RandNumGenerator.instance();
                return new Color(rng.nextInt(256), rng.nextInt(256), rng.nextInt(256));
            }
        };
        ColorChoice other = new ColorChoice("Other ...", Color.white) {
            public void select() {
                _colorComboBox.hidePopup();
                Component parentFrame = JOptionPane.getFrameForComponent(EditToolbar.this);
                Color chosen = JColorChooser.showDialog(parentFrame, "Choose Fish Color", getColor());
                if (null != chosen) {
                    setColor(chosen);
                }
            }
        };
        ColorChoice[] standardChoices = {new ColorChoice("Red", Color.red),
                                         new ColorChoice("Orange", new Color(255, 128, 0)),
                                         new ColorChoice("Yellow", Color.yellow),
                                         new ColorChoice("Green", Color.green),
                                         new ColorChoice("Blue", new Color(0, 128, 255)),
                                         new ColorChoice("Purple", new Color(128, 0, 128)),
                                         random,
                                         other};
        _colorComboBox = new JComboBox(standardChoices);
        _colorComboBox.setSelectedItem(random);
        _colorComboBox.setRenderer(new ChoiceWithIconRenderer(_colorComboBox));
        _colorComboBox.setAlignmentX(LEFT_ALIGNMENT);
        _colorComboBox.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                ColorChoice cc = (ColorChoice)_colorComboBox.getSelectedItem();
                cc.select();
            }
        });
        add(_colorComboBox);

        add(Box.createGlue());
    }

    /**
     * Shows/hides the fish toolbar. In order to avoid disrupting the current layout, resize the parent frame by the
     * amount need to grow/shrink to accommodate the change in window size without resizing any other components.
     *
     * @param visible true if to be made visible
     */
    public void setVisible(boolean visible) {
        if (isVisible() != visible) {
            super.setVisible(visible);
            Dimension cur = _parentFrame.getSize();
            Dimension change;
            if (HORIZONTAL == getOrientation()) {
                change = new Dimension(0, (visible ? 1 : -1) * getPreferredSize().height);
            }
            else {
                change = new Dimension((visible ? 1 : -1) * getPreferredSize().width, 0);
            }
            _parentFrame.setSize(new Dimension(cur.width + change.width, cur.height + change.height));
            _parentFrame.validate(); // force immediate validation
        }
    }

    /**
     * Enables or disables the edit toolbar.
     *
     * @param enabled new state
     */
    public void setEnabled(boolean enabled) {
        if (isEnabled() != enabled) {
            super.setEnabled(enabled);
            setVisible(enabled);
        }
    }

    /**
     * Interface that unifies choices with icons (fish, color) so we can use one common renderer for both.
     */
    private interface ChoiceWithIcon {
        Icon getIcon();

        /**
         * Callback if this choice was selected.
         */
        void select();
    }

    /**
     * Nested class used to hold the per-item information for the entries in the combo box of color choices. Each item
     * represents a color choice which is basically just a Color object and a name.
     */
    private class ColorChoice implements ChoiceWithIcon {
        private Color color;
        private String name;
        private ColorIcon icon;

        public ColorChoice(String n, Color c) {
            color = c;
            name = n;
            icon = new ColorIcon(color, 16, 16);
        }

        public void setColor(Color c) {
            color = c;
            icon.setColor(c);
        }

        public Color getColor() {
            return color;
        }

        public String toString() {
            return name;
        }

        public Icon getIcon() {
            return icon;
        }

        public void select() {
            // do nothing by default
        }
    }

    /**
     * Nested class used to hold the per-item information for the entries in the combo box of fish choices. Each item
     * represents a fish choice which is a AFish class.
     */
    private class FishChoice implements ChoiceWithIcon {
        private String _fishClassName;
        private FishIcon _icon;

        public FishChoice(String fishClassName, IDisplayAdapter da, ISecurityAdapter sa)
        throws ClassNotFoundException, NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException{
            _fishClassName = fishClassName;
            _icon = new FishIcon(_fishClassName, Color.gray, 16, 16, da, sa);
        }

        public String getFishClassName() {
            return _fishClassName;
        }

        public String toString() {
            return _fishClassName;
        }

        public Icon getIcon() {
            return _icon;
        }

        public void select() {
            // by default do nothing
        }
    }

    // Custom cell renderer that displays text and icon.
    // (Default renderer can only display one or the other.)
    private static class ChoiceWithIconRenderer extends JLabel implements ListCellRenderer {
        private JComboBox cb;

        public ChoiceWithIconRenderer(JComboBox b) {
            setOpaque(true);
            setHorizontalAlignment(LEFT);
            setVerticalAlignment(CENTER);
            setBorder(BorderFactory.createEmptyBorder(2, 2, 2, 2));
            cb = b;
        }

        public Component getListCellRendererComponent(JList list,
                                                      Object value,
                                                      int index,
                                                      boolean selected,
                                                      boolean cellHasFocus) {
            setBackground(selected ? list.getSelectionBackground() : list.getBackground());
            setForeground(selected ? list.getSelectionForeground() : list.getForeground());
            if (!cb.isEnabled()) {
                setText("No choice"); // drawFish differently when disabled
                setIcon(null);
            }
            else {
                setText(value.toString());
                setIcon(((ChoiceWithIcon)value).getIcon());
            }
            return this;
        }
    }


    /**
     * Nested class used to drawFish the color swatch icon used for color choice entries in the color combo box. This
     * simple class just draws a rectangle filled with the color and edged with a black border.
     */
    private static class ColorIcon implements Icon {
        // so we have a unique object to compare == to
        public static final Color RANDOM_COLOR = new Color(0, 0, 0);
        private static final int MARGIN = 2;

        private Color _color;
        private int _width;
        private int _height;

        public ColorIcon(Color c, int w, int h) {
            _color = c;
            _width = w;
            _height = h;
        }

        public void setColor(Color c) {
            _color = c;
        }

        public int getIconWidth() {
            return _width;
        }

        public int getIconHeight() {
            return _height;
        }

        public void paintIcon(Component comp, Graphics g, int x, int y) {
            Graphics2D g2 = (Graphics2D)g;
            Rectangle r = new Rectangle(x + MARGIN, y + MARGIN, _width - 2 * MARGIN, _height - 2 * MARGIN);
            if (_color != RANDOM_COLOR) {
                g2.setColor(_color);
                g2.fill(r);
            }
            else {
                for(int k = 0; k < r.width; k++)  // drawFish rainbow lines
                {
                    g2.setColor(Color.getHSBColor((float)k / r.width, .95f, 1));
                    g2.drawLine(r.x + k, r.y, r.x + k, r.y + r.height);
                }
            }
            g2.setColor(Color.black);
            g2.draw(r);
        }
    }

    /**
     * Nested class used to drawFish the fish icon used for fish entries in the fish combo box. We construct a AFish and
     * then hand it off to its display object to drawFish.
     */
    static class FishIcon implements Icon {
        private AFish _fish = null;
        private String _fishClassName;
        private Color _color;
        private int _width;
        private int _height;
        IDisplayAdapter _da;
        ISecurityAdapter _sa;

        public FishIcon(String fishClassName, Color c, int w, int h, IDisplayAdapter da, ISecurityAdapter sa)
        throws ClassNotFoundException, NoSuchMethodException, InstantiationException, IllegalAccessException, InvocationTargetException {
            _fishClassName = fishClassName;
            _color = c;
            _width = w;
            _height = h;
            _da = da;
            _sa = sa;

            if (null == _fish) {
                // NOTE: Introduced class loader here
                Class fishClass = null;
                // Note: DrJava incompatibility.
                fishClass = _sa.getClassLoader().loadClass(_fishClassName);
                Constructor envCtor = fishClass.getConstructor(new Class[]{Color.class});
                _fish = (AFish)envCtor.newInstance(new Object[]{Color.BLUE});
            }
        }

        public int getIconWidth() {
            return _width;
        }

        public int getIconHeight() {
            return _height;
        }

        public void paintIcon(Component comp, Graphics g, int x, int y) {
            Graphics2D g2 = (Graphics2D)g;
            AffineTransform savedTransform = g2.getTransform(); // save current
            g2.translate(x + getIconWidth() / 2, y + getIconHeight() / 2);
            g2.setStroke(new BasicStroke(1.0f / getIconHeight()));
            g2.scale(_width, _height);
            _fish.paint(g2, comp);
            g2.setTransform(savedTransform); // restore coordinate system
        }
    }

}


