// This class is based on the EnvFileChooser class, version 1 August 2002
// by Julie Zenekski

// Original copyright notice:

// AP(r) Computer Science Marine Biology Simulation:
// The  EnvFileChooser class is copyright(c) 2002 College Entrance
// Examination Board (www.collegeboard.com).
//
// This class is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation.
//
// This class is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

package view;

import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import java.awt.*;
import java.io.File;

/**
 * Environment file chooser class.
 *
 * @author Mathias Ricken
 */

public class EnvFileChooser extends JFileChooser {
    /**
     * Standard extension.
     */
    private static final String ENV_EXT = ".txt";

    /**
     * Open accessory.
     */
    private JPanel openAccessory;

    /**
     * Create a new EnvFileChooser. The default starting directory will be the "DataFiles" subdirectory on the current
     * working directory.
     */
    public EnvFileChooser() {
        super(new File(System.getProperty("user.dir") + File.separator + "DataFiles"));
        setFileFilter(new FileFilter() {
            public boolean accept(File f) {
                return (f.getName().toLowerCase().endsWith(ENV_EXT) || f.isDirectory());
            }

            public String getDescription() {
                return ("RiceMBS data files (*" + ENV_EXT + ')');
            }
        });
        setFileSelectionMode(JFileChooser.FILES_ONLY);
    }

    /**
     * Bring up a modal file chooser dialog allowing the user to choose the environment file and specify the
     * bounded/unbounded environment class.
     *
     * @param parent the parent of the dialog
     *
     * @return the return state of the dialog once finished
     */
    public int showOpenDialog(Component parent) {
        setDialogTitle("Open environment file");
        setAccessory(openAccessory);
        rescanCurrentDirectory();
        return super.showOpenDialog(parent);
    }


    /**
     * Bring up a modal file chooser dialog allowing the user to save to an environment file.
     *
     * @param parent the parent of the dialog
     *
     * @return the return state of the dialog once finished
     */
    public int showSaveDialog(Component parent) {
        setDialogTitle("Save environment file");
        setAccessory(null);
        rescanCurrentDirectory();
        return super.showSaveDialog(parent);
    }


    /**
     * Called when the user hits the approve button (Save/Open) to confirm the selected file is acceptable. Overrides
     * the JFileChooser to confirm overwrite if choosing a file that already exists for a save action.
     */
    public void approveSelection() {
        if (SAVE_DIALOG == getDialogType()) {
            File file = getSelectedFile();
            if (!file.getName().endsWith(ENV_EXT))	// add extension if missing
            {
                setSelectedFile(file = new File(file.getAbsolutePath() + ENV_EXT));
            }
            if (file.exists() && JOptionPane.OK_OPTION != JOptionPane.showConfirmDialog(this,
                                                                                        "File " + file.getName() + " exists. Overwrite?",
                                                                                        "Confirm overwrite",
                                                                                        JOptionPane.OK_CANCEL_OPTION)) {
                return;
            }
        }
        super.approveSelection(); //if we get here, this will dismiss the dialog
    }
}
