package view;

import controller.IDisplayAdapter;
import controller.IEnvAdapter;
import controller.IScrollAdapter;
import controller.ISimAdapter;
import lrs.LRStruct;
import lrs.visitor.Apply;
import model.ILambda;
import model.RandNumGenerator;
import sysModel.NoOpLambda;
import sysModel.env.AEnvFactory;

import javax.swing.*;
import javax.swing.event.HyperlinkEvent;
import javax.swing.event.HyperlinkListener;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.KeyEvent;
import java.awt.event.WindowEvent;
import java.net.URL;
import java.io.IOException;

/**
 * The main frame class of the GUI.
 *
 * @author Mathias Ricken
 */
public class MBSView extends JFrame {
    /**
     * Program date.
     */
    private static final String VERSION_DATE = "July 2004";

    /**
     * List of menu items that are only enabled if an environment is present.
     */
    private LRStruct _componentsThatNeedAnEnvironment;

    /**
     * List of menu items that are disabled if the simulation is running.
     */
    private LRStruct _componentsDisabledDuringRun;

    /**
     * Edit toolbar.
     */
    private EditToolbar _editToolbar;

    /**
     * Simulation toolbar.
     */
    private SimToolbar _simToolbar;

    /**
     * Display panel.
     */
    public DisplayPanel _displayPanel;

    /**
     * Display viewport.
     */
    private DisplayViewport _displayViewport;

    /**
     * Scroll panel.
     */
    private JScrollPane _scrollPane;

    /**
     * Environment file chooser dialog.
     */
    private EnvFileChooser _fileChooser;

    /**
     * Create environment dialog.
     */
    private CreateEnvDialog _createEnvDialog;

    /**
     * Seed value.
     */
    private int _randSeed = RandNumGenerator.instance().getSeed();

    /**
     * Step count.
     */
    private int _stepCount = 10;

    /**
     * Iteration command to run indefinitely.
     */
    private ILambda _indefinitelyIterLambda = new ILambda() {
        public Object apply(Object param) {
            // don't do anything, just redraw
            _displayViewport.repaint();
            return null;
        }
    };

    /**
     * Display adapter.
     */
    IDisplayAdapter _displayAdapter;

    /**
     * Simulation adapter.
     */
    ISimAdapter _simAdapter;

    /**
     * Environment adapter to the model.
     */
    IEnvAdapter _envAdapter;

    /**
     * Scroll adapter.
     */
    IScrollAdapter _scrollAdapter;

    /**
     * Processes window events occurring on this component. Hides the window or disposes of it, as specified by the
     * setting of the <code>defaultCloseOperation</code> property.
     *
     * @param e the window event
     *
     * @see #setDefaultCloseOperation
     * @see Window#processWindowEvent
     */
    protected void processWindowEvent(WindowEvent e) {
        super.processWindowEvent(e);

        if (WindowEvent.WINDOW_CLOSING == e.getID()) {
            _envAdapter.getSecurityAdapter().setProtected(false);
            System.exit(0);
        }
    }


    /**
     * Create a new MBSGUIFrame and all its controls.
     *
     * @param da display adapter
     * @param sa simulation adapter
     * @param ea environment adapter
     */
    public MBSView(IDisplayAdapter da, ISimAdapter sa, IEnvAdapter ea) {
        _displayAdapter = da;
        _simAdapter = sa;
        _envAdapter = ea;
        _scrollAdapter = new IScrollAdapter() {
            public void setCorner(int x, int y) {
                _displayPanel.setCorner(x, y);
            }

            public void resetScrolling() {
                _displayPanel.resetPan();
                _displayViewport.resetViewport();
            }
        };

        _componentsThatNeedAnEnvironment = new LRStruct();
        _componentsDisabledDuringRun = new LRStruct();

        System.setProperty("sun.awt.exception.handler", GUIExceptionHandler.class.getName());

        setTitle("Rice Marine Biology Simulation");
        setLocation(25, 15);


        JPanel content = new JPanel();
        setContentPane(content);
        content.setBorder(BorderFactory.createEmptyBorder(15, 15, 15, 15));
        content.setLayout(new BorderLayout());

        _displayPanel = new DisplayPanel(_displayAdapter, _envAdapter);
        _scrollPane = new JScrollPane(_displayPanel,
            JScrollPane.VERTICAL_SCROLLBAR_ALWAYS,
            JScrollPane.HORIZONTAL_SCROLLBAR_ALWAYS);
        _displayViewport = new DisplayViewport(_scrollPane, _displayAdapter);
        _scrollPane.setViewport(_displayViewport);
        _scrollPane.setViewportView(_displayPanel);
        _scrollPane.setPreferredSize(_displayPanel.getPreferredSize());

        content.add(_scrollPane, BorderLayout.CENTER);

        _editToolbar = new EditToolbar(this, _envAdapter, _displayAdapter);
        content.add(_editToolbar, BorderLayout.EAST);

        _simToolbar = new SimToolbar(_simAdapter, new IRunIdleAdapter() {
            public void setRunState() {
                setComponentsEnabled(_componentsDisabledDuringRun, false);
                setEditModeEnable(false);
                _displayViewport.repaint();
            }

            public void setIdleState() {
                setComponentsEnabled(_componentsDisabledDuringRun, true);
                _displayViewport.repaint();
            }
        });
        _simToolbar.setVisible(true);
        _simToolbar.setEnabled(false);
        content.add(_simToolbar, BorderLayout.SOUTH);
        _componentsThatNeedAnEnvironment.insertFront(_simToolbar);

        _fileChooser = new EnvFileChooser();
        _createEnvDialog = new CreateEnvDialog(this, _envAdapter);

        setEditModeEnable(false);
        makeMenus();

        _simAdapter.setIterationLambda(_indefinitelyIterLambda);
    }

    /**
     * Create the drop-down menus on the frame.
     */
    private void makeMenus() {
        int menuMask = getToolkit().getMenuShortcutKeyMask();
        JMenuBar mbar = new JMenuBar();
        JMenu menu;
        JMenuItem mItem;

        mbar.add(menu = new JMenu("File"));
        menu.add(mItem = new JMenuItem("Open environment file..."));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (JFileChooser.APPROVE_OPTION == _fileChooser.showOpenDialog(MBSView.this)) {
                    _simAdapter.stop();
                    _simToolbar.setControlsInIdleState();
                    setComponentsEnabled(_componentsThatNeedAnEnvironment, true);
                    setEditModeEnable(false);
                    _envAdapter.loadEnvironment(_fileChooser.getSelectedFile().getAbsoluteFile().toString());
                    _displayPanel.revalidate();
                    _scrollAdapter.resetScrolling();
                    _displayAdapter.returnHome(_scrollAdapter);
                }
            }
        });
        mItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_O, menuMask));

        menu.add(mItem = new JMenuItem("Create new environment..."));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                AEnvFactory settings = _createEnvDialog.showDialog();
                if (null != settings) {
                    _simAdapter.stop();
                    _simToolbar.setControlsInIdleState();
                    setComponentsEnabled(_componentsThatNeedAnEnvironment, true);
                    setEditModeEnable(false);
                    _envAdapter.createEnvironment(settings);
                    _displayPanel.revalidate();
                    _scrollAdapter.resetScrolling();
                    _displayAdapter.returnHome(_scrollAdapter);
                }
            }
        });
        mItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_N, menuMask));

        menu.add(mItem = new JMenuItem("Edit environment..."));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                setEditModeEnable(true);
            }
        });
        mItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_E, menuMask));
        _componentsThatNeedAnEnvironment.insertFront(mItem);

        menu.add(mItem = new JMenuItem("Save environment as..."));
        mItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_S, menuMask));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                if (JFileChooser.APPROVE_OPTION == _fileChooser.showSaveDialog(MBSView.this)) {
                    setComponentsEnabled(_componentsThatNeedAnEnvironment, true);
                    setEditModeEnable(false);
                    _envAdapter.saveEnvironment(_fileChooser.getSelectedFile().getAbsoluteFile().toString());
                }
            }
        });
        _componentsThatNeedAnEnvironment.insertFront(mItem);

        menu.add(mItem = new JMenuItem("Quit"));
        mItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_Q, menuMask));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _envAdapter.getSecurityAdapter().setProtected(false);
                System.exit(0);
            }
        });

        mbar.add(menu = new JMenu("Seed"));
        ButtonGroup bGroup = new ButtonGroup();
        menu.add(mItem = new JRadioButtonMenuItem("Don't change seed", true));
        bGroup.add(mItem);
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                // don't do anything
                _envAdapter.setSeedLambda(NoOpLambda.instance());
            }
        });
        menu.add(mItem = new JRadioButtonMenuItem("Use fixed seed..."));
        bGroup.add(mItem);
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                final Integer seed = queryForInteger("Enter seed for random number generator:",
                    "Input",
                    (0 == _randSeed) ? 17 : _randSeed,
                    MBSView.this);
                if (null != seed) {
                    _randSeed = seed.intValue();
                    _envAdapter.setSeedLambda(new ILambda() {
                        public Object apply(Object param) {
                            // set fixed seed
                            RandNumGenerator.instance().setSeed(seed.intValue());
                            return null;
                        }
                    });
                }
            }
        });
        menu.add(mItem = new JRadioButtonMenuItem("Prompt for seed"));
        bGroup.add(mItem);
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _envAdapter.setSeedLambda(new ILambda() {
                    public Object apply(Object param) {
                        Integer seed = queryForInteger("Enter seed for random number generator:",
                            "Input",
                            (0 == _randSeed) ? 17 : _randSeed,
                            MBSView.this);
                        if (null != seed) {
                            _randSeed = seed.intValue();
                            RandNumGenerator.instance().setSeed(seed.intValue());
                        }
                        return null;
                    }
                });
            }
        });

        mbar.add(menu = new JMenu("Run"));
        bGroup = new ButtonGroup();
        menu.add(mItem = new JRadioButtonMenuItem("Run Indefinitely", true));
        bGroup.add(mItem);
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _simAdapter.setStartLambda(new ILambda() {
                    public Object apply(Object param) {
                        // don't do anything
                        return null;
                    }
                });
                _simAdapter.setIterationLambda(_indefinitelyIterLambda);
            }
        });
        _componentsDisabledDuringRun.insertFront(mItem);

        menu.add(mItem = new JRadioButtonMenuItem("Use fixed number of steps..."));
        bGroup.add(mItem);
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                final Integer steps = queryForInteger("Enter number of steps:", "Input", _stepCount, MBSView.this);
                if (null != steps) {
                    _stepCount = steps.intValue();
                    final StepItLambda itLambda = new StepItLambda(_simAdapter, _displayViewport, _simToolbar);
                    _simAdapter.setStartLambda(new ILambda() {
                        public Object apply(Object param) {
                            // change counter
                            itLambda.setSteps(steps.intValue());
                            return null;
                        }
                    });
                    _simAdapter.setIterationLambda(itLambda);
                }
            }
        });
        _componentsDisabledDuringRun.insertFront(mItem);

        menu.add(mItem = new JRadioButtonMenuItem("Prompt for number of steps"));
        bGroup.add(mItem);
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                final StepItLambda itLambda = new StepItLambda(_simAdapter, _displayViewport, _simToolbar);
                _simAdapter.setStartLambda(new ILambda() {
                    public Object apply(Object param) {
                        Integer steps = queryForInteger("Enter number of steps:",
                            "Input",
                            _stepCount,
                            MBSView.this);
                        if (null != steps) {
                            _stepCount = steps.intValue();
                            itLambda.setSteps(steps.intValue());
                        }
                        return null;
                    }
                });
                _simAdapter.setIterationLambda(itLambda);
            }
        });
        _componentsDisabledDuringRun.insertFront(mItem);

        mbar.add(menu = new JMenu("View"));
        menu.add(mItem = new JMenuItem("Zoom in"));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _displayPanel.zoomIn();
            }
        });
        mItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_UP, menuMask));
        _componentsThatNeedAnEnvironment.insertFront(mItem);

        menu.add(mItem = new JMenuItem("Zoom out"));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _displayPanel.zoomOut();
            }
        });
        mItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_DOWN, menuMask));
        _componentsThatNeedAnEnvironment.insertFront(mItem);

        menu.add(mItem = new JMenuItem("Bring (0, 0) to upper left"));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                _scrollAdapter.resetScrolling();
                _displayAdapter.returnHome(_scrollAdapter);
            }
        });
        mItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_C, menuMask));
        _componentsThatNeedAnEnvironment.insertFront(mItem);

        mbar.add(menu = new JMenu("Help"));
        menu.add(mItem = new JMenuItem("About RiceMBS..."));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                showAboutPanel();
            }
        });
        menu.add(mItem = new JMenuItem("Help..."));
        mItem.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                showHelp();
            }
        });
        mItem.setAccelerator(KeyStroke.getKeyStroke(KeyEvent.VK_HELP, menuMask));

        setComponentsEnabled(_componentsThatNeedAnEnvironment, false);
        setComponentsEnabled(_componentsDisabledDuringRun, true);
        setJMenuBar(mbar);
    }

    /**
     * Set the enabled status of those menu items that need an environment to be valid.
     *
     * @param list   list of components
     * @param enable true if components should be enabled
     */
    public static void setComponentsEnabled(LRStruct list, final boolean enable) {
        list.execute(Apply.Singleton, new ILambda() {
            /**
             * Execute command.
             */
            public Object apply(Object param) {
                ((JComponent)param).setEnabled(enable);
                return null;
            }
        });
    }

    /**
     * Enable or disable the edit mode.
     *
     * @param enable true if edit mode should be enabled
     */
    public void setEditModeEnable(boolean enable) {
        if (enable != _editToolbar.isEnabled()) {
            _editToolbar.setEnabled(enable);
            _displayPanel.enableMouseAdapter(enable);
        }
    }

    /**
     * Brings up a simple dialog with some general information.
     */
    private void showAboutPanel() {
        String html = "<html><h2>Rice Marine Biology Simulation Program</h2>" + "A tool for running and viewing the AP&reg; Computer Science<p>" + "Marine Biology Simulation case study program.<p><p>" + "<font size=-1>Version: " + VERSION_DATE + "</font><p>" + "<font size=-1>Copyright&copy; 2002 College Entrance Examination Board " + "(www.collegeboard.com).<br>" + "<font size=-1>Copyright&copy; 2003 Rice University " + "(www.rice.edu).</font></font></html>";
        JOptionPane.showMessageDialog(this, new JLabel(html), "About RiceMBS", JOptionPane.INFORMATION_MESSAGE, null);
    }

    /**
     * Brings up a window with a scrolling text pane that display the help information for the simulation.
     */
    private void showHelp() {
        JDialog dialog = new JDialog(this, "RiceMBS Help");
        final JEditorPane helpText = new JEditorPane();
        try {
            helpText.setPage(
                new URL("file",
                    "",
                    System.getProperty("user.dir") + java.io.File.separator + "Resources" + java.io.File.separator + "MBSHelp.html"));
        }
        catch(IOException e) {
            helpText.setText("Couldn't load help file.");
        }
        helpText.setEditable(false);
        helpText.addHyperlinkListener(new HyperlinkListener() {
            public void hyperlinkUpdate(HyperlinkEvent ev) {
                if (ev.getEventType() == HyperlinkEvent.EventType.ACTIVATED) {
                    try {
                        helpText.setPage(ev.getURL());
                    }
                    catch(IOException ex) {
                        System.err.println(ex);
                    }
                }
            }
        });
        JScrollPane sp = new JScrollPane(helpText);
        sp.setPreferredSize(new Dimension(650, 500));
        dialog.getContentPane().add(sp);
        dialog.setLocation(getX() + getWidth() - 200, getY() + 50);
        dialog.pack();
        dialog.setVisible(true);
    }

    /**
     * Handle an exception that occurred during the simulation.
     * @param t exception caught during simulation
     */
    public void handleException(Throwable t) {
        Object[] options = { "OK" };
        StringBuffer message = new StringBuffer(t.toString());
        if (null != t.getCause()) {
            message.append('\n');
            message.append(t.getCause().toString());
        }
        JOptionPane.showOptionDialog(JOptionPane.getFrameForComponent(this),
                                     message,
                                     "Error Loading Fish",
                                     JOptionPane.DEFAULT_OPTION,
                                     JOptionPane.WARNING_MESSAGE,
                                     null,
                                     options,
                                     options[0]);
    }

    /**
     * Nested class that is registered as the handler for exceptions on the Swing event thread. The handler will put up
     * an alert panel, dump the stack trace to the console, and then exit entire program rather than persist in an
     * inconsistent state, which would be the default behavior.
     */
    public class GUIExceptionHandler {
        /**
         * Handle the exception by displaying a dialog box and exiting.
         *
         * @param e exception that occurred
         */
        public void handle(Throwable e) {
            e.printStackTrace();
            JOptionPane.showMessageDialog(null,
                "An error occurred. The simulation must exit." + "\nReason: " + e,
                "Error",
                JOptionPane.ERROR_MESSAGE);
            _envAdapter.getSecurityAdapter().setProtected(false);
            System.exit(0);
        }
    }

    /**
     * Query for an integer.
     *
     * @param message     message to display
     * @param prompt      prompt to display
     * @param suggestion  suggested value
     * @param parentFrame parent frame
     *
     * @return Integer object or null if error
     */
    private static Integer queryForInteger(String message, String prompt, int suggestion, JFrame parentFrame) {
        String str = (String)JOptionPane.showInputDialog(parentFrame,
            message,
            prompt,
            JOptionPane.QUESTION_MESSAGE,
            null,
            null,
            Integer.toString(suggestion));
        if (null != str) {
            try {
                return new Integer(Integer.parseInt(str.trim()));
            }
            catch(NumberFormatException e) {
                Toolkit.getDefaultToolkit().beep();
            }
        }
        return null;
    }

    /**
     * Return name of current fish class.
     *
     * @return name of current fish class
     */
    public String getCurrentFish() {
        return _editToolbar.getCurrentFish();
    }

    /**
     * Return current color.
     *
     * @return current color
     */
    public Color getCurrentColor() {
        return _editToolbar.getCurrentColor();
    }

    /**
     * Iteration command for a set number of steps.
     */
    private static class StepItLambda implements ILambda {
        int _steps;
        ISimAdapter _simAdapter;
        DisplayViewport _displayViewport;
        SimToolbar _simToolbar;

        public StepItLambda(ISimAdapter sa, DisplayViewport vp, SimToolbar st) {
            _steps = 0;
            _simAdapter = sa;
            _displayViewport = vp;
            _simToolbar = st;
        }

        public void setSteps(int steps) {
            _steps = steps;
        }

        public Object apply(Object param) {
            _steps--;
            if (0 >= _steps) {
                _simAdapter.stop();
                _simToolbar.setControlsInIdleState();
            }
            _displayViewport.repaint();
            return null;
        }
    }
}
