package view;

import controller.ISimAdapter;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;

/**
 * Toolbar for simulation control.
 *
 * @author Mathias Ricken
 */
public class SimToolbar extends JToolBar {
    /// default value and range for speed slider
    private static final int MIN_DELAY_MSECS = 10;
    private static final int MAX_DELAY_MSECS = 1000;
    private static final int INITIAL_DELAY = MIN_DELAY_MSECS + (MAX_DELAY_MSECS - MIN_DELAY_MSECS) / 2;

    /**
     * RunIdleAdapter.
     */
    IRunIdleAdapter _riAdapter;

    /**
     * Step button.
     */
    JButton _stepButton;

    /**
     * Run button.
     */
    JButton _runButton;

    /**
     * Stop button.
     */
    JButton _stopButton;

    /**
     * Speed slider.
     */
    JSlider _speedSlider;

    /**
     * Simulation adapter.
     */
    ISimAdapter _simAdapter;

    /**
     * Create a new SimToolbar and all its tools.
     *
     * @param sa  simulation adapter
     * @param ria run-idle adapter
     */
    public SimToolbar(ISimAdapter sa, IRunIdleAdapter ria) {
        super(SwingConstants.VERTICAL);
        setFloatable(false);
        makeTools();
        _riAdapter = ria;
        _simAdapter = sa;
    }

    /**
     * Create the UI components for the toolbar.
     */
    private void makeTools() {
        setLayout(new BoxLayout(this, BoxLayout.X_AXIS));
        setBorder(BorderFactory.createTitledBorder("Simulation"));
        add(Box.createRigidArea(new Dimension(5, 0)));

        add(_stepButton = new JButton("Step"));
        add(_runButton = new JButton("Run"));
        add(_stopButton = new JButton("Stop"));
        _runButton.setEnabled(false);
        _stepButton.setEnabled(false);
        _stopButton.setEnabled(false);

        add(Box.createRigidArea(new Dimension(5, 0)));
        add(new JLabel("Slow"));
        _speedSlider = new JSlider(MIN_DELAY_MSECS, MAX_DELAY_MSECS, INITIAL_DELAY);
        _speedSlider.setInverted(true);
        _speedSlider.setPreferredSize(new Dimension(100, _speedSlider.getPreferredSize().height));
        _speedSlider.setMaximumSize(_speedSlider.getPreferredSize());
        add(_speedSlider);
        _speedSlider.setEnabled(false);
        add(new JLabel("Fast"));
        add(Box.createRigidArea(new Dimension(5, 0)));

        _stepButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                setControlsInRunState();
                _simAdapter.step();
                setControlsInIdleState();
            }
        });
        _runButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                setControlsInRunState();
                _simAdapter.setSpeed(_speedSlider.getValue());
                _simAdapter.start();
            }
        });
        _stopButton.addActionListener(new ActionListener() {
            public void actionPerformed(ActionEvent e) {
                setControlsInIdleState();
                _simAdapter.stop();
            }
        });
        _speedSlider.addChangeListener(new ChangeListener() {
            public void stateChanged(ChangeEvent evt) {
                _simAdapter.setSpeed(((JSlider)evt.getSource()).getValue());
            }
        });
    }

    /**
     * Set the controls in idle state.
     */
    public void setControlsInIdleState() {
        _stepButton.setEnabled(true);
        _runButton.setEnabled(true);
        _stopButton.setEnabled(false);
        _speedSlider.setEnabled(true);
        _riAdapter.setIdleState();
    }

    /**
     * Set the controls in run state.
     */
    public void setControlsInRunState() {
        _stepButton.setEnabled(false);
        _runButton.setEnabled(false);
        _stopButton.setEnabled(true);
        _speedSlider.setEnabled(true);
        _riAdapter.setRunState();
    }

    /**
     * Enables or disables the simulation toolbar.
     *
     * @param enabled new state
     */
    public void setEnabled(boolean enabled) {
        //if(isEnabled() != enabled) {
        super.setEnabled(enabled);
        _stepButton.setEnabled(enabled);
        _runButton.setEnabled(enabled);
        _stopButton.setEnabled(false);
        _speedSlider.setEnabled(enabled);
        //}
    }
}
