//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include <fstream>
#include <iostream>
#include <set>

#include <sysexits.h>
#include <sys/wait.h>

#include "ast/attrinfo.h"
#include "analysis/analysis.h"
#include "buildoptions.h"
#include "buildoptionfactory.h"
#include "graph/graphutils.h"
#include "graph/partitioner.h"
#include "profiling/hpcdatabase.h"
#include "profiling/SloDatabase.h"
#include "rpu/rpu.h"
#include "rpu/rpufactory.h"
#include "rpu/rpugraphutil.h"
#include "sourcefile.h"
#include "sourcefileparser.h"
#include "utils/options.h"

#include "llvm/Support/Debug.h"
#include "llvm/Support/ManagedStatic.h"
#include "llvm/Support/raw_ostream.h"
#include "config/llvm.h"

using namespace std;
using namespace aap;

static Agraph_t* buildGraph (void);

static void extractRpus    (RPU_set& rpus);
static void dumpFinalGraph (RPU_set& rpus, Agraph_t &graph);
static void deleteRpus     (RPU_set& rpus);
static void handleBuildOptions (void);
static void outputMakefile (RPU_set&);
static void parseTestFiles (void);
static void printVersion   (void);
static void makeOutputPath (void);

int main(int argc, char **argv, char * const *) {

  Agraph_t* graph = NULL;
  RPU_set rpus;

  llvm::cl::SetVersionPrinter(&printVersion);
  SourceFileParser::Initialize(argv[0]);
  llvm::cl::ParseCommandLineOptions(argc,argv);

  makeOutputPath();

  if (Options::Attributes()) AttrInfo::SetTotalTime(0);

  parseTestFiles();
  handleBuildOptions();

  if (SourceFile::allFiles().empty()) {
      llvm::errs() << "there are no files to parse\n";
      return EXIT_SUCCESS;
  }

  if (aap::safeArrays.getNumOccurrences())
      return aap::ArrayPaddingTest::check();

  if (!Options::ignoreProfile()) graph = buildGraph();
  if (graph) rpus = RpuFactory::ConvertGraph (graph);

  if (Options::parseAllFiles()) SourceFileParser::ParseKnownFiles();

  if (Options::Attributes())
      AttrInfo::HandleAttributesDependences();
  extractRpus (rpus);

  if (graph) {
      dumpFinalGraph (rpus, *graph);

      //if (agisdirected(graph))
      //    agclose (graph);
      //else
      //    GraphUtils::DeleteGraph(graph);
  }
  deleteRpus (rpus);

  if (Options::padArrays()) ArrayPadding::report();
  if (!Options::verbose() && SourceFileParser::DiagnosticsOccured())
      llvm::dbgs() << "use -verbose to view diagnostics\n";

  llvm::llvm_shutdown();

  return EXIT_SUCCESS;
}

// All files should be parsed before invoking extractRpus. Files will not be
// parsed as a side-effect.
static void
extractRpus (RPU_set& rpus)
{
    RPU_set::iterator it, end;

    // determine what source objects are needed for the RPU
    end = rpus.end();
    for (it=rpus.begin(); it != end; ++it)
        (*it)->registerDependencies();

    // account for any leftover source objects
    const map<string, SourceFile*>& files = SourceFile::allFiles();
    map<string, SourceFile*>::const_iterator file_it = files.begin(),
        files_end = files.end();
    while (file_it != files_end) {
        SourceFile* file = (*file_it).second;
        RPU_set* rpu_set = RpuFactory::registerLeftovers (*file);
        rpus.insert (rpu_set->begin(), rpu_set->end());
        delete rpu_set;
        ++file_it;
    }

    // export all of the source objects into RPU files
    for (it=rpus.begin(); it != rpus.end(); it++)
        (*it)->exportToFile();

    // generate a makefile
    outputMakefile (rpus);
}

static void
deleteRpus (RPU_set& rpus)
{
    RPU_set::iterator it = rpus.begin();
    while (it != rpus.end()) {
        delete (*it);
        it++;
    }
}

static Agraph_t*
buildGraph (void)
{
    Agraph_t* graph = NULL;

    if (Options::hasGraph()) {
        graph = HpcDatabase::readGraph (Options::inputGraph());
        SourceFileParser::ParseFilesInGraph (graph);
        return graph;
    }

    string outPath = Options::outPath();
    Partitioner partitioner;
    if ((Options::ProfilingDatabase() == Options::SLODBS) ||
        (Options::ProfilingDatabase() == Options::SLOHPC) ) {
        SloDatabase SD;
        SD.ProcessBRD();
        SD.CreateGraph();
        if (Options::ProfilingDatabase() == Options::SLOHPC) {
            Agraph_t *HpcGraph;
            HpcDatabase hpc;
            HpcGraph = hpc.generateGraph();
            HpcDatabase::assignWeights(HpcGraph);
            SD.IncludeHpcInfo(HpcGraph);
            agclose(HpcGraph);
        }
        SD.CreateInitialSubGraphs();
        graph = SD.GetGraph();
    } else {
        // Assume the profile is HPC unless explicitly SLO.
        HpcDatabase hpc;
        graph = hpc.generateGraph();
        if (graph) {
            HpcDatabase::assignWeights(graph);
            if (Options::ProfilingDatabase() == Options::HPCSLO) {
                Agraph_t *SloGraph;
                SloDatabase SD;
                SD.ProcessBRD();
                SD.CreateGraph();
                SloGraph = SD.GetGraph();
                graph = HpcDatabase::IncludeSLOInfo(graph,SloGraph);
                //agclose(SloGraph);
            }
        }
    }

    if (!graph) return NULL;

    SourceFileParser::ParseFilesInGraph(graph);
    GraphUtils::setLineCounts(graph);

    if (Options::keep_graph ())
        GraphUtils::dump_graph (graph, outPath + "/aap.dot");

    graph = partitioner.partition (graph);
    if (Options::keep_graph())
    {
        GraphUtils::dump_graph (graph, outPath + "/rpu.dot");
        if (Options::breakout()) exit (0);
    }
    return graph;
}

static void
makeOutputPath(void)
{
    std::string errors;
    llvm::sys::Path outPath(Options::outPath());
    if (outPath.createDirectoryOnDisk(true, &errors)) {
        llvm::errs() << errors << "\n";
        exit(EX_CANTCREAT);
    }
}

static string
objFile (const string& source)
{
    llvm::sys::Path obj(source);
    if (obj.getSuffix().compare("c") == 0) {
        obj.eraseSuffix();
    }
    obj.appendSuffix("o");
    return obj.str();
}

static void
outputBuildOptions(ofstream& os, const vector<string>& options)
{
    vector<string>::const_iterator op_it = options.begin();
    while (op_it != options.end()) {
        os << (*op_it) << " ";
        op_it++;
    }
}

static void
outputMakefile (RPU_set& RPUs)
{
    ofstream    makefile;
    set<string> objects;
    string      path (Options::outPath() + "/Makefile");
    string      prog = Options::programName();
    unsigned i;

    makefile.open (path.c_str());

    const vector<string>& cflags = BuildOptions::globalCompileFlags();
    const vector<string>& ldflags = BuildOptions::globalLinkerFlags();

    makefile << "CFLAGS=-x c -Wno-unused-function ";
    if (cflags.size() > 0)
        for (i = 0; i < cflags.size(); i++) makefile << cflags[i] << " ";
    makefile << endl << endl;

    if (ldflags.size() > 0) {
        makefile << "LDFLAGS=";
        for (i = 0; i < ldflags.size(); i++) makefile << ldflags[i] << " ";
        makefile << endl << endl;
    }

    makefile << "all: " << prog << endl << endl;

    RPU_set::iterator rpu_it = RPUs.begin();
    while (rpu_it != RPUs.end()) {
        RPU* rpu = (*rpu_it);
        if (rpu->outputted()) {
            string source = rpu->path();
            string obj = objFile (source);
            const BuildOptions& options = rpu->buildOptions();
            objects.insert (obj);
            makefile << obj << ": " << source << endl;
            makefile << "\t" << "$(CC) " << "$(CFLAGS) ";
            outputBuildOptions (makefile, options.preprocFlags());
            outputBuildOptions (makefile, options.compileFlags());
            makefile << "-c " << source << " -o " << obj;
            makefile << endl << endl;
        }
        rpu_it++;
    }

    makefile << "OBJECTS = ";
    set<string>::iterator obj_it = objects.begin();
    while (obj_it != objects.end()) {
        makefile << *obj_it << " ";
        obj_it++;
    }
    makefile << endl << endl ;
    makefile << prog << ": $(OBJECTS)" << endl;
    makefile << "\t$(CC) -o " << prog << " $(OBJECTS)";
    if (ldflags.size() > 0) makefile << " $(LDFLAGS)";
    makefile << endl << endl;

    makefile << "clean:" << endl;
    makefile << "\t rm -f " << prog << " $(OBJECTS)";
    makefile << endl << endl;

    makefile.close();
}

static void
dumpFinalGraph (RPU_set& rpus, Agraph_t &graph)
{
    if (!Options::keep_graph()) return;

    RPU_set::iterator it;

    for (it = rpus.begin(); it != rpus.end(); it++) {
        const RPU& rpu = *(*it);
        RpuGraphUtil::updateGraph (graph, rpu);
    }
    GraphUtils::dump_graph (&graph, Options::outPath() + "/final.dot");
}

static void
parseTestFiles(void)
{
    if (!testFiles.getNumOccurrences()) return;

    unsigned size = testFiles.size();
    for (unsigned i = 0; i < size; ++i)
        SourceFile::getFileForPath(testFiles[i]);
    SourceFileParser::ParseKnownFiles();
}

static void
printVersion(void)
{
    llvm::dbgs() << "AAP Version 0.1\n"
                 << "\t--help to get usage\n";
}

static void
handleBuildOptions(void)
{
    BuildOptionFactory builder;
    map<string,BuildOptions> buildoptions = builder.parse();
    SourceFile::setCompileFlags(buildoptions);
}
