//===------------------------------------------------------------*- C++ -*-===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#ifndef AAP_ARRAYPADDING_H
#define AAP_ARRAYPADDING_H

#include "analysis/pragmainfo.h"
#include "utils/options.h"

#include "clang/AST/Decl.h"

#include <map>
#include <set>
#include <string>

namespace aap
{

/// Analysis framework to detect arrays that can be safely padded.
class ArrayPadding
{
public:
    /// Indicates that a scope block will close.
    static void pop  (void);

    /// Indicates that a new scope block is being entered. By default, the
    /// scope is unnamed, but function scope should be given a name.
    static void push (const std::string& name = "");

    /// Records a variable name as a potentially safe loop index.
    static void registerSafeIndex      (const std::string&);
    /// Records a variable as unsafe for indexing into a padded array.
    static void markAsUnsafeIndex      (const std::string& name);

    /// Record a global \param array as potentially safe for padding.
    static void registerGlobalArray(const std::string& array,
                                    int dimensions,
                                    int offset = 0);
    /// Record a local \param array as potentially safe for padding.
    static void registerLocalArray(const std::string& array,
                                   int dimensions, //< Number of dimensions.
                                   int offset); //< Line offset within scope.
    /// Records \param array as not safe to pad.
    static void registerUnsafeArray(const clang::ValueDecl* array);

    /// Records the use of a variable as an index into an array.
    static void recordArrayIndex       (const clang::ValueDecl* arrayDecl,
                                        const std::string& indexVariable);

    /// Presents a report of arrays that could be padded
    static void report (void);

    /// Returns true if the padding subsystem is in effect.
    static bool paddingActive (void);

    /// Returns true if the given array is safe for padding.
    static bool shouldPadArray (const std::string&);

    /// Returns the set of array names that are safe for padding.
    static std::set<std::string> safeArrays (void);

    /// Records that the given array is used by the given RPU
    static void associateArrayWithRPU (const std::string& array,
                                       const std::string& rpu);

    static int arrayDimensions (const std::string& arrayName);

    /// Records the given variable as being declared in the local scope.
    static void registerDeclaration (const std::string& var,
                                     int offset = 0);

    static std::set<PragmaInfo>
    paddingPragmasForFunction(const std::string& function);

    /// Returns a pragma to indicate that \param array is safe for padding.
    static PragmaInfo safePaddingPragma(const std::string& array, int offset);

private:
    static const bool kPaddingActive = true;
};

inline bool
ArrayPadding::paddingActive (void)
{
    return kPaddingActive && Options::padArrays();
}

} // namespace aap

#endif
