//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//
//
// Test the array padding system.
//
//===----------------------------------------------------------------------===//
#include <iostream>
#include <set>
#include <string>
#include <sysexits.h>

#include "analysis/arraypadding.h"
#include "analysis/arraypaddingtest.h"
#include "sourcefile.h"
#include "sourcefileparser.h"

#include "llvm/Support/Debug.h"
#include "llvm/Support/raw_ostream.h"

using namespace std;
using namespace llvm;
using namespace aap;

cl::list<string> aap::safeArrays("safe-arrays", cl::Hidden, cl::CommaSeparated,
                                 cl::desc("Arrays expected to be safe (CSV)"));
/// If the test-files option is used the AAP will not attempt to load a build
/// options description file.
cl::list<string> aap::testFiles("test-files", cl::Hidden, cl::CommaSeparated,
                                cl::desc("Files to explicitly parse (CSV)"));

static string kSafePaddingPragma = "AAP ArrayPadding safe";

static void printArray (const set<string>& arrays);
static void printArray (const cl::list<string>& arrays);

enum ErrorCode
{
    kSuccess,
    kFalseNegativeError = 2,
    kFalsePositiveError,
    kMismatchError,
};

int ArrayPaddingTest::check(void)
{
    if (!ArrayPadding::paddingActive()) {
        llvm::errs() << "Support for array padding must be enabled\n";
        exit(EX_USAGE);
    }

    ErrorCode result = kSuccess;
    set<string>::iterator array, end;

    set<string> arrays = ArrayPadding::safeArrays();

    if (arrays.empty()) arrays.insert ("(None)");

    if (arrays.size() > safeArrays.size()) {
        result = kFalsePositiveError;
    } else if (arrays.size() < safeArrays.size()) {
        result = kFalseNegativeError;
    } else {
        for (unsigned i = 0; i < safeArrays.size(); ++i)
            if (arrays.count (safeArrays[i]) == 0) result = kMismatchError;
        }

    if (result != kSuccess) {
        // report the expected arrays
        llvm::errs() << "error: mismatch in ";
        printArray (testFiles);
        llvm::errs() << "\t""expected: ";
        printArray (safeArrays);
        llvm::errs() << "\t""found: ";
        printArray (arrays);
    } else {
        DEBUG (llvm::dbgs() << "padding matches expectations\n");
    }

    return result;
}

static void
printArray (const set<string>& arrays)
{
    set<string>::iterator array, end;
    array = arrays.begin();
    end = arrays.end();
    while (array != end) {
        llvm::errs() << *array << " ";
        ++array;
    }
    llvm::errs() << "\n";
}

static void
printArray (const cl::list<string>& arrays)
{
    for (unsigned i = 0; i < arrays.size(); ++i)
        llvm::errs() << arrays[i] << " ";
    llvm::errs() << "\n";
}
