//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2011, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include "ast/aapaddinfoaction.h"
#include "ast/aapaddinfoconsumer.h"
#include "ast/attrinfo.h"
#include "sourcefile.h"
#include "clang/Basic/SourceManager.h"

#include "llvm/Support/Debug.h"

using namespace clang;
using namespace aap;

static bool AttrPure;
static bool AttrConst;

// Does statement access global or argument variables.
static bool VariableAccess;
// Local pointer variables which are assigned with global or 
// pointer argument variables.
static std::vector<const VarDecl*> PointerLocals;

void AAPAddInfoConsumer::HandleTranslationUnit(ASTContext &context)
{
    DeclContext *DContext;
    DeclContext::decl_iterator DIter, DEnd;
    sourceManager = &context.getSourceManager();
    DContext = TranslationUnitDecl::
        castToDeclContext(context.getTranslationUnitDecl());
    DEnd = DContext->decls_end();
    for (DIter = DContext->decls_begin(); DIter != DEnd; DIter++)
        if ((DIter->getKind()) == Decl::Function)
            HandleFunctionDecl(cast<FunctionDecl>(*DIter));
}

int AAPAddInfoConsumer::HandleFunctionDecl(const FunctionDecl *FD)
{
    AttrInfo *MyInfo;

    if (FD->getStorageClass() == SC_Static)
        MyInfo = AttrInfo::GetOrInsertAttrInfo(FD->getNameAsString(),
                                         AAPAddInfoAction::getFile().name());
    else
        MyInfo = AttrInfo::GetOrInsertAttrInfo(FD->getNameAsString());
    if (MyInfo->IsAnalyzed()) return 0;

    AttrConst = false;
    AttrPure = false;
    for (Decl::attr_iterator AI = FD->attr_begin(), AE = FD->attr_end();
         ((AI != AE) && (AttrPure == false)); AI++)
        if (ConstAttr::classof(*AI)) {
            AttrConst = true;
            AttrPure = true;
        }
        else if (PureAttr::classof(*AI))
            AttrPure = true;
    // Attributes are set by the programmer.
    if (AttrPure == true) {
        MyInfo->SetAnalyzed(true);
        MyInfo->SetPure(true);
        MyInfo->SetDefault();
        MyInfo->SetConst(AttrConst);
        return 0;
    }

    if (FD->getResultType().getTypePtr()->isVoidType()) {
        MyInfo->SetAnalyzed(true);
        MyInfo->SetPure(false);
        MyInfo->SetConst(false);
        return 0;
    }

    // Functions with const attribute cannot have pointer argument 
    // and result type.
    AttrConst = true;
    if (FD->getResultType().getTypePtr()->isPointerType())
        AttrConst = false;
    for (FunctionDecl::param_const_iterator I = FD->param_begin(),
             E = FD->param_end();
         ((I != E) && (AttrConst == true)); I++)
        if ((*I)->getOriginalType().getTypePtr()->isPointerType())
            AttrConst = false;

    // Analyze function body.
    if (FD->isThisDeclarationADefinition()) {
        AttrPure = true;
        VariableAccess = false;
        HandleStmt(MyInfo, FD->getBody());
        if (MyInfo->DependencesSize() > 0) {
            if (!AttrPure)
                MyInfo->ClearDependences();
        }
        else
            MyInfo->SetPure(AttrPure);
        MyInfo->SetConst(AttrConst);
        MyInfo->SetAnalyzed(true);
        PointerLocals.clear();
    }
}

void AAPAddInfoConsumer::HandleStmt (AttrInfo *MyInfo, const Stmt *S)
{
    AttrInfo *FuncInfo;

    if (DeclRefExpr::classof(S)) {
        const DeclRefExpr *DRE = cast<DeclRefExpr>(S);
        const ValueDecl *VD = DRE->getDecl();
        if (const FunctionDecl *FD = dyn_cast<FunctionDecl>(VD)) {
            if (FD->getStorageClass() == SC_Static)
                FuncInfo =
                    AttrInfo::SearchAttr(FD->getNameAsString(),
                                         AAPAddInfoAction::getFile().name());
            else
                FuncInfo = AttrInfo::SearchAttr(FD->getNameAsString());
            if (FuncInfo->IsAnalyzed()) {
                if (FuncInfo->IsPure() == false)
                    if (FuncInfo->DependencesSize())
                        MyInfo->AddDependence(FuncInfo);
                    else {
                        AttrPure = false;
                        AttrConst = false;
                    }
            }
            else
                MyInfo->AddDependence(FuncInfo);
        }
        else if (const VarDecl *Var = dyn_cast<VarDecl>(VD)) {
            // Access global or pointer argument variables
            // or local pointer variables which are assigned with
            // global or pointer argument variables.
            if (Var->hasGlobalStorage()) {
                AttrConst = false;
                if (Var->getType().getTypePtr()->isPointerType())
                    VariableAccess = true;
            }
            else if (ParmVarDecl::classof(Var)) {
                if (Var->getType().getTypePtr()->isPointerType())
                    VariableAccess = true;
            }
            else if ((std::find(PointerLocals.begin(),
                                PointerLocals.end(), Var)) !=
                     PointerLocals.end())
                VariableAccess = true;
               
        }
    }
    else if (const BinaryOperator *BO = dyn_cast<BinaryOperator>(S)) {
        if (BO->isAssignmentOp()) {
            // Left hand side modifies variable and 
            // needs to be treated special
            HandleStmt(MyInfo,BO->getRHS());
            if (AttrPure)
                // Left hand side
                HandleLHSExpr(MyInfo,BO->getLHS()->IgnoreParenCasts());
            VariableAccess = false;
        }
        else {
            HandleStmt(MyInfo,BO->getLHS());
            HandleStmt(MyInfo,BO->getRHS());
        }
    }
    else if (const UnaryOperator *UO = dyn_cast<UnaryOperator>(S)) {
        if (UO->isIncrementDecrementOp())
            // Modifies variable value and needs to be treated special.
            HandleLHSExpr(MyInfo,UO->getSubExpr()->IgnoreParenCasts());
        else
            HandleStmt(MyInfo,UO->getSubExpr()->IgnoreParenCasts());
    }
    else {
        for (Stmt::const_child_iterator I = S->child_begin(),
                 E = S->child_end();
             ((I != E) && (AttrPure == true)); I++) {
            if ((*I) != NULL) HandleStmt(MyInfo, (*I));
        }
    }
}

void AAPAddInfoConsumer::HandleLHSExpr(AttrInfo *MyInfo, const Expr* E)
{
    bool MemberWhile = false;
    bool PointerMember = false;
    E = E->IgnoreParenCasts();
    if (const MemberExpr *ME = dyn_cast<MemberExpr>(E)) {
        // Structure or Union member is modified.
        MemberWhile = true;
        if (ME->getMemberDecl()->getType()->isPointerType())
            PointerMember = true;
        do {
            E = ME->getBase();
            E = E->IgnoreParenCasts();
        } while (ME = dyn_cast<MemberExpr>(E));
    }
    if (const DeclRefExpr *DRE = dyn_cast<DeclRefExpr>(E)) {
        const ValueDecl *VD = DRE->getDecl();
        if (const VarDecl *Var = dyn_cast<VarDecl>(VD)) {
            if (Var->hasGlobalStorage()) {
                // Global variable modified.
                AttrConst = false;
                AttrPure = false;
            }
            else if (Var->getType()->isPointerType()) {
                if (ParmVarDecl::classof(Var))
                    // Pointer parameter modified.
                    AttrPure = false;
                else {
                    if ((MemberWhile == true) &&
                        ((std::find(PointerLocals.begin(),
                                    PointerLocals.end(), Var)) !=
                         PointerLocals.end()) ) {
                        // A structure or union member of a local pointer
                        // variable which points to pointer argument or
                        // global variable is modified.
                        AttrConst = false;
                        AttrPure = false;
                    }
                    else if (VariableAccess)
                        // A global variable or pointer argument is
                        // assigned to local pointer variable.
                        PointerLocals.push_back(Var);
                }
            }
            else {
                if ((MemberWhile == true) &&
                    ((std::find(PointerLocals.begin(),
                                PointerLocals.end(), Var)) !=
                     PointerLocals.end()) ) {
                    // A structure or union member of a local variable
                    // with a pointer member which points to pointer argument
                    // or global variable is modified.
                    AttrConst = false;
                    AttrPure = false;
                }
                else if (PointerMember && VariableAccess)
                    // A global variable or pointer argument is assigned to
                    // a pointer struct/union member of a 
                    // local struct/union variable.
                    PointerLocals.push_back(Var);
            }
        }
    }
    else if (const BinaryOperator *BO = dyn_cast<BinaryOperator>(E)) {
        // Pointer arithmetic on left hand side expression.
        HandleLHSExpr(MyInfo,BO->getLHS()->IgnoreParenCasts());
        HandleLHSExpr(MyInfo,BO->getRHS()->IgnoreParenCasts());
    }
    else if (const UnaryOperator *UO = dyn_cast<UnaryOperator>(E)) {
        // Pointer dereference or increment/decrement on left hand side.
        HandleLHSExpr(MyInfo,UO->getSubExpr()->IgnoreParenCasts());
    }
    else if (const ArraySubscriptExpr *ASE =
             dyn_cast<ArraySubscriptExpr>(E)) {
        // Array indexing
        HandleStmt(MyInfo,ASE->getIdx());
        HandleLHSExpr(MyInfo,ASE->getBase());
    }
}

