//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2010, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include "ast/aapaction.h"
#include "ast/aapconsumer.h"
#include "ast/attrinfo.h"
#include "analysis/arraypadding.h"
#include "sourcefile.h"
#include "utils/options.h"

#include <sstream>

#include "clang/Basic/SourceManager.h"
#include "clang/Basic/Specifiers.h"
#include "clang/AST/OperationKinds.h"
#include "clang/AST/TypeLoc.h"
#include "llvm/Support/Debug.h"
#include "clang/Basic/AttrKinds.h"

using namespace clang;
using namespace aap;

const static std::string StrFunction   ("function ");
const static std::string StrEnum       ("enum ");
const static std::string StrStruct     ("struct ");
const static std::string StrUnion      ("union ");
const static std::string StrTypedef    ("typedef ");
const static std::string StrStatic     ("static ");
const static std::string StrExtern     ("extern ");
const static std::string StrDefinition (" definition");
const static std::string StrTypeof     ("typeof");
const static std::string StrAnonymous  ("<anonymous ");
const static std::string StrRestrict   ("restrict");
const static std::string StrConst      ("const");

static SourceObjRef NewSourceObj(const std::string&, clang::SourceRange&,
                                 clang::SourceManager*, const std::string& = "",
                                 const std::string& = "",
                                 unsigned = 1, unsigned = 0);

/// Processes parsed translation unit (file). 
/// Global or Static declarations of translation unit is determined and 
/// the corresponding processing functions are called. 
/// The determined global decrations are records (structures and unions), 
/// enums, typedefs and functions.
void AAPConsumer::HandleTranslationUnit(ASTContext &context)
{
    DeclContext *DContext;
    DeclContext::decl_iterator DIter, DEnd;
    sourceManager_ = &context.getSourceManager();
    DContext = TranslationUnitDecl::
        castToDeclContext(context.getTranslationUnitDecl());
    DEnd = DContext->decls_end();
    for (DIter = DContext->decls_begin(); DIter != DEnd; DIter++) {
        switch (DIter->getKind()) {
        case Decl::Enum:
            HandleEnumDecl(cast<EnumDecl>(*DIter));
            break;
        case Decl::Record:
            HandleRecordDecl(cast<RecordDecl>(*DIter));
            break;
        case Decl::Typedef:
            HandleTypedefDecl(cast<TypedefDecl>(*DIter));
            break;
        case Decl::Function:
            HandleFunctionDecl(cast<FunctionDecl>(*DIter));
            break;
        case Decl::Var:
            HandleVarDecl(cast<VarDecl>(*DIter));
            break;
        default:
            llvm::errs() << "Not Handled (" << DIter->getKind() << "): ";
            (*DIter)->dump();
            llvm::errs() << "\n";
        }
    }

    // Registers objects of the global declaration to the source file class
    std::map<const NamedDecl*,SourceObjRef>::iterator I,E;
    for (I = ObjectsMap.begin(), E = ObjectsMap.end(); I != E; I++)
        if (I->second) AAPAction::getFile().registerDef(I->second);
    ObjectsMap.clear();
}


/// Registers an enumeration as being declared in the current source file.
/// Anonymous enums are given a unique name, if they are not part of typedef.
void AAPConsumer::HandleEnumDecl(const EnumDecl *ED)
{
    std::string Name;
    SourceRange SR = ED->getSourceRange();
    const NamedDecl *ND = ED;
    SourceObjRef EnumObj;

    if ((ED->getNameAsString()).empty()) {
        // Anonymous enum
        TypedefDecl *TD = ED->getTypedefForAnonDecl();
        if (TD) {
            // Anonymous enum is part of typedef declaration and
            // source object is created as typedef object.
            Name = TD->getNameAsString();
            EnumObj = NewSourceObj(Name, SR, sourceManager_, StrTypedef, Name);
            ND = TD;
        } else {
            // Unique name is given to the anonymous enum.
            PresumedLoc PL =
                sourceManager_->getPresumedLoc(SR.getBegin());
            std::string Buffer = PL.getFilename();
            size_t Found = Buffer.rfind("/");
            if (Found != std::string::npos)
                Buffer.erase(0,Found+1);
            Found = Buffer.find(".");
            while (Found != std::string::npos) {
                Buffer.replace(Found,1,"_");
                Found = Buffer.find(".",Found+1);
            }
            std::stringstream Stream;
            Stream << StrEnum << "Anonymous_" << Buffer << "_"
                   << PL.getLine() << "_" << PL.getColumn();
            Name = Stream.str();
            EnumObj = NewSourceObj(Name, SR, sourceManager_, Name, "", 1, 4);
        }
    } else {
        // Named enum
        Name = StrEnum+ED->getNameAsString();
        EnumObj = NewSourceObj(Name,SR,sourceManager_);
    }

    ObjectsMap[ND] = EnumObj;
    HandleEnumConsts(EnumObj, ED);
}

/// Registers a record as being declared in the current source file.
/// A record can be a structure or union.
/// Anonymous records are given a unique name if they are not part of typedef.
void AAPConsumer::HandleRecordDecl(const RecordDecl *RD)
{
    unsigned int Offset;
    std::string Name;
    SourceRange SR = RD->getSourceRange();
    Type *T = RD->getTypeForDecl();
    SourceObjRef RecordObj;
    const NamedDecl *ND = RD;

    if (T->isUnionType()) {
        Offset = 5;
        Name = StrUnion;
    } else {
        Offset = 6;
        Name = StrStruct;
    }

    if (RD->field_empty()) {
        // The record is only declared without defining fields.
        clang::SourceLocation SL;
        SR.setEnd(SL);
        Name.append(RD->getNameAsString());
        RecordObj = NewSourceObj (Name, SR, sourceManager_, Name);
        // Declaration needs to depend on definition.
        RecordObj->addDependency(Name+StrDefinition);
    } else {
        if ((RD->getNameAsString()).empty()) {
            // Anonymous record
            TypedefDecl *TD = RD->getTypedefForAnonDecl();
            if (TD) {
                // Anonymous record is a part of typedef declarations and
                // source object is created as typedef object
                Name = TD->getNameAsString();
                RecordObj = NewSourceObj(Name, SR, sourceManager_,
                                         StrTypedef, Name);
                ND = TD;
            } else {
                // Unique name is given to the anonymous record.
                PresumedLoc PL =
                    sourceManager_->getPresumedLoc(SR.getBegin());
                std::string Buffer = PL.getFilename();
                size_t Found = Buffer.rfind("/");
                if (Found != std::string::npos)
                    Buffer.erase(0,Found+1);
                Found = Buffer.find(".");
                while (Found != std::string::npos) {
                    Buffer.replace(Found,1,"_");
                    Found = Buffer.find(".",Found+1);
                }
                std::stringstream Stream;
                Stream << Name << "Anonymous_" << Buffer << "_"
                       << PL.getLine() << "_" << PL.getColumn();
                Name = Stream.str();
                RecordObj = NewSourceObj(Name, SR, sourceManager_,
                                         Name, "", 1, Offset);
            }
        } else {
            // Named Record
            Name.append(RD->getNameAsString());
            Name.append(StrDefinition);
            RecordObj = NewSourceObj(Name,SR, sourceManager_);
        }
        RecordObj->setIsDefinition(true);
    }
    ObjectsMap[ND] = RecordObj;
    HandleRecordFields(RecordObj,RD);
}

/// Registers a typedef as being declared in the current source file.
/// Typedefs for anonynous enums or records are not handled.
/// They are handled in HandleRecordDecl and HandleEnumDecl, respectively.
void AAPConsumer::HandleTypedefDecl(const TypedefDecl *TD)
{
    std::string Name (TD->getNameAsString());
    SourceObjRef TDObj;
    SourceRange SR = TD->getSourceRange();
    clang::SourceLocation SL;
    Type *T;
    T = (TD->getUnderlyingType()).getTypePtr();
    SR.setEnd(SL);
    // Typedef declarations of anonymous enums or records have same type
    // name as typedef declaration name and they are treated in functions
    // HandleRecordDecl and HandleEnumDecl.
    if (Name.compare(TD->getUnderlyingType().getAsString()) == 0) return;

    LangOptions LO;
    std::string TypeName = Name;
    TD->getUnderlyingType().getAsStringInternal(TypeName,
                                                PrintingPolicy(LO));
    TypeName.insert(0,StrTypedef);
    size_t Loc = TypeName.find(StrTypeof);
    if (Loc != std::string::npos)
        TypeName.replace(Loc,StrTypeof.length(),"__typeof__");
    Loc = TypeName.find(StrRestrict);
    if (Loc != std::string::npos)
        TypeName.replace(Loc,StrRestrict.length(),"__restrict");
    TDObj = NewSourceObj(Name,SR, sourceManager_,TypeName);
    ObjectsMap[TD] = TDObj;
    TDObj->setIsStatic(TD->getLinkage() == InternalLinkage);
    HandleDifferentTypes(TDObj,T);
}

static int arrayDimension(const ConstantArrayType* CAT)
{
    int dimension = 0;
    const ConstantArrayType* T = CAT;
    while (T) {
        ++dimension;
        T = dyn_cast<ConstantArrayType>((T->getElementType()).getTypePtr());
    }
    return dimension;
}


/// Registers global and static variables as being declared in the current
/// source file. Global or static variables can be of type named enum type,
/// named record type, typedef type, built-in type (int,float,...), 
/// anonymous enum type and anonymous record type. 
/// Global variables can also have default value.
void AAPConsumer::HandleVarDecl(const VarDecl *VD)
{
    std::string Name (VD->getNameAsString());
    SourceRange SR = VD->getSourceRange();
    std::string VarName, PostName;
    Type *T;

    if (VD->getStorageClass() == SC_Static)
        VarName = StrStatic;
    else if (VD->getStorageClass() == SC_Extern)
        VarName = StrExtern;

    T = (VD->getType()).getTypePtr();

    if (ArrayPadding::paddingActive() && ConstantArrayType::classof(T)) {
        int dimension = arrayDimension(cast<ConstantArrayType>(T));
        int pos = 0;
        if (function_) {
            int funcPos = function_->source().first_line();
            clang::SourceLocation decLoc = VD->getLocStart();
            int decPos = sourceManager_->getInstantiationLineNumber(decLoc);
            pos = decPos - funcPos;
        }
        ArrayPadding::registerGlobalArray(Name, dimension, pos);
    }

    std::string TypeName = Name;
    LangOptions LO;
    VD->getType().getAsStringInternal(TypeName,PrintingPolicy(LO));
    size_t Loc = TypeName.find(StrAnonymous);
    if (Loc != std::string::npos) {
        // Global variable of type anonymous enum or record.
        // The unique name given in functions HandleEnumDecl and
        // HandleRecordDecl is used as type name.
        Type *NT = T;
        while (ArrayType::classof(NT)){
            ArrayType *AT = dyn_cast<ArrayType>(NT);
            if (AT) NT = (AT->getElementType()).getTypePtr();
        }
        const TagType *Tag = dyn_cast<TagType>(NT);
        const TagDecl *TagD = Tag->getDecl();
        TypeName.replace(TypeName.begin() + Loc,
                         TypeName.begin() + TypeName.find(">") + 1,
                         ObjectsMap[TagD]->name());
        if (TagD->isEnum()) {
            Loc = TypeName.find(StrEnum);
            TypeName.replace(Loc,StrEnum.size(),"");
        } else if (TagD->isUnion()) {
            Loc = TypeName.find(StrUnion);
            TypeName.replace(Loc,StrUnion.size(),"");
        } else {
            Loc = TypeName.find(StrStruct);
            TypeName.replace(Loc,StrStruct.size(),"");
        }
    }

    SR.setEnd(clang::SourceLocation());
    VarName.append(TypeName);

    unsigned int Offset = 0;
    const Expr *E = VD->getInit();
    if (E) {
        // Global variable is given a default value.
        E = E->IgnoreParenCasts();
        std::ostringstream Stream;
        while (CastExpr::classof(E))
            E = cast<CastExpr>(E)->getSubExpr();
        const UnaryOperator *UO = NULL;
        if (UnaryOperator::classof(E)) {
            // The expression is enclosed in a unary operator such as ++, --,
            // &, etc. The unary operator is considered and expression enclosed
            // by unary operator extracted.
            UO = cast<UnaryOperator>(E);
            E = UO->getSubExpr();
            Stream << " = ";
            if (UO->isPostfix() == false)
                Stream << UnaryOperator::getOpcodeStr(UO->getOpcode());
            Stream << "(";
        }

        if (CharacterLiteral::classof(E)) {
            // Initializing variable is character. The integer value is 
            // extracted and a cast to char is used.
            if (UO) Stream <<    "(char)";
            else    Stream << " = (char)";
            Stream << cast<CharacterLiteral>(E)->getValue();
            PostName = Stream.str();
            Stream.clear();
        } else if (IntegerLiteral::classof(E)) {
            // Initialized using an integer
            if (!UO) Stream << " = ";
            std::string Value =
                cast<IntegerLiteral>(E)->getValue().toString(10,true);
            Stream << Value;
            PostName.append(Stream.str());
            Stream.clear();
        } else if (FloatingLiteral::classof(E)) {
            // Initialized by a floating point constant
            if (!UO) Stream << " = ";
            Stream << cast<FloatingLiteral>(E)->getValue().convertToDouble();
            PostName = Stream.str();
            Stream.clear();
        } else if (StringLiteral::classof(E)) {
            // Initialized by string constant.
            // The special characters "\n" and "\" within string constant
            // needs to be modified, so it does not process those characters
            // as special characters during extraction.
            std::string Value = cast<StringLiteral>(E)->getString().str();
            size_t Found;
            Found = Value.find("\n");
            while (Found != std::string::npos) {
                Value.replace(Found,1,"\\n");
                Found = Value.find("\n",Found+2);
            }
            Found = Value.find("\"");
            while (Found != std::string::npos) {
                Value.replace(Found,1,"\\\"");
                Found = Value.find("\"",Found+2);
            }
            if (!UO) Stream << " = ";
            Stream << "\"" << Value << "\"";
            PostName = Stream.str();
            Stream.clear();
        } else if (InitListExpr::classof(E)) {
            // Use source range from the file for initialization expression
            SourceRange ESR = E->getSourceRange();
            Offset = 1;
            VarName.append(" =");
            SR.setBegin(E->getLocStart());
            SR.setEnd(E->getLocEnd());
        } else if (DeclRefExpr::classof(E)) {
            // The initializing value is a variable reference.
            const DeclRefExpr *DRE = cast<DeclRefExpr>(E);
            const ValueDecl *Val = DRE->getDecl();
            if (!UO) Stream << " = ";
            Stream << Val->getNameAsString();
            PostName = Stream.str();
            Stream.clear();
        }

        if (UO) {
            if (UO->isPostfix())
                PostName.append(UnaryOperator::getOpcodeStr(UO->getOpcode()));
            PostName.append(")");
        }
    }

    SourceObjRef VarObj = NewSourceObj(Name, SR, sourceManager_,
                                VarName, PostName, Offset);

    if (E) HandleStmt(VarObj,E);
    HandleDifferentTypes(VarObj,T);

    ObjectsMap[VD] = VarObj;
    VarObj->setIsVariable (true);
    if (VD->getStorageClass() == SC_Static)
        VarObj->setIsStatic (true);
    else if (VD->getStorageClass() == SC_Extern)
        VarObj->setIsExtern (true);
}

/// Registers a function as being declared in the current source file.
/// Handles both prototype declarations and complete definitions.
void AAPConsumer::HandleFunctionDecl(const FunctionDecl *FD)
{
    bool Redeclaration;
    std::string Name, Result, Declaration, Current;
    SourceRange SRDef = FD->getSourceRange();
    SourceRange SRDec = FD->getSourceRange();
    clang::SourceLocation SL;

    if (!SRDec.getEnd().isValid()) return;

    // Check if this function is declared before,
    // creates the prototype only one time.
    Redeclaration = false;
    for (FunctionDecl::redecl_iterator I = FD->redecls_begin(),
             E = FD->redecls_end(); I != E; I++) {
        if (ObjectsMap.find(*I) != ObjectsMap.end())
            Redeclaration = true;
    }
    Name = FD->getNameAsString();
    if (FD->getStorageClass() == SC_Static)
        Result = StrStatic;
    else if (FD->getStorageClass() == SC_Extern)
        Result = StrExtern;
    Result.append(FD->getResultType().getAsString());
    if (Redeclaration == false) {
        // Create prototype for the function.
        LangOptions LO;
        Declaration = Result + " " + Name + " (";
        SRDec.setEnd(SL);
        Current = "";
        for (FunctionDecl::param_const_iterator I = FD->param_begin(),
                 E = FD->param_end(); I != E; I++) {
            Declaration.append(Current);
            Current = (*I)->getNameAsString();
            (*I)->getType().getAsStringInternal(Current,
                                                PrintingPolicy(LO));
            size_t Loc = Current.find(StrRestrict);
            if (Loc != std::string::npos)
                Current.replace(Loc,StrRestrict.length(),"__restrict");
            Loc = Current.find(StrConst);
            if (Loc != std::string::npos)
                Current.replace(Loc,StrConst.length(),"__const");
            Declaration.append(Current);
            Current = ", ";
        }
        const FunctionType *FT = FD->getType()->getAs<FunctionType>();
        const FunctionProtoType *FPT = dyn_cast<FunctionProtoType>(FT);
        if (FPT && FPT->isVariadic())
                Declaration.append(", ...");
        Declaration.append(")");
        SourceObjRef Func
            = NewSourceObj(Name, SRDec, sourceManager_, Declaration);
        if (Options::Attributes()) {
            if (FD->getStorageClass() == SC_Static)
                AttrInfo::InsertSourceObj(Func,Name,
                                          AAPAction::getFile().name());
            else
                AttrInfo::InsertSourceObj(Func,Name);
        }
        Func->setIsFunction (true);
        if (FD->getStorageClass() == SC_Static)
            Func->setIsStatic (true);
        else if (FD->getStorageClass() == SC_Extern)
            Func->setIsExtern (true);
        ObjectsMap[FD] = Func;
        HandleFunctionParams(Func,FD);
    }
    if (FD->isThisDeclarationADefinition()) {
        // Process function body
        Name.insert(0,StrFunction);
        function_ = NewSourceObj(Name, SRDef, sourceManager_, Result);
        function_->setIsFunction (true);
        function_->setIsStatic(FD->getStorageClass() == SC_Static);
        HandleFunctionParams(function_,FD);
        function_->setIsDefinition(true);
        ArrayPadding::push(function_->globalName());
        HandleStmt(function_,FD->getBody());
        ArrayPadding::pop();
        AAPAction::getFile().registerDef(function_);
        function_ = NULL;
    }
}

/// Checks an assignment to determine if either side of the expression would
/// invalidate the safety of padding an array.
static void checkAssignment (const BinaryOperator* oper)
{
    if (!oper->isAssignmentOp()) return;
    // Increment and decrement are safe.
    if (oper->getOpcode() == BO_AddAssign) return;
    if (oper->getOpcode() == BO_SubAssign) return;

    Expr* rhs = oper->getRHS()->IgnoreParenCasts();
    Expr* lhs = oper->getLHS()->IgnoreParenCasts();
    DeclRefExpr* DRE = dyn_cast<DeclRefExpr>(lhs);

    // Assignments to something other than a simple variable are not a concern.
    if (!DRE) return;

    // Assignments to a constant value are safe.
    if (IntegerLiteral::classof(rhs)) {
        ArrayPadding::registerSafeIndex(DRE->getDecl()->getName());
        return;
    }

    // Any other assignment makes the LHS unsafe as an array index.
    ArrayPadding::markAsUnsafeIndex(DRE->getDecl()->getName());
    DEBUG (llvm::dbgs() << "\t""the LHS of an assignment\n");

    // An array on the RHS of an assignment might be aliasing
    if (DeclRefExpr::classof(rhs)) {
        DeclRefExpr* DRE = cast<DeclRefExpr>(rhs);
        llvm::StringRef name = DRE->getDecl()->getName();
        ArrayPadding::registerUnsafeArray(DRE->getDecl());
        DEBUG (llvm::dbgs() << "\t""the RHS of an assignment\n");
    }
}

static void checkUnaryOperator (const UnaryOperator *oper)
{
    // We allow ++ / --, but anything else makes the variable unsafe.
    if (oper->isIncrementDecrementOp()) return;

    Expr* sub = oper->getSubExpr()->IgnoreParenCasts();
    if (DeclRefExpr::classof(sub)) {
        ValueDecl* decl = cast<DeclRefExpr>(sub)->getDecl();
        llvm::StringRef name = decl->getName();
        ArrayPadding::registerUnsafeArray(decl);
        DEBUG (llvm::dbgs() << "\t""array used in unary operator\n");
        ArrayPadding::markAsUnsafeIndex(name);
    }
}

// Handles the expressions found within an array subscript. They are valid when
// the subscript is an integer constant, a simple variable, or a simple variable
// plus/minus an integer constant.
static void checkSubscriptExpr (const clang::ValueDecl* decl, const Expr* rhs)
{
    llvm::StringRef name = decl->getName();

    // constants are always safe as an array subscript
    if (IntegerLiteral::classof(rhs)) return;

    if (isa<DeclRefExpr>(rhs)) {
        const ValueDecl* index = cast<DeclRefExpr>(rhs)->getDecl();
        ArrayPadding::recordArrayIndex(decl, index->getName());
        return;
    }

    const BinaryOperator* bin = dyn_cast<BinaryOperator>(rhs);
    if (bin != NULL) {
        if (!bin->isAdditiveOp()) {
            ArrayPadding::registerUnsafeArray(decl);
            DEBUG (llvm::dbgs() <<"\t""index a non-additive operation\n");
            return;
        }
        const Expr* binRhs = bin->getRHS()->IgnoreParenCasts();
        const Expr* binLhs = bin->getLHS()->IgnoreParenCasts();
        if (isa<IntegerLiteral>(binRhs)) {
            checkSubscriptExpr (decl, binLhs);
            return;
        }
        if (isa<IntegerLiteral>(binLhs)) {
            checkSubscriptExpr (decl, binRhs);
            return;
        }
    }

    ArrayPadding::registerUnsafeArray(decl);
    DEBUG (llvm::dbgs() <<"\t""index isn't integer or variable\n");
}

static void verifyArrayDimensions(const VarDecl* var, int dimensions)
{
    llvm::StringRef name = var->getName();

    int registeredDimensions = ArrayPadding::arrayDimensions (name);
    if (registeredDimensions && (registeredDimensions != dimensions)) {
        ArrayPadding::registerUnsafeArray(var);
        DEBUG (llvm::dbgs() << "\t""dimensions do not match\n");
    }
}

/// Handles expressions with array subscripts specially. When array padding
/// analysis is active, we want to know what other variables are used to index
/// into an array. Because any occurrence of an array without subscripts is
/// considered unsafe for padding, the references to the array variable
/// declaration are handled specifically here.
void AAPConsumer::HandleArrayExpr (SourceObjRef obj,
                                   const ArraySubscriptExpr *ASE)
{
    // Determine the dimensionality of the array and the base variable by
    // chasing down the LHS (a series of subscript expressions, the last of
    // which has a declaration reference).
    const Expr *rhs, *lhs = ASE->getLHS()->IgnoreParenCasts();
    int dimensions = 1;
    while(ArraySubscriptExpr::classof(lhs)) {
        lhs = cast<ArraySubscriptExpr>(lhs)->getLHS()->IgnoreParenCasts();
        ++dimensions;
    }

    while (const MemberExpr* member = dyn_cast<MemberExpr>(lhs))
        lhs = member->getBase()->IgnoreParenCasts();

    const DeclRefExpr* DRE = dyn_cast<DeclRefExpr>(lhs);

    if (!DRE) {
        llvm::errs() << "LHS of array does not reference a declaration\n";
        ASE->dump();
        return;
    }

    const VarDecl* varDecl = dyn_cast<VarDecl>(DRE->getDecl());
    if (!varDecl) {
        llvm::errs() << "LHS of array does not reference a variable\n";
        ASE->dump();
        return;
    }

    HandleVarDeclRef(obj, varDecl);

    verifyArrayDimensions(varDecl, dimensions);

    // Do a complete pass again, handling the RHS expressions along the way. In
    // addition to the default statement handling, record variables used to
    // index the array. If anything other than a simple variable or integer
    // constant is used as a subscript, mark the array as unsafe for padding.
    lhs = ASE;
    do {
        ASE = cast<ArraySubscriptExpr>(lhs);
        lhs = ASE->getLHS()->IgnoreParenCasts();
        rhs = ASE->getRHS()->IgnoreParenCasts();
        checkSubscriptExpr (varDecl, rhs);
        HandleStmt (obj, rhs);
    } while(ArraySubscriptExpr::classof(lhs));
}

/// Statements can access already declared variables, fields,
/// enum constants and functions. It can also declare new variables.
void AAPConsumer::HandleStmt (SourceObjRef Obj, const Stmt *S)
{
    std::string Dependence;
    const DeclRefExpr *DRE;
    const ValueDecl *VD;
    std::map<const NamedDecl*,SourceObjRef>::iterator ObjectsIter;

    if (ArrayPadding::paddingActive() && isa<CompoundStmt>(S))
        ArrayPadding::push();

    if (DeclRefExpr::classof(S)) {
        // Accesses Variables or functions or fields or enum constants
        DRE = cast<DeclRefExpr>(S);
        VD = DRE->getDecl();
        if (FunctionDecl::classof(VD)) {
            // Function call
            Obj->addDependency (VD->getNameAsString());
        } else if (const VarDecl *Var = dyn_cast<VarDecl>(VD)) {
            HandleVarDeclRef(Obj, Var);
            // If padding analysis is active, array subscript expressions are
            // handled explicitly. Anything seen here is unsafe to pad.
            ArrayPadding::registerUnsafeArray(VD);
        } else if (const FieldDecl *FD = dyn_cast<FieldDecl>(VD)) {
            // Accesses structure field
            const RecordDecl *RD = FD->getParent();
            Obj->addDependency (ObjectsMap[RD]->name());
        } else if (const EnumConstantDecl *ECD =
                 dyn_cast<EnumConstantDecl>(VD)) {
            // Accesses enum constants
            const DeclContext *DC = ECD->getDeclContext();
            const EnumDecl *ED = dyn_cast<EnumDecl>(DC);
            ObjectsIter = ObjectsMap.find(ED);
            if (ObjectsIter != ObjectsMap.end()) {
                Obj->addDependency (ObjectsIter->second->name());
            } else if (const TypedefDecl *TD = ED->getTypedefForAnonDecl()) {
                ObjectsIter = ObjectsMap.find(TD);
                if (ObjectsIter != ObjectsMap.end())
                    Obj->addDependency (ObjectsIter->second->name());
            }
        }
    } else if (const DeclStmt *DS = dyn_cast<DeclStmt>(S)) {
        for (DeclStmt::const_decl_iterator I = DS->decl_begin(),
                 E = DS->decl_end(); I != E; I++)
        {
            const VarDecl *Var = dyn_cast<VarDecl>(*I);
            if (!Var) continue;
            if (Var->hasGlobalStorage()) {
                // The declaration declares a static or extern variable.
                HandleVarDecl(Var);
                ObjectsIter = ObjectsMap.find(Var);
                if (ObjectsIter != ObjectsMap.end()) {
                    if (Var->getStorageClass() == SC_Static) {
                        // No dependency added, because multiple
                        // declarations of static variables not possible.
                        ObjectsIter->second->setFunctionScope (Obj->name());
                    } else {
                        // Dependency added for extern variable.
                        // Multiple declarations in a file is possible.
                        Obj->addDependency (ObjectsIter->second->name());
                    }
                }
            } else {
                // If not a global variable, and is an array declaration,
                // add it to the list of local arrays.
                ConstantArrayType* array
                    = dyn_cast<ConstantArrayType>(Var->getType().getTypePtr());
                if (ArrayPadding::paddingActive() && array) {
                    std::string name = Var->getNameAsString();
                    int dimension = arrayDimension(array);
                    int functionStart = Obj->source().first_line();
                    int declarationStart = sourceManager_->getInstantiationLineNumber(Var->getLocStart());
                    int pos = declarationStart - functionStart;
                    ArrayPadding::registerLocalArray(name, dimension, pos);
                }
            }
            HandleDifferentTypes(Obj,(Var->getType()).getTypePtr());

            if (ArrayPadding::paddingActive())
                ArrayPadding::registerDeclaration(Var->getName());
        }
    } else if (ExplicitCastExpr::classof(S)) {
        const ExplicitCastExpr *ECE = cast <ExplicitCastExpr>(S);
        HandleDifferentTypes(Obj,ECE->getType().getTypePtr());
    } else if (SizeOfAlignOfExpr::classof(S)) {
        const SizeOfAlignOfExpr *SAE = cast<SizeOfAlignOfExpr>(S);
        HandleDifferentTypes(Obj,SAE->getTypeOfArgument().getTypePtr());
    }

    if (ArrayPadding::paddingActive()) {
        if (BinaryOperator::classof(S))
            checkAssignment(cast<BinaryOperator>(S));
        else if (UnaryOperator::classof(S))
            checkUnaryOperator(cast<UnaryOperator>(S));
        else if (ArraySubscriptExpr::classof(S)) {
            HandleArrayExpr(Obj, cast<ArraySubscriptExpr>(S));
            // HandleArrayExpr handles it's sub-expressions explicitly, so don't
            // handle it's children again (below).
            return;
        }
    }

    for (Stmt::const_child_iterator I = S->child_begin(),
             E = S->child_end(); I != E; I++) {
        if ((*I) != NULL) HandleStmt(Obj, (*I));
    }

    if (ArrayPadding::paddingActive() && isa<CompoundStmt>(S))
        ArrayPadding::pop();
}

/// Adds enum type, record types and typedef types as dependency to the
/// source object Obj. Built-in types (int,float,...) are not considered.
void AAPConsumer::HandleDifferentTypes(SourceObjRef Obj, Type *T)
{
    std::map<const NamedDecl*,SourceObjRef>::iterator MapIter;
    std::string Name;
    TagDecl *TagD = NULL;
    NamedDecl *ND = NULL;

    if (TypedefType::classof(T)) {
        // Typedef type is checked before other checking.
        const TypedefType *TT = cast<TypedefType>(T);
        ND = TT->getDecl();
        // The types starting with string __builtin don't
        // need to be considered as dependency.
        size_t Loc = ND->getNameAsString().find("__builtin");
        if (Loc == 0) ND = NULL;

    } else {

        while (ArrayType::classof(T)) {
            if (ArrayType *AT = dyn_cast<ArrayType>(T)) {
                T = (AT->getElementType()).getTypePtr();
                if (VariableArrayType::classof(AT)) {
                    VariableArrayType *VAT = dyn_cast<VariableArrayType>(AT);
                    if (VAT)  HandleStmt(Obj,VAT->getSizeExpr());
                }
            }
        }

        while (PointerType::classof(T))
            T = (T->getPointeeType()).getTypePtr();

        if (TypedefType::classof(T)) {
            const TypedefType *TT = cast<TypedefType>(T);
            ND = TT->getDecl();
        } else if (T->isStructureType()) {
            const RecordType *RT = T->getAsStructureType();
            if (RT) TagD = RT->getDecl();
            Name = StrStruct;
        } else if (T->isUnionType()) {
            const RecordType *RT = T->getAsUnionType();
            if (RT) TagD = RT->getDecl();
            Name = StrUnion;
        } else if (T->isEnumeralType()) {
            const EnumType *ET = dyn_cast<EnumType>(T);
            if (ET) TagD = ET->getDecl();
            Name = StrEnum;
        } else if (T->isFunctionProtoType()) {
            const FunctionProtoType *FPT = dyn_cast<FunctionProtoType>(T);
            if (FPT) {
                HandleDifferentTypes(Obj,FPT->getResultType().getTypePtr());
                for (FunctionProtoType::arg_type_iterator
                         AI = FPT->arg_type_begin(), AE = FPT->arg_type_end();
                     AI != AE; AI++) {
                    HandleDifferentTypes(Obj,(*AI).getTypePtr());
                }
            }
        }
    }

    if (ND) {
        MapIter = ObjectsMap.find(ND);
        if (MapIter != ObjectsMap.end())
            Obj->addDependency (MapIter->second->name());
    } else if (TagD) {
        MapIter = ObjectsMap.find(TagD);
        TagDecl::redecl_iterator I,E;
        I = TagD->redecls_begin();
        E = TagD->redecls_end();
        while ((MapIter == ObjectsMap.end()) && (I != E)) {
            MapIter = ObjectsMap.find(*I);
            I++;
        }
        if (MapIter != ObjectsMap.end()) {
            Obj->addDependency (MapIter->second->name());
        } else if (TagD->getNameAsString().empty() == false) {
            Name.append(TagD->getNameAsString());
            Obj->addDependency (Name);
        }
    }
}

/// Creates dependencies in \param Obj on the types used \param RD.
void AAPConsumer::HandleRecordFields(SourceObjRef Obj, const RecordDecl *RD)
{
    for (RecordDecl::field_iterator I = RD->field_begin(),
             E = RD->field_end(); I != E; I++)
    {
        Type* T = ((*I)->getType()).getTypePtr();
        if (ArrayType::classof(T)) {
            TypeLoc TL = (*I)->getTypeSourceInfo()->getTypeLoc();
            ArrayTypeLoc *ATL = dyn_cast<ArrayTypeLoc>(&TL);
            if (ATL) {
                Stmt* S = ATL->getSizeExpr();
                if (S) HandleStmt(Obj,S);
                else DEBUG(llvm::errs() << "skipping a NULL statement\n");
            }
        }

        if (RecordType* RT = dyn_cast<RecordType>(T)) {
            RecordDecl* RDField = RT->getDecl();
            if (isa<RecordDecl>(RDField->getDeclContext()))
                // Field also contains a record declaration
                HandleRecordFields(Obj,RDField);
        } else if (EnumType* ET = dyn_cast<EnumType>(T)) {
            EnumDecl* ED = ET->getDecl();
            if (isa<EnumDecl>(ED->getDeclContext()))
                // Field also contains a enum declaration
                HandleEnumConsts(Obj,ED);
        }
        HandleDifferentTypes(Obj,T);
    }
}

/// Creates dependencies in \param Obj on constants defined by \param ED.
void AAPConsumer::HandleEnumConsts (SourceObjRef Obj, const EnumDecl *ED)
{
    for (EnumDecl::enumerator_iterator I = ED->enumerator_begin(),
             End = ED->enumerator_end(); I != End; I++) {
        Expr* E = (*I)->getInitExpr();
        if (E != NULL) {
            Type* T = E->getType().getTypePtr();
            HandleDifferentTypes(Obj, T);
        }
    }
}

/// Creates dependencies in \param Obj on the parameters types of \param FD.
void AAPConsumer::HandleFunctionParams(SourceObjRef Obj, const FunctionDecl* FD)
{
    Type *T;
    for (FunctionDecl::param_const_iterator I = FD->param_begin(),
             E = FD->param_end(); I != E; I++) {
        T = ((*I)->getOriginalType()).getTypePtr();
        HandleDifferentTypes(Obj,T);
    }
    T = (FD->getResultType()).getTypePtr();
    HandleDifferentTypes(Obj,T);
}

void AAPConsumer::HandleVarDeclRef(SourceObjRef Obj, const VarDecl* Var)
{
    std::map<const NamedDecl*,SourceObjRef>::iterator ObjectsIter;

    if (!Var->hasGlobalStorage()) return;

    // Reference global or static variable
    ObjectsIter = ObjectsMap.find(Var);
    if (ObjectsIter == ObjectsMap.end()) return;

    if (ObjectsIter->second->isFunctionScope()) {
        // Declaration of the extern and static variable happened within a
        // function body. The dependency is not added for function containing
        // the declaration.
        const std::string FunctionName = ObjectsIter->second->ScopeFunction();
        if (FunctionName.compare (Obj->name()) != 0)
            Obj->addDependency (ObjectsIter->second->name());
    } else {
        Obj->addDependency (ObjectsIter->second->name());
    }
}

static SourceObjRef NewSourceObj (const std::string& Name,
                                  clang::SourceRange& SR,
                                  clang::SourceManager* SM,
                                  const std::string& Type,
                                  const std::string& PostName,
                                  unsigned int EndOffset,
                                  unsigned int StartOffset)
{
    static unsigned long ObjectID = 0;
    aap::SourceLocation Source (SR,*SM,EndOffset,StartOffset);
    ObjectID++;
    SourceObj *Obj = new SourceObj(Name,Source,Type,PostName,ObjectID);
    return SourceObjRef(Obj);
}
