//===----------------------------------------------------------------------===//
//
//                    The PACE Application Aware Partitioner
//
// Copyright (C) 2009 - 2011, ET International, Inc. All rights reserved.
//
// The information and source code contained herein is the exclusive property
// of ET International, Inc. and may not be disclosed, examined or reproduced
// in whole or in part without explicit written authorization from the company.
//
// This software was produced under a U.S. Government contract with the Air
// Force Research Lab. The U.S. Government is licensed to use, reproduce,
// modify, and distribute this software for use within the U.S. Government.
// These rights are equivalent to:
// GOVERNMENT PURPOSE RIGHTS, CONTRACT: F33615-09-C-7915
//
//===----------------------------------------------------------------------===//

#include "ast/attrinfo.h"
#include "utils/options.h"

#include <iostream>
#include <fstream>

using namespace aap;

llvm::cl::opt<double> AttrInfo::ColdTimePercentage (
    "cold", llvm::cl::init(2.0),
    llvm::cl::desc("Maximum function time percentage for cold attribute"));
llvm::cl::opt<double> AttrInfo::HotTimePercentage (
    "hot", llvm::cl::init(10.0),
    llvm::cl::desc("Minimum function time percentage for hot attribute"));

std::map<std::string,AttrInfo*>    AttrInfo::FunctionAttributes;
unsigned int AttrInfo::TotalTime = 0;

AttrInfo::AttrInfo (const std::string FuncName, const std::string FileName)
{
    std::string Name;
    Name = FuncName + FileName;
    pure = false;
    AttrConst = false;
    analyzed = false;
    depth = 0;
    setdefault = false;
    Time = 0;
    FunctionAttributes.insert(std::pair<std::string,AttrInfo*>(Name,this));
    if (Options::DebugAttribute()) {
        this->FuncName = FuncName;
        this->FileName = FileName;
    }
}

void AttrInfo::SetAnalyzed (bool analyzed)
{
    this->analyzed = analyzed;
}

bool AttrInfo::IsAnalyzed (void)
{
    return analyzed;
}

void AttrInfo::SetDefault (void)
{
    setdefault = true;
}

void AttrInfo::SetPure (bool pure)
{
    this->pure = pure;
}

bool AttrInfo::IsPure (void)
{
    return pure;
}

void AttrInfo::SetConst (bool AttrConst)
{
    this->AttrConst = AttrConst;
}

bool AttrInfo::IsConst (void)
{
    return AttrConst;
}

void AttrInfo::AddDependence (AttrInfo *Depends)
{
    Dependences.push_back(Depends);
}

AttrInfo * AttrInfo::GetDependence (unsigned int Pos)
{
    return Dependences[Pos];
}

unsigned int AttrInfo::DependencesSize (void)
{
    return Dependences.size();
}

void AttrInfo::EraseDependence (unsigned int Pos)
{
    if ((Dependences.begin()+Pos) < Dependences.end())
        Dependences.erase(Dependences.begin()+Pos);
}

void AttrInfo::ClearDependences (void)
{
    Dependences.clear();
}

bool AttrInfo::SearchDependence (AttrInfo *Info)
{
    unsigned int i;
    bool Found = false;
    for (i = 0; ((i<Dependences.size()) && (Found == false)); i++)
        if (Info == Dependences[i])
            Found = true;
    return Found;
}

void AttrInfo::AddObjRef (SourceObjRef Object)
{
    obj.push_back(Object);
}

SourceObjRef AttrInfo::GetObjRef (unsigned int Pos)
{
    return obj[Pos];
}

unsigned int AttrInfo::ObjRefVectorSize (void)
{
    return obj.size();
}

void AttrInfo::ClearObjRefVector (void)
{
    obj.clear();
}

AttrInfo * AttrInfo::GetOrInsertAttrInfo (const std::string FuncName,
					  const std::string FileName)
{
    AttrInfo * AI;
    AI = SearchAttr(FuncName,FileName);
    if (AI == NULL)
        AI = new AttrInfo(FuncName,FileName);
    return AI;
}

AttrInfo * AttrInfo::SearchAttr (const std::string FuncName,
				 const std::string FileName)
{
    std::string Name;
    std::map<std::string,AttrInfo*>::iterator FA;
    Name = FuncName + FileName;
    FA = FunctionAttributes.find(Name);
    if (FA == (FunctionAttributes.end()))
        return NULL;
    else
        return FA->second;
}

void AttrInfo::InsertSourceObj (SourceObjRef Object,
                                const std::string FuncName,
				const std::string FileName)
{
    AttrInfo * AI;
    AI = SearchAttr(FuncName,FileName);
    if (AI == NULL)
        AI = new AttrInfo(FuncName,FileName);
    AI->AddObjRef(Object);
}

void AttrInfo::HandleAttributesDependences (void)
{
    bool HotCold, HaveAttr;
    double ColdTime, HotTime;
    AttrInfo *AT;
    std::map<std::string,AttrInfo*>::iterator AttrIter;

    for (AttrIter  = FunctionAttributes.begin();
         AttrIter != FunctionAttributes.end(); AttrIter++) {
        AT = AttrIter->second;
        if (AT->DependencesSize())
            AT->SetAnalyzed(false);
        else
            AT->SetAnalyzed(true);
    }

    for (AttrIter  = FunctionAttributes.begin();
         AttrIter != FunctionAttributes.end(); AttrIter++) {
        AT = AttrIter->second;
        if (AT->IsAnalyzed() == false) {
            AT->SetDepth(1);
            AT->HandleDependence();
        }
    }

    SourceObjRef Object;
    HotCold = false;
    if (TotalTime > 100) {
        ColdTime = (ColdTimePercentage/100.0) * ((double)TotalTime);
        HotTime  = (HotTimePercentage /100.0) * ((double)TotalTime);
        HotCold = true;
    }
    for (AttrIter  = FunctionAttributes.begin();
         AttrIter != FunctionAttributes.end(); AttrIter++) {
        HaveAttr = false;
        AT = AttrIter->second;
        std::string AttrStr;
        AttrStr = " __attribute__ ((";
        if (AT->IsPure()) {
            if (AT->IsConst())
                AttrStr.append("const");
            else
                AttrStr.append("pure");
            HaveAttr = true;
        }
        if (HotCold) {
            if (((double)AT->Time) < ColdTime) {
                if (HaveAttr)
                    AttrStr.append(", ");
                AttrStr.append("cold");
                HaveAttr = true;
            }
            if (((double)AT->Time) > HotTime) {
                if (HaveAttr)
                    AttrStr.append(", ");
                AttrStr.append("hot");
                HaveAttr = true;
            }
        }
        AttrStr.append("))");
        if (HaveAttr)
            for (unsigned int i = 0; i < AT->ObjRefVectorSize(); i++) {
                Object = AT->GetObjRef(i);
                Object->SetAttribute(AttrStr);
            }
        AT->ClearObjRefVector();
        AttrStr.clear();
    }
    if (Options::DebugAttribute()) {
        std::fstream Debug("Attributes.inf", std::fstream::out);
        Debug << "Attribute Information" << std::endl;
        if (HotCold) {
            Debug << "-> Maximum execution time for cold attribute: "
                  << ColdTime << std::endl;
            Debug << "-> Minimum execution time for hot attribute : "
                  << HotTime << std::endl;
        }
        Debug << "---------------------" << std::endl;
        for (AttrIter  = FunctionAttributes.begin();
             AttrIter != FunctionAttributes.end(); AttrIter++) {
            HaveAttr = false;
            AT = AttrIter->second;
            std::string AttrStr;
            AttrStr = ":";
            if (AT->IsPure()) {
                if (AT->IsConst())
                    AttrStr.append(" const pure");
                else
                    AttrStr.append(" pure");
                HaveAttr = true;
            }
            if (HotCold) {
                if (((double)AT->Time) < ColdTime) {
                    AttrStr.append(" cold");
                    HaveAttr = true;
                }
                if (((double)AT->Time) > HotTime) {
                    AttrStr.append(" hot");
                    HaveAttr = true;
                }
            }
            if (HaveAttr)
                if (AT->FileName.size())
                    Debug << "Static function " << AT->FuncName
                          << " in file " << AT->FileName
                          << AttrStr << std::endl;
                else
                    Debug << "Global function " << AT->FuncName
                          << AttrStr << std::endl;
            AT->FuncName.clear();
            AT->FileName.clear();
            AttrStr.clear();
        }
        Debug.close();
    }
}

void AttrInfo::SetDepth (int depth)
{
    this->depth = depth;
}

int AttrInfo::GetDepth (void)
{
    return depth;
}

int AttrInfo::HandleDependence (void)
{
    unsigned int Pos;
    int CurDepth, MinDepth;
    bool Pure;
    std::vector<AttrInfo*> NewDependence;
    AttrInfo *Child;

    Pure = true;
    MinDepth = depth;
    for (Pos = 0; ((Pos < DependencesSize()) && (Pure == true)); Pos++) {
        Child = GetDependence(Pos);
        if (Child->IsAnalyzed()) {
            Pure = Child->IsPure();
        }
        else {
            if (Child->GetDepth() == 0) {
                Child->SetDepth(depth+1);
                CurDepth = Child->HandleDependence();
                if (CurDepth == 0) {
                    Pure = Child->IsPure();
                }
                else if (CurDepth != depth) {
                    if (CurDepth < MinDepth)
                        MinDepth = CurDepth;
                    NewDependence.push_back(Child);
                }
            }
            else {
                CurDepth = Child->GetDepth();
                if (CurDepth < MinDepth)
                    MinDepth = CurDepth;
                NewDependence.push_back(Child);
            }
        }
        if (AttrConst)
            AttrConst = Child->IsConst();
    }

    ClearDependences();
    if (Pure == false) {
        depth = 0;
        analyzed = true;
        AttrConst = false;
        ClearDependences();
        return 0;
    }

    if (MinDepth < depth) {
        depth = 0;
        Dependences = NewDependence;
        NewDependence.clear();
        return MinDepth;
    }

    pure = true;
    analyzed = true;
    depth = 0;
    return 0;
}

void AttrInfo::AddTime (unsigned int Time)
{
    this->Time += Time;
}

void AttrInfo::SetTotalTime (unsigned int Time)
{
    TotalTime = Time;
}
